# -*- coding: utf-8 -*-
import json
import logging

from collections import defaultdict

import sandbox
from sandbox.projects.browser.autotests_qa_tools.classes.test_statuses import TestStatuses


logger = logging.getLogger(__file__)


class FlakyAnalyzer(object):

    @staticmethod
    @sandbox.common.utils.singleton
    def get_test(report, test_full_name, platform):
        for test in report.tests:
            if test.full_name == test_full_name and test.platform == platform:
                return test
        return None

    @staticmethod
    def get_tests_from_dump(dump_data, test_name, platform=None, feature=None, test_id=None):
        result = []
        for test in dump_data['tests']:
            _r = test['name'] == test_name
            if not _r:
                continue
            if platform is not None:
                _r = _r and test['platform'] == platform
            if feature is not None:
                _r = _r and test['feature'] == feature
            if test_id is not None:
                _r = _r and test['test_id'] == test_id
            if _r:
                result.append(test)
        return result

    def get_failures_table(self, dump_paths, save_to_file=None):

        failures_set = set()
        for _dump_path in dump_paths:
            with open(_dump_path, 'r') as _f:
                data = json.load(_f)
            _report_failures = set(
                test['name'] for test in data['tests'] if test['status'] in TestStatuses.FAILED.value)
            failures_set = failures_set.union(_report_failures)

        result = {}
        for _dump_path in dump_paths:
            with open(_dump_path, 'r') as _f:
                _dump_data = json.load(_f)
                logger.info('{} dump loaded'.format(_dump_path))
            if _dump_data.get('sandbox_task') is None:
                raise RuntimeError(u"Не известен sandbox_task отчета")

            for test_name in failures_set:

                _target = result.setdefault(test_name, {}).setdefault(_dump_data['sandbox_task'], [])
                report_tests = self.get_tests_from_dump(_dump_data, test_name)
                for _t in report_tests:
                    _target.append(
                        {
                            "status": _t['status'],
                            "platform": _t['platform'],
                            "test_id": _t['test_id'],
                            "feature": _t['feature'],
                            "failure_message": _t['failure_message'],
                            "log": _t['log']
                        })
        if save_to_file:
            with open(save_to_file, "w") as _f:
                json.dump(result, _f, indent=4)

        return result

    def analyse_failures_table(self, table_json_path, save_to_file=None):
        with open(table_json_path, 'r') as _f:
            tests = json.load(_f)

        statistics = defaultdict(dict)
        for test_name, builds in tests.iteritems():
            for tests in builds.values():
                for test in tests:
                    _stat = statistics[test_name].setdefault(test['platform'], {
                        'passed': 0,
                        'failed': 0,
                        'total': 0,
                        'cases': []})
                    if test['status'] in TestStatuses.PASSED.value:
                        _stat['passed'] += 1
                    elif test['status'] in TestStatuses.FAILED.value:
                        _stat['failed'] += 1
                    _stat['total'] += 1
                    if test['test_id'] not in _stat['cases']:
                        _stat['cases'].append(test['test_id'])

        result = {
            'flaky': [],
            'slow_flaky': [],
            'stable': []
        }

        def _fails_percent(_s):
            return int(round((float(
                _s['failed']) * 100) / (_s['failed'] + _s['passed'])) if (_s['failed'] + _s['passed']) > 0 else 0)

        for test_name, stat in statistics.iteritems():
            if all(_fails_percent(_s) in [0, 100] for _s in stat.values()):
                _target = result['stable']
            elif any(_fails_percent(_s) >= 30 for _s in stat.values()):
                _target = result['flaky']
            else:
                _target = result['slow_flaky']

            _target.append({
                test_name: statistics[test_name]})

        for _item in result.values():
            _item.sort()

        if save_to_file:
            with open(save_to_file, "w") as _f:
                json.dump(result, _f, indent=4)

        return result
