# -*- coding: utf-8 -*-
from abc import ABCMeta, abstractmethod
import re
import jinja2
import sandbox


TICKET_JINJA_PACKAGE_PATH = 'projects.browser.autotests_qa_tools.classes.autoduty'
BIGBOSSES = ['timy4', 'golubtsov', 'veral', 'styskin', 'tigran', 'volozh']
DEFAULT_FOLLOWERS = ['nik-isaev']
MODILE_BLACKLISTS_BASE = 'src/build/yandex/autotests/binary/blacklists'
MODILE_BLACKLISTS_TARGETS = {
    'searchapp': {
        'apad': '{}/searchapp/android_apad.yaml'.format(MODILE_BLACKLISTS_BASE),
        'aphone': '{}/searchapp/android_aphone.yaml'.format(MODILE_BLACKLISTS_BASE),
        'ios': '{}/searchapp/ios.yaml'.format(MODILE_BLACKLISTS_BASE)
    },
    'browser': {
        'apad': '{}/browser/android_apad.yaml'.format(MODILE_BLACKLISTS_BASE),
        'aphone': '{}/browser/android_aphone.yaml'.format(MODILE_BLACKLISTS_BASE),
        'ios': '{}/browser/ios.yaml'.format(MODILE_BLACKLISTS_BASE)
    }
}


class Checks(object):

    @staticmethod
    def test_type(test, issue):
        return test.test_type == issue.test_type

    @staticmethod
    def test_platform(test, issue):
        if issue.platforms:
            return test.platform in issue.platforms
        else:
            return True

    @staticmethod
    def full_coverage(test, issue):
        return test.testpalm_case.mapped_attributes.get('Automation Status', []) == ['Ready']

    @staticmethod
    def same_assigner(test, issue):
        return (test.owners[0] == issue.assignee)

    @staticmethod
    def same_component(test, issue):
        return (test.feature in issue.components)

    @staticmethod
    def exclude_components(test, issue):
        return (test.feature not in issue.exclude_components)

    @staticmethod
    def not_success(test):
        templ = ".*Expected that .* finished with SUCCESS, but it wasn't.*"
        return re.match(re.compile(templ, re.DOTALL),
                        test.failure['message'])

    @staticmethod
    def same_failure_message(test, issue):
        return test.failure['message'] in [t.failure['message']
                                           for t in issue.tests]

    @staticmethod
    def same_failure_stack_trace_pattern(test, failure_message_pattern):
        return re.search(re.compile(failure_message_pattern, re.DOTALL),
                         test.failure['stackTrace'])

    @staticmethod
    def same_test_case_id(test, issue):
        return test.test_id in [t.test_id for t in issue.tests]

    @staticmethod
    def same_test_file(test, issue):
        return test.filepath in [t.filepath for t in issue.tests]

    @staticmethod
    def same_binary(test, issue):
        return test.binary in [t.binary for t in issue.tests]

    @staticmethod
    def max_tests_count(test, issue, max_tests_count=50):
        return len(issue.tests) < max_tests_count

    @staticmethod
    def smart_max_tests_count(test, issue, max_tests_count=50):
        return (
            Checks.max_tests_count(test, issue, max_tests_count)
        ) or (
            Checks.same_test_case_id(test, issue)
        )


class Issue(object):

    __metaclass__ = ABCMeta
    _description_template = None
    _title = "Починить {type}-автотесты {components}"
    _platforms = []
    queue = "BROWSER"
    exclude_components = []

    def __init__(self, ya_resources):
        self.ya_resources = ya_resources
        self.tests = []

    @property
    def tested_application(self):
        return list(set(_t.tested_application for _t in self.tests))

    def check_add_test(self, test):
        return (
            Checks.same_assigner(test, self) or
            Checks.same_component(test, self)
        )

    @abstractmethod
    def check_type(cls, test):
        """Checks the validity of creating a ticket based on the test"""

    @abstractmethod
    def affected_versions(self):
        """affected_versions"""

    @abstractmethod
    def fix_versions(self):
        """fix_versions"""

    @abstractmethod
    def blacklists_targets(self):
        """List of blecklists targets"""

    def add_test(self, test):
        self.tests.append(test)

    @property
    def assignee(self):
        return self.tests[0].owners[0]

    @property
    def tests_version(self):
        return self.tests[0].test_version

    @property
    def priority(self):
        return u'Critical'

    @property
    def followers(self):
        followers = set(DEFAULT_FOLLOWERS)
        for test in self.tests:
            followers = followers | set(test.owners)
        boss = self.ya_resources.get_boss(self.assignee)
        if boss and boss not in BIGBOSSES:
            followers = followers | {boss}

        return list(followers)

    @property
    def components(self):
        return list(set(test.feature for test in self.tests))

    @property
    def title(self):
        return self._title.format(components=",".join(self.components),
                                  type=self.test_type)

    @property
    def description(self):
        template = jinja2.Environment(
            loader=jinja2.PackageLoader(TICKET_JINJA_PACKAGE_PATH,
                                        package_path='templates')).get_template(self._description_template)
        return template.render(issue=self)

    @property
    def tags(self):
        tags = ["fix_autotests", "{}_autotests".format(self.test_type)]
        if self._is_health_check:
            tags.append('broken_healthcheck')
        return tags

    @property
    def ticket_components(self):
        return ["Autotests"]

    @property
    @sandbox.common.utils.singleton
    def _is_health_check(self):
        return any('HealthCheck' in _test.testpalm_case.mapped_attributes.get('Checklist', []) for _test in self.tests)

    @property
    def issue_type(self):
        return 1  # Bug

    @property
    def startrek_template(self):
        return {
            'queue': self.queue,
            'summary': self.title,
            'description': self.description,
            'assignee': self.assignee,
            'affectedVersions': self.affected_versions,
            'fixVersions': self.fix_versions,
            'type': self.issue_type,
            'followers': self.followers,
            'priority': self.priority.lower(),
            'components': self.ticket_components,
            'tags': self.tags
        }

    @property
    @sandbox.common.utils.singleton
    def all_tests_log(self):
        res = u""
        for test in self.tests:
            res += u"\n\n{}:\n{}".format(test.full_name, test.test_log)
        return res

    @property
    @sandbox.common.utils.singleton
    def attachments(self):
        return {}


class BinaryTestIssue(Issue):

    test_type = 'binary'
    _description_template = 'binary_tests_failed_issue.jinga'

    @property
    @sandbox.common.utils.singleton
    def attachments(self):
        return {
            "all_tests_log.txt": self.all_tests_log.encode('utf-8')
        }

    @property
    def blacklists_targets(self):
        return set((_t.test_id, _t.filepath) for _t in self.tests)

    @property
    def binarys(self):
        return [_t.binary for _t in self.tests]

    @property
    def affected_versions(self):
        return "Dev"

    @property
    def fix_versions(self):
        return "Dev"


class BinaryTestsFailedIssue(BinaryTestIssue):

    @classmethod
    def check_type(cls, test):
        return (
            Checks.test_type(test, cls) and
            Checks.full_coverage(test, cls)
        )


class MobileTestsFailedIssue(BinaryTestIssue):

    _description_template = 'mobile_tests_failed_issue.jinga'

    @property
    def blacklists_targets(self):
        return set((_t.test_id,
                    MODILE_BLACKLISTS_TARGETS[self.tested_application[0]][_t.platform]) for _t in self.tests)

    @property
    def title(self):
        title= "{}Починить {} автотест(ов) {}".format(
            ''.join(set(["[{}]".format(_t.platform) for _t in self.tests])).upper(),
            len(self.tests),
            ', '.join(set([_t.binary for _t in self.tests])))
        return title

    @classmethod
    def check_type(cls, test):
        return (
            Checks.test_platform(test, cls)
        )

    @property
    def tags(self):
        tags = ["Autotests"]
        if self._is_health_check:
            tags.append('broken_healthcheck')
        return tags

    @property
    def attachments(self):
        return {}

    @property
    def affected_versions(self):
        return "Master"

    @property
    def fix_versions(self):
        return "Master"

    @property
    def ticket_components(self):
        return []


class IosTestsFailedIssue(MobileTestsFailedIssue):
    queue = 'IBRO'
    platforms = ['ios']

    def check_add_test(self, test):
        return (
            Checks.same_test_case_id(test, self) or
            Checks.same_test_file(test, self)
        ) and (
            Checks.same_assigner(test, self)
        )


class AndroidTestsFailedIssue(MobileTestsFailedIssue):
    queue = 'ABRO'
    platforms = ['apad', 'aphone']

    @property
    def issue_type(self):
        return 173  # Technical Story

    def check_add_test(self, test):
        return (
            Checks.same_binary(test, self)
        ) and (
            Checks.same_assigner(test, self)
        ) and (
            Checks.smart_max_tests_count(test, self, max_tests_count=50)
        )


class PythonTestIssue(Issue):

    _description_template = "python_tests_issue.jinga"
    test_type = 'python'
    infra_problems_pattern = 'Could not download https://teamcity.browser.yandex-team.ru'

    @property
    def blacklists_targets(self):
        return None

    @property
    def title(self):
        return "Починить {} {}-автотест(ов) {}".format(
            len(self.tests),
            self.test_type,
            ",".join(self.components))

    @classmethod
    def check_type(cls, test):
        return Checks.test_type(test, cls)

    def check_add_test(self, test):
        return (
            Checks.same_failure_message(test, self) or
            Checks.same_test_case_id(test, self) or
            Checks.same_test_file(test, self)
        )

    @property
    def affected_versions(self):
        return "Dev"

    @property
    def fix_versions(self):
        return "Dev"


class PythonInstallUpdateTestIssue(PythonTestIssue):

    components = ['installer', 'updater']
    blacklists_targets = []

    @classmethod
    def check_type(cls, test):
        return Checks.test_type(test, cls) and Checks.same_component(test, cls) and not Checks.same_failure_stack_trace_pattern(
            test, cls.infra_problems_pattern)


class PythonLegacyTestIssue(PythonTestIssue):

    exclude_components = ['installer', 'updater']
    blacklists_targets = []
    _description_template = "python_legacy_tests_issue.jinga"

    @property
    def title(self):
        return "Перевести в бинарные и удалить {} {}-автотест(ов) {}".format(
            len(self.tests),
            self.test_type,
            ",".join(self.components))

    @classmethod
    def check_type(cls, test):
        return Checks.test_type(test, cls) and Checks.exclude_components(test, cls) and not Checks.same_failure_stack_trace_pattern(
            test, cls.infra_problems_pattern)

    @property
    def assignee(self):
        return 'kochetov-a'

    @property
    def tags(self):
        tags = ["fix_autotests", "{}_autotests".format(self.test_type), 'legacy_python_tests']
        if self._is_health_check:
            tags.append('broken_healthcheck')
        return tags


class PythonInstallUpdateTestIssueInfra(PythonInstallUpdateTestIssue):

    _title = u"Починить проблемы инфры {type}-автотестов {components}"

    @property
    def assignee(self):
        return 'nik-isaev'

    @classmethod
    def check_type(cls, test):
        return Checks.test_type(test, cls) and Checks.same_component(test, cls) and Checks.same_failure_stack_trace_pattern(
            test, cls.infra_problems_pattern)

    @property
    def title(self):
        return self._title.format(components=",".join(self.components),
                                  type=self.test_type)
