# -*- coding: utf-8 -*-
import json
import re
import math
from copy import deepcopy

import sandbox
from sandbox.projects.browser.autotests_qa_tools.common import FRAMEWORK_BITBUCKET_PROJECT, FRAMEWORK_BITBUCKET_REPO
from sandbox.projects.browser.autotests_qa_tools.classes.testing_platforms import TestingPlatforms, SupportedPlaftorms
from sandbox.projects.browser.autotests_qa_tools.common import SPECIFICATION_NAME, SPECIFICATION_PATH


BINARY_TESTS_NODE_INSTANCES = 8
BINARY_TESTS_PACK = {
    "pack_name": None,
    "node_label": None,
    "specify_by": {
        "key": "",
        "path": "tests_binary"
    },
    "launch_arguments": {
        "mergeArray": True,
        "reruns": "0",
        "tests_launch_arguments": "",
    }
}
PYTHON_TESTS_PLATFORM_PREFIX = "python_"
PYTHON_TESTS_PLATFORMS = {
    # 'mac': [
    #     SupportedPlaftorms.MacOS_10_14,
    # ],
    'win': [
        # SupportedPlaftorms.Win7x32,
        SupportedPlaftorms.Win7x64,
        SupportedPlaftorms.Win10x64,
    ]
}

# temporary solution, see: https://st.yandex-team.ru/ABRO-62408
ANDROID_DEPRECATED_CASES = [
    "abro-25442",
    "abro-28710",
    "abro-26383",
    "abro-21943",
    "abro-28429",
    "abro-28428",
    "abro-25277",
    "abro-23972",
    "abro-27561",
    "abro-27131",
    "abro-6127",
    "abro-15479",
    "abro-7445",
    "abro-7444",
    "abro-7447",
    "abro-7446",
    "abro-25884",
    "abro-23155",
    "abro-23178",
    "abro-26236",
    "abro-27141",
    "abro-27140",
    "abro-23231",
    "abro-7448",
    "abro-27123",
    "abro-35039",
    "abro-22167",
    "abro-20726",
    "abro-34347",
    "abro-22830",
    "abro-18136",
    "abro-25533",
    "abro-34451",
    "abro-27560",
    "abro-34453",
    "abro-28431",
    "abro-24212",
    "abro-27449",
    "abro-25551",
    "abro-29154",
    "abro-24106",
    "abro-24105",
    "abro-31126",
    "abro-31202",
    "abro-21572",
    "abro-31125",
    "abro-22156",
    "abro-12967",
    "abro-22132",
    "abro-28050",
    "abro-27559",
    "abro-28053",
    "abro-21571",
    "abro-27476",
    "abro-15478",
    "abro-27125",
    "abro-18824",
    "abro-28354"
]


class IsolateAutotestBundle(object):

    number_of_shards = 1
    tested_application = 'browser'

    def __init__(self, browser_tests_build, ya_clients):
        self.browser_tests_build = browser_tests_build
        self.ya_clients = ya_clients

    @property
    @sandbox.common.utils.singleton
    def _configs(self):
        artifact_folder = 'autotests_{}'.format(self.tested_application)
        artifact_names = [_a.name for _a in self.browser_tests_build.artifacts()]
        # Temporary compatibility crutch
        if artifact_folder not in artifact_names and 'autotests' in artifact_names:
            artifact_folder = 'autotests'

        configs = {}
        if artifact_folder in artifact_names:
            configs = {_a.name: json.loads(_a.download().data)
                       for _a in self.browser_tests_build.artifacts(artifact_folder) if _a.name.endswith(".json")}
        return configs

    def get_launch_config_by_cases(self, cases, ignore_blacklists=True):
        result = {}
        for case_id, testrun_environment in cases:
            for _case_info in self._get_case_info(case_id, testrun_environment):
                if not ignore_blacklists and _case_info['blacklists']:
                    continue
                platform_cases = result.setdefault(_case_info['platform'], [])
                if case_id not in platform_cases:
                    platform_cases.append(case_id)
        return result

    def get_launch_config_all_cases(self, ignore_blacklists=True):
        all_cases = []
        for config_name, config in self._configs.iteritems():
            for component_data in config.values():
                all_cases += [
                    (_c , config_name.replace(".json", "")) for _c in component_data.get("test_cases", {}).keys()]
        return self.get_launch_config_by_cases(all_cases, ignore_blacklists)

    def get_launch_config_changed_cases(self):
        path, filename = '_changed_autotests', '_config.json'
        artifacts = [a for a in self.browser_tests_build.artifacts(path) if a.name == filename]
        if len(artifacts) != 1:
            raise RuntimeError('Expected that teamcity build #{} has one artifact "{}{}" but found {}'.format(
                self.browser_tests_build.id, path, filename, len(artifacts)))

        config = json.loads(artifacts[0].download().data)
        result = {}
        for platform, components in config.iteritems():
            platform_cases = []
            for component_data in components.values():
                platform_cases += component_data.get("test_cases", {}).keys()
            if platform_cases:
                result[platform] = platform_cases
        return result

    @sandbox.common.utils.singleton
    def _get_case_info(self, case_id, testrun_environment):
        result = []
        config_name = TestingPlatforms.get_isolates_artifact_name(testrun_environment)
        config = self._configs.get(config_name)
        if config:
            for _component, _component_info in config.iteritems():
                for _case_id, _tests in _component_info.get('test_cases', {}).iteritems():
                    if case_id == _case_id:
                        result.append(
                            dict(platform=config_name.replace(".json", ""),
                                 component=_component,
                                 tests=_tests,
                                 blacklists=_component_info.get('blacklists', {}).get(_case_id)))
        return result

    def is_automated_in_tests(self, case_id, testrun_environment, ignore_blacklists=False):
        case_info = self._get_case_info(case_id, testrun_environment)
        return bool(case_info) and any([not x['blacklists'] or ignore_blacklists for x in case_info])

    @property
    @sandbox.common.utils.singleton
    def browser_branch(self):
        return self.browser_tests_build.branch_name

    @property
    @sandbox.common.utils.singleton
    def browser_commit(self):
        return self.browser_tests_build.revisions[0]['version']

    @property
    @sandbox.common.utils.singleton
    def build_extra_args(self):
        return {}


class SearchappIsolateAutotestBundle(IsolateAutotestBundle):

    tested_application = 'searchapp'


class AndroidAutotestsBundle(IsolateAutotestBundle):

    number_of_shards = 1

    def __init__(self, browser_tests_build, browser_build, ya_clients):
        self.browser_tests_build = browser_tests_build
        self.browser_build = browser_build
        self.ya_clients = ya_clients

    @staticmethod
    def _get_artifact_url(build, reg_exp):
        result = None
        artifact_names = [_a.name for _a in build.artifacts()]
        artifact_folder = None
        # see https://st.yandex-team.ru/BYIN-12471
        if 'extracted_apks' in artifact_names:
            artifact_folder = 'extracted_apks'
        elif 'apks' in artifact_names:
            artifact_folder = 'apks'

        if artifact_folder is not None:
            for _a in build.artifacts(artifact_folder):
                if re.match(reg_exp, _a.name):
                    result = "{}?guest=1".format(_a.url)
                    break
        return result

    @property
    @sandbox.common.utils.singleton
    def build_extra_args(self):
        build_extra_args = {}
        artifact_url = self._get_artifact_url(self.browser_build, '.*browser-x86.*release.apk$')
        if artifact_url:
            build_extra_args.setdefault('appium_tests-extra-args', []).append(
                u"--app-url={}".format(artifact_url))
        return build_extra_args

    @property
    @sandbox.common.utils.singleton
    def is_appium_tests_available(self):
        return any(str(_p).startswith('--app-url=') for _p in self.build_extra_args.get('appium_tests-extra-args', []))

    def _get_case_info(self, case_id, testrun_environment):
        # temporary solution, see: https://st.yandex-team.ru/ABRO-62408
        if case_id in ANDROID_DEPRECATED_CASES:
            return []

        # Skip appium_tests if apk app-url is not available. see: https://st.yandex-team.ru/BYIN-14601
        _case_info = super(AndroidAutotestsBundle, self)._get_case_info(case_id, testrun_environment)
        result = []
        for _case_info_item in _case_info:
            if any(_test.startswith('appium_tests.') for _test in _case_info_item['tests']):
                if not self.is_appium_tests_available:
                    continue
            result.append(_case_info_item)
        return result


class AndroidSearchappAutotestBundle(AndroidAutotestsBundle):

    tested_application = 'searchapp'
    number_of_shards = 1

    def __init__(self, browser_tests_build, browser_build, fake_build, ya_clients):
        self.browser_tests_build = browser_tests_build
        self.browser_build = browser_build
        self.fake_build = fake_build
        self.ya_clients = ya_clients

    @property
    @sandbox.common.utils.singleton
    def build_extra_args(self):
        build_extra_args = {}

        if self.browser_build:
            artifact_url = self._get_artifact_url(self.browser_build, '.*mobileyandex-x86.*release.apk$')
            if artifact_url:
                build_extra_args.setdefault('appium_tests-extra-args', []).append(
                    u"--app-url={}".format(artifact_url))
                build_extra_args['appium_tests-extra-args'].append(
                    u"--app-version={}".format('{}.{}{}'.format(*self.browser_build.number.split('.')[:3])))
        if self.fake_build:
            artifact_url = self._get_artifact_url(self.fake_build, '.*mobileyandex-x86.*release.apk$')
            if artifact_url:
                build_extra_args.setdefault('appium_tests-extra-args', []).append(
                    u"--old-app-url={}".format(artifact_url))
                build_extra_args['appium_tests-extra-args'].append(
                    u"--old-app-version={}".format('{}.{}{}'.format(*self.fake_build.number.split('.')[:3])))
        return build_extra_args


class DesktopAutotestsBouldle(IsolateAutotestBundle):

    def __init__(self, browser_tests_build, browser_build, fake_build, ya_clients):
        super(DesktopAutotestsBouldle, self).__init__(browser_tests_build, ya_clients)
        self.browser_build = browser_build
        self.fake_build = fake_build

    @property
    @sandbox.common.utils.singleton
    def _configs(self):
        configs = {}

        artifact_folder = 'autotests_{}'.format(self.tested_application)
        artifact_names = [_a.name for _a in self.browser_tests_build.artifacts()]
        # Temporary compatibility crutch
        if artifact_folder not in artifact_names and 'autotests' in artifact_names:
            artifact_folder = 'autotests'

        if artifact_folder in artifact_names:
            configs = {_a.name: json.loads(_a.download().data)
                       for _a in self.browser_tests_build.artifacts(artifact_folder) if _a.name.endswith(".json") and _a.name not in ["win_7_x32.json", "win_8_x64.json"]}
        return configs

    @property
    @sandbox.common.utils.singleton
    def python_tests_bundle(self):
        if self.browser_build:
            return get_latest_autotests_bundle_by_browser_build(
                branded_browser_build_id=self.browser_build.id,
                browser_tests_build_id=None,
                ya_clients=self.ya_clients)
        else:
            return None

    @property
    @sandbox.common.utils.singleton
    def build_extra_args(self):
        build_extra_args = {}
        if self.python_tests_bundle:
            build_extra_args = {'python_tests-extra-args': {}}
            build_extra_args['python_tests-extra-args']['build_id'] = self.browser_build.id if self.browser_build else 0
            build_extra_args['python_tests-extra-args']['fake_build_id'] = self.fake_build.id if self.fake_build else 0
        return build_extra_args

    def _is_automated_in_isolates(self, case_id, testrun_environment, ignore_blacklists=False):
        return super(DesktopAutotestsBouldle, self).is_automated_in_tests(case_id, testrun_environment, ignore_blacklists)

    def _is_automated_in_python_tests(self, case_id, testrun_environment, ignore_blacklists):
        if not self.python_tests_bundle:
            return False
        _platform = TestingPlatforms.get_autotest_platform(testrun_environment)
        if not _platform or _platform not in PYTHON_TESTS_PLATFORMS.get(self._platform_type, []):
            return False
        return self.python_tests_bundle.is_automated_in_tests(case_id, testrun_environment, ignore_blacklists)

    def is_automated_in_tests(self, case_id, testrun_environment, ignore_blacklists=False):
        return bool(self._is_automated_in_isolates(case_id, testrun_environment, ignore_blacklists) or
                    self._is_automated_in_python_tests(case_id, testrun_environment, ignore_blacklists))

    def get_launch_config_by_cases(self, cases, ignore_blacklists=True):
        result = {}
        for case_id, testrun_environment in cases:
            config_name = TestingPlatforms.get_isolates_artifact_name(testrun_environment)
            if config_name:
                config_name = config_name.replace(".json", "")
                if self._is_automated_in_isolates(case_id, testrun_environment, ignore_blacklists):
                    result.setdefault(config_name, []).append(case_id)
                if self._is_automated_in_python_tests(case_id, testrun_environment, ignore_blacklists):
                    result.setdefault("{}{}".format(PYTHON_TESTS_PLATFORM_PREFIX, config_name), []).append(case_id)
        return result

    @property
    @sandbox.common.utils.singleton
    def _platform_type(self):
        platform_type = None
        if self.browser_tests_build.build_type.id == 'Browser_Tests_Build_Win':
            platform_type = 'win'
        elif self.browser_tests_build.build_type.id == 'Browser_Tests_Build_Mac':
            platform_type = 'mac'
        return platform_type

    @sandbox.common.utils.singleton
    def get_launch_config_all_cases(self, ignore_blacklists=True):
        result = super(DesktopAutotestsBouldle, self).get_launch_config_all_cases(ignore_blacklists)
        # add python tests
        if self.python_tests_bundle:
            for case_id, _python_platform in self.python_tests_bundle.get_all_python_cases(self._platform_type, ignore_blacklists):
                isolane_config_name = TestingPlatforms.get_isolates_artifact_name(_python_platform)
                if isolane_config_name:
                    _fake_platform = "{}{}".format(PYTHON_TESTS_PLATFORM_PREFIX, isolane_config_name.replace(".json", ""))
                    result.setdefault(_fake_platform, []).append(case_id)
        return result


# legacy
class AutotestBundle(object):

    def __init__(self, bundle_commit, browser_tests_build, ya_clients):
        self.bundle_commit = bundle_commit
        self.browser_tests_build = browser_tests_build
        self.ya_clients = ya_clients

    @property
    @sandbox.common.utils.singleton
    def _python_tests_specification(self):
        return json.loads(self.ya_clients.bitbucket.load_file(FRAMEWORK_BITBUCKET_PROJECT,
                                                              FRAMEWORK_BITBUCKET_REPO,
                                                              '{}/{}'.format(SPECIFICATION_PATH, SPECIFICATION_NAME),
                                                              at=self.bundle_commit))

    @property
    @sandbox.common.utils.singleton
    def _binary_tests_specification(self):

        result = {"cases": {},
                  "total": {}}
        if not self.browser_tests_build:
            return result

        configs = {}
        if self.browser_tests_build and 'autotests' in [_a.name for _a in self.browser_tests_build.artifacts()]:
            configs = {_a.name: json.loads(_a.download().data)
                       for _a in self.browser_tests_build.artifacts('autotests') if _a.name.endswith(".json")}
        for _config_name, _components in configs.iteritems():
            config_info = TestingPlatforms.get_binary_config_platform_data(_config_name)
            if config_info is None:
                continue
            _platform, _node = config_info
            for _component in _components.values():
                for case_id in _component.get("test_cases", {}):
                    result["cases"].setdefault(
                        case_id, {}).setdefault(
                            "platforms", {}).setdefault(
                            _platform, {}).update(
                                {"case_id": case_id,
                                 "blacklisted": case_id in _component.get("blacklists", {}),
                                 "node_label": _node})
                    _node_count = result["total"].setdefault(_node, 0)
                    result["total"][_node] = _node_count + 1
        return result

    def _generate_python_packs(self, python_packs_dict):
        python_packs_templates = {
            pack["pack_name"]: pack
            for pack in self._python_tests_specification["launch_config"]["packs"]}

        result = []
        for pack_name, cases in python_packs_dict.iteritems():
            new_pack = deepcopy(python_packs_templates[pack_name])
            pack_cases = {_c["case_id"] for _c in cases}
            tests_launch_arguments = new_pack.setdefault(
                'launch_arguments',
                {}).setdefault('tests_launch_arguments', '')
            new_pack['launch_arguments'][
                'tests_launch_arguments'] = "{} {}".format(
                    tests_launch_arguments,
                    "--ya-testcase-filter={}".format(",".join(pack_cases)))
            new_pack['launch_arguments']["mergeArray"] = True
            result.append(new_pack)

        return result

    def _generate_binary_packs(self, binary_packs_dict):
        result = []
        for _node, cases in binary_packs_dict.iteritems():
            _cases = set(_c["case_id"] for _c in cases)
            # distribute cases to parts of at least a third of nightly builds
            min_cases_in_pack = max(int(round(self._binary_tests_specification["total"][_node] / BINARY_TESTS_NODE_INSTANCES / 3)), 1)

            packs_count = min(int(math.ceil(float(len(_cases)) / min_cases_in_pack)),
                              BINARY_TESTS_NODE_INSTANCES)

            pack_templates = [set() for _ in range(packs_count)]
            _next_pack = 0
            for case_id in _cases:
                pack_templates[_next_pack].add(case_id)
                _next_pack = _next_pack + 1 if _next_pack < packs_count - 1 else 0

            for num, _pack_tpl in enumerate(pack_templates):
                new_pack = deepcopy(BINARY_TESTS_PACK)
                new_pack["node_label"] = _node
                new_pack["pack_name"] = "{}_binary_{}".format(_node, num)
                new_pack["launch_arguments"][
                    "tests_launch_arguments"] = "--ya-testcase-filter={}".format(",".join(set(_pack_tpl)))
                result.append(new_pack)

        return result

    def get_launch_config_by_cases(self, cases):

        result = deepcopy(self._python_tests_specification["launch_config"])
        result.setdefault('launch_arguments', {}).setdefault('ya-settings', {})[
            'skip_all_tests'] = "No"

        python_packs_dict = {}
        binary_packs_dict = {}
        for case_id, case_platform in cases:
            case_info = self._get_case_info(case_id, case_platform)
            if case_info["python"]:
                for pack_name in case_info["python"]["packs"]:
                    python_packs_dict.setdefault(pack_name, []).append(
                        case_info["python"])

            if case_info["binary"]:
                binary_packs_dict.setdefault(
                    case_info["binary"]["node_label"], []).append(case_info["binary"])

        result["packs"] = self._generate_python_packs(python_packs_dict) + self._generate_binary_packs(binary_packs_dict)
        if not result["packs"]:
            return None
        return result

    def _get_case_info(self, case_id, testrun_environment):
        autotest_platform = TestingPlatforms.get_allure_platform_name(testrun_environment)
        python_info = self._python_tests_specification.get("cases", {}).get(case_id, {}).get("platforms", {}).get(autotest_platform, {})
        if python_info:
            python_info.update({"case_id": case_id})

        return dict(
            python=python_info,
            binary=self._binary_tests_specification.get("cases", {}).get(case_id, {}).get("platforms", {}).get(autotest_platform, {})
        )

    def is_automated_in_tests(self, case_id, testrun_environment, ignore_blacklists=False):
        case_info = self._get_case_info(case_id, testrun_environment)
        if not ignore_blacklists:
            return (
                not case_info["python"].get("blacklisted", True)) or (
                not case_info["binary"].get("blacklisted", True)
            )
        else:
            return bool(case_info["python"] or case_info["binary"])

    @sandbox.common.utils.singleton
    def get_all_python_cases(self, platform_type, ignore_blacklists=True):
        _platform_names = [
            _p.value['allure_name'] for _p in PYTHON_TESTS_PLATFORMS.get(platform_type, [])
        ]
        result = []
        for case_id, platforms in self._python_tests_specification.get('cases', {}).iteritems():
            for _platform in platforms.get('platforms', {}):
                if _platform not in _platform_names or (case_id, _platform) in result:
                    continue
                if not ignore_blacklists and not self.is_automated_in_tests(case_id, _platform, False):
                    continue
                result.append((case_id, _platform))
        return result


def get_latest_autotests_bundle_by_browser_build(branded_browser_build_id,
                                                 browser_tests_build_id,
                                                 ya_clients):
    branded_browser_build = ya_clients.teamcity.Build(id=branded_browser_build_id)
    browser_tests_build = ya_clients.teamcity.Build(id=browser_tests_build_id) if browser_tests_build_id else None

    if branded_browser_build.branch_name in {'master', 'master-next', 'master-mobile', '<default>'}:
        framework_branch = 'bundle/master'
    else:
        r = re.search(r'((\d+\.\d+)\.\d+)\.\d+/\d+', branded_browser_build.number)
        framework_branch = 'bundle/feature/{}'.format(r.group(2))

    branches = list(
        ya_clients.bitbucket.get_branches(FRAMEWORK_BITBUCKET_PROJECT, FRAMEWORK_BITBUCKET_REPO, filter_text=framework_branch))

    if not branches:
        return None

    return AutotestBundle(branches[0]['latestCommit'],
                          browser_tests_build,
                          ya_clients)


def get_latest_autotests_bundle_by_framework_branch(framework_branch,
                                                    browser_tests_build_id,
                                                    ya_clients):
    framework_branch = 'bundle/{}'.format(framework_branch)
    branches = list(
        ya_clients.bitbucket.get_branches(FRAMEWORK_BITBUCKET_PROJECT, FRAMEWORK_BITBUCKET_REPO, filter_text=framework_branch))

    if not branches:
        return None
    return AutotestBundle(branches[0]['latestCommit'],
                          ya_clients.teamcity.Build(id=browser_tests_build_id) if browser_tests_build_id else None,
                          ya_clients)


def get_isolate_autotest_bundle(browser_tests_build_id, ya_clients, tested_application='browser'):
    browser_tests_build = ya_clients.teamcity.Build(id=browser_tests_build_id)
    bundle_class = IsolateAutotestBundle if tested_application == 'browser' else SearchappIsolateAutotestBundle
    return bundle_class(browser_tests_build, ya_clients)


def get_android_bro_autotest_bundle(browser_tests_build_id, browser_build_id, ya_clients):
    browser_tests_build = ya_clients.teamcity.Build(id=browser_tests_build_id)
    browser_build = ya_clients.teamcity.Build(id=browser_build_id) if browser_build_id else None
    return AndroidAutotestsBundle(browser_tests_build, browser_build, ya_clients)


def get_android_searchapp_autotest_bundle(browser_tests_build_id, browser_build_id, fake_build_id, ya_clients):
    browser_tests_build = ya_clients.teamcity.Build(id=browser_tests_build_id)
    browser_build = ya_clients.teamcity.Build(id=browser_build_id) if browser_build_id else None
    fake_build = ya_clients.teamcity.Build(id=fake_build_id) if fake_build_id else None
    return AndroidSearchappAutotestBundle(browser_tests_build, browser_build, fake_build, ya_clients)


def get_desktop_autotest_bundle(browser_tests_build_id, browser_build_id, fake_build_id, ya_clients):
    browser_tests_build = ya_clients.teamcity.Build(id=browser_tests_build_id)
    browser_build = ya_clients.teamcity.Build(id=browser_build_id) if browser_build_id else None
    fake_build_id = ya_clients.teamcity.Build(id=fake_build_id) if fake_build_id else None
    return DesktopAutotestsBouldle(browser_tests_build, browser_build, fake_build_id, ya_clients)
