# -*- coding: utf-8 -*-
from collections import OrderedDict
import itertools
import logging
import time

import urlparse
import jinja2

import sandbox
from sandbox.projects.browser.autotests_qa_tools.common import (
    build_number_tuple, get_platform, get_browser_version_number, is_dev_sandbox)
from sandbox.projects.browser.autotests_qa_tools.classes.group_regression_manager.common import GroupRegressionManager, RUN_CREATION_PAUSE, TICKET_JINJA_PACKAGE_PATH
from sandbox.projects.browser.autotests_qa_tools.classes.ya_clients import create_startrek_version_if_not_exists
from sandbox.projects.browser.autotests_qa_tools.classes.testpalm_helper import version_url


logger = logging.getLogger(__file__)
logger.setLevel(logging.DEBUG)


class _DbroGroupRegressionManager(GroupRegressionManager):

    assessor_files_order = {
        "default": [
            'build',
        ]
    }

    @property
    @sandbox.common.utils.singleton
    def group_ticket_properties(self):
        properties = self._group_ticket_properties
        properties['components'].append(self.platform)
        properties['deadline'] = self._main_context.deadline
        properties['affectedVersions'] = [
            create_startrek_version_if_not_exists(
                self.clients.startrek,
                self.assessors_ticket_queue,
                self.st_version)]
        return properties

    @property
    @sandbox.common.utils.singleton
    def assessors_ticket_properties(self):
        properties = self._assessors_ticket_properties
        properties['components'].append(self.platform)
        properties['deadline'] = self._main_context.deadline
        properties['affectedVersions'] = [
            create_startrek_version_if_not_exists(
                self.clients.startrek,
                self.assessors_ticket_queue,
                self.st_version)]
        return properties

    @property
    @sandbox.common.utils.singleton
    def tested_builds(self):
        browser_build = self.clients.teamcity.Build(id=self._parameters.build_id)
        fake_build = (self.clients.teamcity.Build(id=self._parameters.fake_build)
                      if getattr(self._parameters, "fake_build", None) else None)
        if fake_build and browser_build.state == 'finished' and fake_build.state == 'finished':
            browser_build, fake_build = sorted([browser_build, fake_build], key=build_number_tuple)
        result = {
            "browser_build": browser_build,
        }
        if fake_build:
            result["fake_build"] = fake_build
        if getattr(self._parameters, "browser_tests_build_id", None):
            result["tests_build"] = self.clients.teamcity.Build(id=self._parameters.browser_tests_build_id)
        return result

    @property
    def manual_version_template(self):
        return '_'.join([
            self.version_number,
            self._parameters.regression_type.replace(".yaml", "", 1),
            self.platform,
            'Manual']
        )

    @property
    def assessor_version_template(self):
        return '_'.join(
            [self.version_number,
             self.platform,
             self._parameters.regression_type.replace(".yaml", "", 1),
             self.group.name])

    @property
    @sandbox.common.utils.singleton
    def version_number(self):
        return get_browser_version_number(self.tested_builds["browser_build"])

    def get_manual_run_title(self, run_template):
        return u''.join('[{}]'.format(v) for v in [
            u'Regression', self.version_number,
            self._parameters.regression_type.replace(".yaml", "", 1), ]) + u'[{platform}][{component}] Manual run'.format(
                platform=self.platform,
                component=run_template["component"])

    def get_manual_run_properties(self, run_template):
        return {'Brand/PartnerID': u"{}/{}".format(
                run_template["build_info"]["distribution_type"],
                run_template["build_info"]["distribution_name"])}

    @property
    @sandbox.common.utils.singleton
    def platform(self):
        return get_platform(self.tested_builds["browser_build"])

    @property
    @sandbox.common.utils.singleton
    def st_version(self):
        return self.version_number

    @property
    @sandbox.common.utils.singleton
    def ticket_summary_template(self):
        return u''.join('[{}]'.format(v) for v in [u'Regression',
                                                   self.version_number,
                                                   self.platform,
                                                   self._parameters.regression_type.replace(".yaml", "", 1),
                                                   self.group.name])

    def render_assessor_ticket_template(self, runs):
        data = dict(
            group=self.group,
            testpalm_versions=set(set(version_url(run["instance"].project,
                                                  run["instance"].version)
                                  for run in runs)),
            builds=self._main_context.assessors_links,
            assessors_launch_comment=self._parameters.assessors_launch_comment)

        return self.assessor_ticket_template.render(**data)

    @property
    @sandbox.common.utils.singleton
    def assessors_ticket_summary_template(self):
        return self.ticket_summary_template

    def get_assessor_run_title(self, version, run_template, platform):
        return u'[{version}][{platform}][{component}] Part {part}'.format(
            version=version,
            part=run_template["part"],
            component=run_template["component"],
            platform=self.platform)

    @sandbox.common.utils.singleton
    def get_assessors_links(self, run_template):
        return self._main_context.assessors_links[run_template["build_info"]["distribution_type"]][
            run_template["build_info"]["distribution_name"]]

    def create_assessor_runs(self, run_temlpates):
        created_runs = []
        for run_template in run_temlpates:
            version = self.get_assessor_version(run_template)
            logger.debug(" ############## asessor_version = {}".format(version))
            run_title = self.get_assessor_run_title(version=version,
                                                    run_template=run_template,
                                                    platform=self.platform)

            assessors_links = self.get_assessors_links(run_template)
            testrun_properties = OrderedDict(
                {'Brand/PartnerID': u"{}/{}".format(run_template["build_info"]["distribution_type"],
                                                    run_template["build_info"]["distribution_name"])}.items() +
                [(assessors_file, assessors_links[assessors_file])
                 for assessors_file in self.assessor_files_order.get("Network" if self.group.name.startswith("Network") else self.group.name,
                                                                     self.assessor_files_order['default'])
                 if assessors_file in assessors_links]
            )

            run_cases = {_key: _value for _key, _value in run_template["cases"].iteritems()}
            if run_template["suite_info"].get("honey_logs"):
                honey_logs_case = self.honey_logs_cases.get(run_template["project"])
                if honey_logs_case is not None and honey_logs_case not in run_cases:
                    run_cases[honey_logs_case] = {}

            run_id = self.clients.testpalm.create_testrun(
                version=version,
                cases=self._run_template_2_run_cases_list(run_cases),
                project=run_template["project"],
                title=run_title,
                properties=testrun_properties,
                current_environment=run_template["environment"],
                tags=run_template["suite_info"]["tags"],
            )

            run_instance = self.clients.testpalm.get_testrun(run_id,
                                                             run_template["project"])
            logger.info("assessors run added: {}".format(run_instance.url))
            created_runs.append({"suite_info": run_template["suite_info"],
                                 "instance": run_instance})
            time.sleep(RUN_CREATION_PAUSE)

        return created_runs

    def publish_group_diff_tickets(self, diff_issues, old_build_url, new_build_url, additional_issues_query):

        if not self.group_issue_key:
            self.create_group_ticket()

        self.comment_about_ticket_diff(diff_issues, old_build_url, new_build_url, additional_issues_query)

        if not is_dev_sandbox():
            fields = {
                'testingStoryPoints': (self.group_issue.testingStoryPoints or 0) + diff_issues['testingStoryPoints'],
                'ignore_version_change': True
            }
            self.clients.startrek.issues.update(self.group_issue, **fields)

    def comment_about_ticket_diff(self, diff_issues, old_build_url, new_build_url, additional_issues_query):

        env = jinja2.Environment(
            loader=jinja2.PackageLoader(TICKET_JINJA_PACKAGE_PATH,
                                        package_path=self.ticket_jinja_template_package))
        template = env.get_template('group_builds_diff_issues_startrek.jinja')
        diff_report = template.render(
            old_build_url=old_build_url,
            new_build_url=new_build_url,
            maybe_new_issues=group_by_priority(
                diff_issues['maybe_new_issues']),
            new_issues=group_by_priority(diff_issues['new_issues']),
            old_issues=group_by_priority(diff_issues['old_issues']),
            additional_issues=group_by_priority(diff_issues['additional_issues']),
            additional_issues_query=additional_issues_query,
        )
        self._update_issue_description(diff_report)
        logger.info("UPDATED about_ticket_diff {}".format(urlparse.urljoin(self.clients.st_base_url, self.group_issue_key)))


def group_by_priority(tickets):
    def get_priority(ticket):
        if hasattr(ticket, 'priority'):
            return ticket.priority.display
        else:
            return 'Unknown'

    return OrderedDict((k, list(v)) for k, v in itertools.groupby(tickets, get_priority))
