import logging
import requests
import urlparse

from sandbox.common.utils import Enum, GroupEnum

HITMAN_URL = 'https://hitman.yandex-team.ru'
HITMAN_TESTING_URL = 'https://sandbox.hitman.yandex-team.ru'


class JobStatus(Enum):
    class Group(GroupEnum):
        # primary groups
        RUNNING = None
        FINISHED = None

        # secondary groups
        ALL = None
        SUCCESS = None
        FAIL = None

    with Group.RUNNING:
        RUNNING = None
        NEW = None
        RESUMING = None
        IN_CANCELLATION = None

    with Group.FINISHED:
        START_FAILED = None
        FAILED = None
        CANCELED = None
        SUCCEEDED = None
        FIXED = None


JobStatus.Group.ALL = tuple(JobStatus)
JobStatus.Group.SUCCESS = (JobStatus.SUCCEEDED, JobStatus.FIXED)
JobStatus.Group.FAIL = (JobStatus.START_FAILED, JobStatus.FAILED, JobStatus.CANCELED)


class HitmanClient(object):
    def __init__(self, url, token):
        self.token = token
        self.url = url
        self.session = requests.Session()

    def start_process(self, process_id, requester, properties, job_token=None, comment=None):
        """
        https://wiki.yandex-team.ru/hitman/api/#zapustitvypolnenieprocessa

        :param process_id:
        :param properties: variables values for process
        :param job_token: if job with such token exists, new job won't be launched, id of existing job will be returned
        :return: {'id': <id of started process>, 'url' : <url>}
        """
        url = urlparse.urljoin(self.url, '/api/v1/execution/start/{}'.format(process_id))
        data = {'requester': requester, 'comment': comment, 'properties': properties}
        params = {'token': job_token} if job_token else {}
        return self.request('post', url, params, json_data=data)

    def get_job_status(self, job_id):
        """

        :param job_id:
        :return: dict info about job execution
        """
        url = urlparse.urljoin(self.url, '/api/v1/execution/{}'.format(job_id))
        return self.request('get', url)

    def request(self, method, path, params=None, data=None, json_data=None):
        headers = {'Authorization': 'OAuth {}'.format(self.token)}
        response = self.session.request(method, path, params=params, data=data, headers=headers, json=json_data)
        if response.status_code != 200:
            logging.error(response.content)
        response.raise_for_status()
        return response.json()

    def get_job_url(self, job_id):
        return urlparse.urljoin(self.url, 'r/jobs/{}'.format(job_id))
