import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

JNS_URL = 'https://jns.yandex-team.ru'


class JNSClient(object):
    def __init__(self, token, server_url=JNS_URL):
        self._oauth_token = token
        self.server_url = server_url

    def make_request(self, method, path, params=None, data=None, json=None):
        url = '{}/{}'.format(self.server_url, path)
        headers = {
            'Authorization': 'OAuth ' + self._oauth_token,
            'Content-Type': 'application/json',
            'Accept': 'application/json'
        }
        response = _retry_session().request(method=method, url=url, params=params, data=data, headers=headers,
                                            timeout=120, verify=False, json=json)
        response.raise_for_status()
        try:
            return response.json()
        except Exception:
            try:
                return {
                    'text': response.text,
                    'status_code': response.status_code
                }
            except Exception:
                return {'status_code': response.status_code}

    def send_to_channel(self, project, channel, template, message_json):
        msg_format = {
            'project': project,
            'template': template,
            'channel': channel,
            'target_project': project,
            'params': message_json,
        }
        return self.make_request("POST", 'api/messages/send_to_channel_json', json=msg_format)


def _retry_session(retries=1, backoff_factor=0.3, session=None):
    session = session or requests.Session()
    retry = Retry(
        total=retries,
        read=retries,
        connect=retries,
        backoff_factor=backoff_factor
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session
