# -*- coding: utf-8 -*-
import logging

from collections import defaultdict

import sandbox
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.autotests_qa_tools.classes.functionalities import get_components
from sandbox.projects.browser.autotests_qa_tools.common import (
    build_number_tuple, get_platform, get_browser_version_number, is_dev_sandbox, html_cut)
from sandbox.projects.browser.autotests_qa_tools.classes.autotests_bundle import get_desktop_autotest_bundle
from sandbox.projects.browser.autotests_qa_tools.classes.diff_tickets import get_report, sort_issues
from sandbox.projects.browser.autotests_qa_tools.classes.regression_manager.common import RegressionManager
from sandbox.projects.browser.autotests_qa_tools.classes.ya_clients import create_startrek_version_if_not_exists
from sandbox.projects.browser.autotests_qa_tools.classes.group_regression_manager.desktop import DbroGroupRegressionManager

logger = logging.getLogger(__file__)


class _DBroRegressionManager(RegressionManager):

    group_regresson_manager_class = DbroGroupRegressionManager
    main_ticket_queue = "BROWSER"
    ticket_jinja_template_package = "dbro_regression"

    @property
    def tested_application(self):
        return 'browser'

    @property
    def _diff_get_components_method(self):
        return get_components

    @property
    @sandbox.common.utils.singleton
    def tested_builds(self):
        browser_build = self.clients.teamcity.Build(id=self._parameters.build_id)
        fake_build = (self.clients.teamcity.Build(id=self._parameters.fake_build_id)
                      if getattr(self._parameters, "fake_build_id", None) else None)
        if fake_build and browser_build.state == 'finished' and fake_build.state == 'finished':
            browser_build, fake_build = sorted([browser_build, fake_build], key=build_number_tuple)
        result = {
            "browser_build": browser_build,
        }
        if fake_build:
            result["fake_build"] = fake_build
        if getattr(self._parameters, "browser_tests_build_id", None):
            result["tests_build"] = self.clients.teamcity.Build(id=self._parameters.browser_tests_build_id)
        return result

    @property
    @sandbox.common.utils.singleton
    def autotests_bundle(self):
        if not self._parameters.enable_autotests or self._context.autotests_timeout <= 0 or not self.tested_builds.get("tests_build"):
            return None
        return get_desktop_autotest_bundle(
            self.tested_builds["tests_build"].id,
            self.tested_builds["browser_build"].id,
            self.tested_builds["fake_build"].id if self.tested_builds.get("fake_build") else None,
            self.clients)

    def should_check_case(self, case, suite):
        if not any(os.startswith(self.platform) for os in case.oses):
            logger.warning(u'Кейс {} не для {} платформы'.format(case.url, self.platform))
            return False

        if suite.filter_by_diff:
            if not self.is_case_in_diff(case):
                logging.warning(u'{} кейса {} не затронуты в диффе'.format(
                    u"Функциональности" if self._parameters.diff_type == "functionalities_diff" else u"Компоненты",
                    case.url))
                return False
        return True

    def is_automated(self, case, testrun_environment):
        return bool(
            self.autotests_bundle is not None and
            self.is_automated_in_testpalm(case, testrun_environment) and
            self.autotests_bundle.is_automated_in_tests(
                "{}-{}".format(
                    case.project, case.id),
                testrun_environment)
        )

    @property
    @sandbox.common.utils.singleton
    def version_number(self):
        return get_browser_version_number(self.tested_builds["browser_build"])

    @property
    @sandbox.common.utils.singleton
    def platform(self):
        return get_platform(self.tested_builds["browser_build"])

    @property
    @sandbox.common.utils.singleton
    def st_version(self):
        return self.version_number

    @property
    def main_ticket_properties(self):
        properties = self._main_ticket_properties
        properties['deadline'] = self._context.deadline
        properties['components'].append(self.platform)
        properties['affectedVersions'] = [create_startrek_version_if_not_exists(
            self.clients.startrek,
            self.main_ticket_queue,
            self.st_version)
        ]
        return properties

    def create_main_ticket(self):
        if self.main_ticket is not None:
            raise RuntimeError(u"Main ticket already created: {}".format(self.main_ticket))

        old_build = self._parameters.old_build_id
        description = self.main_ticket_template.render(
            brand=self._parameters.distribution_name,
            oses=self.all_manual_oses,
            release_ticket=self._parameters.release_ticket,
            configuration_ticket=self._parameters.configuration_ticket,
            browser_build=self.tested_builds["browser_build"].web_url,
            fake_build=self.tested_builds.get("fake_build").web_url if self.tested_builds.get("fake_build", None) else None,
            tests_build=self.tested_builds.get("tests_build").web_url if self.tested_builds.get("tests_build", None) else None,
            previous_build=self.clients.teamcity.Build(id=old_build).web_url if int(old_build) else None,
            launch_comment=self._parameters.manual_launch_comment,
            task_link=get_task_link(self._context.task_id))
        issue_dict = {
            'description': description,
            'queue': self.main_ticket_queue,
            'summary': u'[{version}][{platform}][{type}] Регрессия'.format(
                version=self.version_number, platform=self.platform, type=self._parameters.regression_type.replace(".yaml", "", 1)
            )
        }
        issue_dict.update(self.main_ticket_properties)
        self.main_ticket = self.clients.startrek.issues.create(**issue_dict)['key']
        return self.main_ticket

    def _get_diff_builds_urls(self):
        old_build_url = self.clients.teamcity.Build(
            id=self._parameters.old_build_id).web_url
        new_build_url = self.clients.teamcity.Build(
            id=self._parameters.build_id).web_url
        return old_build_url, new_build_url

    def publish_diff_tickets(self):
        if not self._parameters.check_resolves:
            return

        new_issues, maybe_new_issues, old_issues = self.get_filtered_diff(
            self.regression_config.get("aggregator_restricting_filter"))
        old_build_url, new_build_url = self._get_diff_builds_urls()

        if self.regression_config.get("aggregator_restricting_filter") and self._parameters.resolves_filter:
            query = '({}) AND ({})'.format(
                self.regression_config.get("aggregator_restricting_filter"),
                self._parameters.resolves_filter,
            )
        else:
            query = self.regression_config.get("aggregator_restricting_filter") or self._parameters.resolves_filter
        additional_issues = set(self.clients.prod_startrek.issues.find(query)) if query else {}
        additional_issues = sort_issues({issue for issue in additional_issues
                                         if issue.key not in {
                                             diff_issue.key for diff_issue in
                                             new_issues + maybe_new_issues + old_issues
                                         }}, self.clients.prod_startrek)

        group_issues = defaultdict(
            lambda: {
                'new_issues': list(),
                'maybe_new_issues': list(),
                'old_issues': list(),
                'additional_issues': list(),
                'testingStoryPoints': 0,
            }
        )

        for key, issues in {'new_issues': new_issues,
                            'maybe_new_issues': maybe_new_issues,
                            'old_issues': old_issues,
                            'additional_issues': additional_issues}.iteritems():
            for issue in issues:
                if hasattr(issue, 'status') and issue.status.key.lower() in ['tested', 'closed']:
                    continue
                components = [component.display for component in
                              getattr(issue, 'components', [])]
                if not components:
                    groups = {self.default_group}
                else:
                    groups = {self.get_group_by_component(component) for component in
                              components if self.get_group_by_component(component) is not None} or {self.default_group}
                    if len(groups) > 1:
                        groups.discard(self.default_group)
                # logging.debug('issue %s going to groups %s', issue.key,
                #               {group.name for group in groups})
                for group in groups:
                    group_issues[group][key].append(issue)
                    group_issues[group]['testingStoryPoints'] += getattr(issue,
                                                                         'testingStoryPoints',
                                                                         0) or 0

        for group, diff_issues in group_issues.iteritems():
            group_manager = self.get_or_create_group_manager(group)
            group_manager.publish_group_diff_tickets(
                diff_issues,
                old_build_url,
                new_build_url,
                query,
            )

        if self.main_ticket:
            report = get_report(old_build_url, new_build_url, new_issues,
                                maybe_new_issues,
                                old_issues, additional_issues, query)
            totaltsp = sum(
                diff_issues['testingStoryPoints'] for group, diff_issues in
                group_issues.iteritems())

            def _modifier(issue):
                result = {
                    'description': u'{}\n=== Дифф по тикетам\n<#{}#>'.format(
                        issue.description,
                        html_cut(u'Дифф по тикетам', report))
                }
                if not is_dev_sandbox():
                    result['testingStoryPoints'] = (issue.testingStoryPoints or 0) + totaltsp
                return result

            self.clients.safely_update_issue(
                self.main_ticket,
                modifier=_modifier
            )
