# -*- coding: utf-8 -*-
import logging

import sandbox
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.autotests_qa_tools.classes.autotests_bundle import get_isolate_autotest_bundle
from sandbox.projects.browser.autotests_qa_tools.classes.regression_manager.common import RegressionManager
from sandbox.projects.browser.autotests_qa_tools.classes.group_regression_manager.common_mobile import MobileGroupRegressionManager

logger = logging.getLogger(__file__)


class MobileRegressionManager(RegressionManager):

    ticket_jinja_template_package = "mobile_regression"
    group_regresson_manager_class = MobileGroupRegressionManager

    @property
    @sandbox.common.utils.singleton
    def tested_builds(self):
        result = {}
        if self._parameters.browser_tests_build_id:
            result["tests_build"] = self.clients.teamcity.Build(id=self._parameters.browser_tests_build_id)
        if self._parameters.build_id:
            result["browser_build"] = self.clients.teamcity.Build(id=self._parameters.build_id)
        if self._parameters.fake_build_id:
            result["fake_build"] = self.clients.teamcity.Build(id=self._parameters.fake_build_id)
        return result

    @property
    @sandbox.common.utils.singleton
    def autotests_bundle(self):
        if not self._parameters.enable_autotests or self._context.autotests_timeout <= 0 or not self.tested_builds.get(
                "tests_build"):
            return None
        return get_isolate_autotest_bundle(self.tested_builds["tests_build"].id,
                                           self.clients,
                                           self.tested_application)

    @property
    def main_ticket_queue(self):
        return self.default_group.manual_ticket_queue

    def create_main_ticket(self):
        if self.main_ticket is not None:
            raise RuntimeError(u"Main ticket already created: {}".format(self.main_ticket))

        description = self.main_ticket_template.render(
            regression_type_name=self.regression_type_name,
            oses=self.all_manual_oses,
            release_configuration=self._parameters.release_configuration,
            builds=self._parameters.builds,
            common_regression_ticket=self._parameters.common_regression_ticket,
            launch_comment=self._parameters.manual_launch_comment,
            task_link=get_task_link(self._context.task_id))

        issue_dict = {
            'description': description,
            'queue': self.main_ticket_queue,
            'summary': u'[{version}] Регрессия'.format(version=self._parameters.version),
        }
        issue_dict.update(self.main_ticket_properties)
        self.main_ticket = self.clients.startrek.issues.create(**issue_dict)['key']
        return self.main_ticket

    def should_check_case(self, case, suite):
        if suite.filter_by_diff:
            if not self.is_case_in_diff(case):
                logging.warning(u'{} кейса {} не затронуты в диффе'.format(
                    u"Функциональности" if self._parameters.diff_type == "functionalities_diff" else u"Компоненты",
                    case.url))
                return False
        return True

    def get_automated_info(self):
        result = None
        automated_cases = set(
            ("{}-{}".format(_j["case_project"], _j["case_id"]), _j["testrun_environment"])
            for _j in self.regression_jobs if _j["is_automated"])

        if automated_cases:
            launch_config = self.autotests_bundle.get_launch_config_by_cases(automated_cases)
            if launch_config is not None:
                result = {
                    "launch_config": launch_config,
                    "browser_branch": self.autotests_bundle.browser_branch,
                    "browser_commit": self.autotests_bundle.browser_commit,
                    "build_extra_args": self.autotests_bundle.build_extra_args,
                    "number_of_shards": self.autotests_bundle.number_of_shards,
                    "tested_application": self.autotests_bundle.tested_application,
                    "summary": {
                        "total_count": len(self.regression_jobs),
                        "automated_count": len(automated_cases)
                    }}
            else:
                logger.error("Empty bundle autotest_config for automated cases")
        return result
