# -*- coding: utf-8 -*-
from copy import deepcopy
from datetime import timedelta
from enum import Enum

import sandbox
from sandbox.projects.browser.autotests_qa_tools.classes.testing_job import get_jobs_statistic, JobExecutors


JNS_PROJECT = 'bro-infra'


class Notifications(Enum):

    common = {
        "regression_failed": {
            "channels": [],
            "template": ""
        },
        "regressoin_summary": {
            "channels": [],
            "template": ""
        }
    }

    android_alpha = {
        "regression_failed": {
            "channels": ["qa_tools_common"],
            "template": "qa_regression_failed"
            },
        "regressoin_summary": {
            "channels": ["qa_android_alpha"],
            "template": "qa_abro_regression_summary"
        }
    }

    desktop_common = {
        "regression_failed": {
            "channels": ["qa_desktop_regressions", "qa_tools_common"],
            "template": "qa_regression_failed"
        },
        "regressoin_summary": {
            "channels": ["qa_desktop_regressions"],
            "template": "qa_desktop_regression_summary"
        }
    }


class RegressionReport():

    _notifications = {}

    def __init__(self, task_info, regression_state, clients=None):
        self.task_info = task_info
        self.regression_state = regression_state
        self.clients = clients

    @property
    @sandbox.common.utils.singleton
    def common_task_data(self):
        return {
            'task_id': self.task_info['task_id'],
            'regression_config': self.regression_state['regression_config'],
            'regression_type': self.regression_state['context']['regression_type_name'],
            'regression_main_ticket': self.regression_state['context']['main_ticket'],
            'date': self.task_info['date']
        }

    @property
    @sandbox.common.utils.singleton
    def jobs_stat(self):
        jobs_stat = {
            'total_jobs': [],
            'sent_to_work': [],
            'autotests_success': [],
            'autotests_status': {}
        }

        for group, group_data in self.regression_state['context']['groups'].iteritems():
            _common_task_data = deepcopy(self.common_task_data)
            _common_task_data.update({"group_name": group})
            jobs_stat['total_jobs'] += get_jobs_statistic(group_data['automated_jobs'] + group_data['no_automated_jobs'], _common_task_data)

            jobs_stat['sent_to_work'] += group_data['jobs_statistics']['sent_to_work']
            jobs_stat['autotests_success'] += group_data['jobs_statistics']['autotests_success']
            jobs_stat['autotests_status'].update(group_data['jobs_statistics']['autotests_status'])

        return jobs_stat

    @property
    @sandbox.common.utils.singleton
    def summary_report(self):
        common_task_data = self.common_task_data
        jobs_stat = self.jobs_stat
        def _total_time(jobs):
            return reduce(lambda x, y: x + y, [_j.get('estimate', 0) for _j in jobs]) if jobs else 0

        def _get_summary_group(sub_group, total=None):
            sub_count = len(sub_group)
            sub_time = _total_time(sub_group)
            percentage_count = round((float(sub_count) * 100) / len(total), 2) if total else None
            percentage_time = round((float(sub_time) * 100) / _total_time(total), 2) if _total_time(total) != 0 else None
            return {
                'count': sub_count,
                'time': round(float(sub_time) / (1000 * 60 * 60), 2),
                'percentage_count': percentage_count,
                'percentage_time': percentage_time
            }

        total_jobs = jobs_stat['total_jobs']
        summary = deepcopy(common_task_data)
        summary['total'] = _get_summary_group(total_jobs)
        summary['total_asessors'] = _get_summary_group(filter(lambda x: x['executor'] == JobExecutors.assessor.value, total_jobs))

        _automated = filter(lambda x: x['is_automated'], total_jobs)
        summary['automated'] = _get_summary_group(
            filter(lambda x: x['is_automated'], total_jobs),
            total_jobs)
        _autotests_success = filter(lambda x: x['uuid'] in jobs_stat["autotests_success"], total_jobs)
        summary['autotests_success'] = _get_summary_group(
            _autotests_success,
            _automated)
        _autotests_no_success = filter(lambda x: x['uuid'] not in jobs_stat["autotests_success"], _automated)
        summary['autotests_no_success'] = _get_summary_group(_autotests_no_success, _automated)

        summary['autotests_total_success'] = _get_summary_group(_autotests_success, total_jobs)
        _sent_to_work = []
        for job_id in jobs_stat['sent_to_work']:
            _sent_to_work.extend(filter(lambda x: x['uuid'] == job_id, total_jobs))
        summary['sent_to_work'] = _get_summary_group(_sent_to_work, total_jobs)
        summary['sent_to_work_asessors'] = _get_summary_group(
            filter(lambda x: x['executor'] == JobExecutors.assessor.value, _sent_to_work),
            _sent_to_work)
        return summary

    @property
    @sandbox.common.utils.singleton
    def summary_text(self):
        _text = ''
        for group in ['total', 'autotests', 'total_autotests']:
            if _text:
                _text += " --------\n"
            for _m in self.summary_message[group]:
                _text += _m + "\n"
        return _text

    @property
    @sandbox.common.utils.singleton
    def summary_message(self):
        summary = self.summary_report
        def _hours_to_dhm(hours):
            duration = timedelta(hours=hours)
            days, seconds = duration.days, duration.seconds
            hours = seconds // 3600
            minutes = (seconds % 3600) // 60
            message = "{}{}{}".format(
                "{} days, ".format(days) if days else "",
                "{} hours, ".format(hours) if hours else "",
                "{} minutes, ".format(minutes) if minutes else "")
            return message if not message else message[:-2]

        def _print_summary_group(title, summary_group):
            message = u"{}: {}{} тестовых сценариев{}".format(
                title,
                summary_group['count'],
                u" ({}%)".format(summary_group['percentage_count']) if summary_group['percentage_count'] else u"",
                u" | {}{} человек-часов | {}".format(
                    summary_group['time'],
                    u" ({}%)".format(summary_group['percentage_time']) if summary_group['percentage_time'] else u"",
                    _hours_to_dhm(summary_group['time'])
                ) if summary_group['time'] else u""
            )
            return message

        message = {
            'total': [],
            'autotests': [],
            'total_autotests': [],
        }

        message['total'].append(_print_summary_group(u"Всего в регрессии", summary['total']))
        message['total'].append(_print_summary_group(u"Всего для асессоров ", summary['total_asessors']))
        message['total'].append(_print_summary_group(u"Всего автоматизированно", summary['automated']))

        if self.regression_state['context']['autotests_task_id']:
            message['autotests'].append(u"Запуск автотестов:  https://sandbox.yandex-team.ru/task/{}/, статус таски: {}, отчёт: {}".format(
                self.regression_state['context']['autotests_task_id'],
                self.regression_state['context']['autotests_task_status'],
                self.regression_state['context']['autotests_report_url'] if self.regression_state[
                    'context']['autotests_report_url'] else u"отсутствует"
            ))
            message['autotests'].append(_print_summary_group(u"Успешно автотестов", summary['autotests_success']))
            message['autotests'].append(_print_summary_group(u"Не успешно автотестов", summary['autotests_no_success']))
        else:
            message['autotests'].append(u"Автотесты не запускались")

        message['total_autotests'].append(
            _print_summary_group(u"Всего пройдено автотестами", summary['autotests_total_success']))
        message['total_autotests'].append(
            _print_summary_group(u"Всего отправлено в ручную работу", summary['sent_to_work']))
        message['total_autotests'].append(_print_summary_group(u"Из них асессорам", summary['sent_to_work_asessors']))
        return message

    def regression_summary_notification(self):
        send_to = self._notifications.get("regressoin_summary")
        if send_to is None:
            return None
        data = dict(
            message_lines=self.summary_message,
            main_ticket=self.regression_state['context']['main_ticket'] or '-',
            task_description=self.regression_state['description'],
            task_id=str(self.task_info['task_id'])
        )
        self._send__notifications(channels=send_to['channels'], template=send_to['template'], data=data)

    def regression_failed_notification(self, task_id, task_status, task_description):
        send_to = self._notifications.get("regression_failed")
        if send_to is None:
            return None

        data = {
            "task_id": str(task_id),
            "task_status": str(task_status),
            "description": str(task_description)
        }
        self._send__notifications(channels=send_to['channels'], template=send_to['template'], data=data)

    def _send__notifications(self, channels, template, data):
        if template:
            for channel in channels:
                self.clients.jns.send_to_channel(project=JNS_PROJECT,
                                                 channel=channel,
                                                 template=template,
                                                 message_json=data)


class BaseRegressionReport(RegressionReport):

    _notifications = Notifications.common.value


class DesktopRegressionReport(BaseRegressionReport):

    _notifications = Notifications.desktop_common.value


class AndroidAlphaReport(RegressionReport):

    _notifications = Notifications.android_alpha.value

    def regression_summary_notification(self):
        send_to = self._notifications.get("regressoin_summary")
        if send_to is None:
            return None

        tickets = [group['group_issue_key'] for group in self.regression_state['context']['groups'].values()]
        if self.regression_state['context']['main_ticket']:
            tickets.append(self.regression_state['context']['main_ticket'])

        data = dict(
            build_type=self.regression_state['summary_notification_type'],
            regression_assignee=self.regression_state['main_regression_ticket_assignee'],
            browser_build_id=str(self.regression_state['browser_build_id']),
            task_id=str(self.task_info['task_id']),
            tickets=tickets,
            total_autotests=self.summary_message['total_autotests']
        )
        self._send__notifications(channels=send_to['channels'], template=send_to['template'], data=data)


def get_report(report_type, params, default=None):

    classes = {
        "desktop_common": DesktopRegressionReport,
        "android_browser_beta": AndroidAlphaReport,
        "android_browser_stable": AndroidAlphaReport,
        "android_browser_alpha": AndroidAlphaReport
    }
    if report_type in classes:
        return classes[report_type](*params)
    elif default:
        return default(*params)
    else:
        return None
