# -*- coding: utf-8 -*-
import logging

import sandbox
from sandbox.projects.common import decorators
from sandbox.projects.browser.autotests_qa_tools.common import is_dev_sandbox
from sandbox.projects.browser.autotests_qa_tools.classes.testpalm_helper import TestpalmClientWrapper
from sandbox.projects.browser.autotests_qa_tools.classes.jns_client import JNSClient
from sandbox.projects.browser.autotests_qa_tools.common import TEAMCITY_URL
from sandbox.projects.browser.common.bitbucket import BitBucket, DEFAULT_BITBUCKET_URL

ROBOT_USERAGENT = 'Ya.Browser stat collector'
ST_TEST_BASE_API_URL = 'https://st-api.test.yandex-team.ru'
ST_BASE_API_URL = 'https://st-api.yandex-team.ru'
YA_BOOKING_API_URL = 'https://booking.yandex-team.ru'

logger = logging.getLogger(__file__)


class YaClients(object):

    def __init__(self, auth_token):
        self.auth_token = auth_token
        if is_dev_sandbox():
            logger.setLevel(logging.DEBUG)
            logger.addHandler(logging.StreamHandler())

    @property
    @sandbox.common.utils.singleton
    def st_base_url(self):
        return 'https://st.test.yandex-team.ru' if is_dev_sandbox() else 'https://st.yandex-team.ru'

    @property
    @sandbox.common.utils.singleton
    def startrek(self):
        from startrek_client import Startrek
        return Startrek(
            useragent=ROBOT_USERAGENT,
            token=self.auth_token,
            base_url=ST_TEST_BASE_API_URL if is_dev_sandbox() else ST_BASE_API_URL)

    @property
    @sandbox.common.utils.singleton
    def prod_startrek(self):
        from startrek_client import Startrek
        return Startrek(
            useragent=ROBOT_USERAGENT,
            token=self.auth_token,
            base_url=ST_BASE_API_URL)

    @property
    @sandbox.common.utils.singleton
    def teamcity(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=TEAMCITY_URL,
            auth=self.auth_token)

    @property
    @sandbox.common.utils.singleton
    def testpalm(self):
        from testpalm_api_client.client import TestPalmClient
        return TestpalmClientWrapper(TestPalmClient(oauth_token=self.auth_token))

    @property
    @sandbox.common.utils.singleton
    def bitbucket(self):
        return BitBucket(DEFAULT_BITBUCKET_URL, 'x-oauth-token',
                         self.auth_token)

    @property
    @sandbox.common.utils.singleton
    def ya_booking(self):
        from ya_booking_client.ya_booking_client import YaBookingClient
        return YaBookingClient(self.auth_token, YA_BOOKING_API_URL)

    @sandbox.common.utils.singleton_property
    def yt(self):
        import yt.wrapper as yt
        return yt.YtClient(proxy='hahn', token=self.auth_token)

    @sandbox.common.utils.singleton_property
    def jns(self):
        return JNSClient(token=self.auth_token)

    @sandbox.common.utils.singleton
    def get_boss(self, login):
        from ids.exceptions import ResponseError
        try:
            person = self.get_ids_repository('staff', 'person').get_one(lookup={'login': login})
        except ResponseError:
            logger.error("Failed to get person:{} from staff".format(login))
            return None
        if person['department_group']['department']['heads']:
            boss = person['department_group']['department']['heads'][0]['person']['login']
        elif person['department_group']['parent']['department']['heads']:
            boss = person['department_group']['parent']['department']['heads'][0]['person']['login']
        else:
            logger.error("Failed to get boss of person:{} from staff".format(login))
            boss = None
        return boss

    @sandbox.common.utils.singleton
    def get_ids_repository(self, service, resource_type):
        from ids.registry import registry
        return registry.get_repository(service, resource_type, user_agent='myservice', oauth_token=self.auth_token)

    @decorators.retries(5, delay=1, backoff=2)
    def safely_update_issue(self, issue_key, modifier=None, params=None):
        params = params or {}
        issue = self.startrek.issues[issue_key]
        if modifier:
            params.update(modifier(issue))
        self.startrek.issues.update(issue, **params)

    def get_build_commit(self, build_instance):
        try:
            return build_instance.revisions[0]['version']
        except AttributeError:
            return None

    # The delay is big because Teamcity checks new changes once in a while
    @decorators.retries(5, delay=120, backoff=1.2)
    def get_change(self, commit, browser_build_type_id):
        change = self.teamcity.Change(buildType__id=browser_build_type_id, version=commit)
        change.id  # force lazy client to request change
        return change

    @sandbox.common.utils.singleton
    @decorators.retries(5, delay=120, backoff=1.2)
    def get_build_revision(self, build_id):
        build = self.teamcity.Build(id=build_id) if build_id else None
        return (build.branch_name,
                self.get_build_commit(build),
                build.queued_date) if build else (None, None, None)


@sandbox.common.utils.singleton
@decorators.retries(5, delay=1, backoff=2)
def create_startrek_version_if_not_exists(startrek_client, st_queue, st_version):
    if st_version not in [_v.name for _v in startrek_client.queues[st_queue].versions]:
        startrek_client.versions.create(name=st_version, queue=st_queue)
    return st_version
