import re
import uuid
import os
import zipfile

from sandbox.common.config import Registry
from sandbox.common.types.misc import Installation


BRANDED_BUILDS_PROJECT = 'BrowserBrandedDistributives'
WEBNTP_BUILDS_PROJECT = "Browser_WebNtp"
TESTS_BUILDS_PROJECT = 'Browser_Tests_BuildTests'
ROBOT_BRO_QA_INFRA_TOKEN_VAULT = 'robot-bro-qa-infra-token'
ROBOT_BRO_QA_INFRA_SSH_KEY_VAULT = 'robot-bro-qa-infra-ssh_key'

TEAMCITY_URL = 'https://teamcity.browser.yandex-team.ru'

SPECIFICATION_NAME = "specification_all_tests.json"
SPECIFICATION_PATH = 'specifications'

FRAMEWORK_BITBUCKET_PROJECT = "autotest"
FRAMEWORK_BITBUCKET_REPO = "browser-test-framework"
FRAMEWORK_DEFAULT_REVIEWERS = ['nik-isaev']

BROWSER_BITBUCKET_PROJECT = "stardust"
BROWSER_BITBUCKET_REPO = "browser"

# from RunBrowserAutotests
DEFAULT_REGRESSION_AUTOTESTS_TIMEOUT = 60 * 3  # minutes
DEFAULT_KILL_TIMEOUT = (DEFAULT_REGRESSION_AUTOTESTS_TIMEOUT + 30) * 60  # seconds
DEFAULT_MAX_FLAKY_RETRIES = 1
INPUT_PATH = 'input.json'
RESULT_DIR = "results"

# regression groups
REGRESSION_GROUPS_ACTIVITIES = ("regression",)

# upload artifacts to s3
DEFAULT_DESKTOP_DISTRIBUTION_CHANNEL = 'beta'


def build_exists(build):
    try:
        build.id
        return True
    except:
        return False


def check_build(build, expected_build_type):
    if not build_exists(build):
        return ['Build with #{} not found'.format(build.id)]
    if not build.build_type.project.id == expected_build_type:
        return ['Wrong build#{}: build should be in project {}'.format(build.id, expected_build_type)]
    return []


def check_browser_build(build):
    return check_build(build, BRANDED_BUILDS_PROJECT)


def check_tests_build(build):
    return check_build(build, TESTS_BUILDS_PROJECT)


def get_platform(browser_build):
    build_type = browser_build.build_type.id
    for platform in ['Win', 'Mac', 'Linux']:
        if platform in build_type:
            return platform
    return None


def get_browser_version_number(browser_build):
    r = re.search(r'((\d+\.\d+)\.\d+)\.\d+/\d+', browser_build.number)
    return r.group(1)


def get_st_browser_version(browser_build):
    if browser_build.branch_name == 'master' or browser_build.branch_name == '<default>':
        return 'Dev'
    return get_browser_version_number(browser_build)


def get_framework_branch(browser_build):
    if browser_build.branch_name == 'master' or browser_build.branch_name == '<default>':
        return 'master'
    r = re.search(r'((\d+\.\d+)\.\d+)\.\d+/\d+', browser_build.number)
    return 'feature/{}'.format(r.group(2))


def split_list(original_list, max_len):
    groups_num = len(original_list) // max_len + (1 if len(original_list) % max_len else 0)
    return [original_list[i * max_len:(i * max_len) + max_len] for i in range(groups_num)]


def build_number_tuple(build):
    return tuple(int(part) for part in build.number.split('/')[0].split('.'))


def html_link(link):
    return '<a href="{link}">{link}</a>'.format(link=link)


def pretty_time_delta(ms):
    negative = ms < 0
    ms = abs(ms)
    seconds = ms / 1000
    days, seconds = divmod(seconds, 86400)
    hours, seconds = divmod(seconds, 3600)
    minutes, seconds = divmod(seconds, 60)
    if days > 0:
        res = '{}d{}h{}m{}s'.format(days, hours, minutes, seconds)
    elif hours > 0:
        res = '{}h{}m{}s'.format(hours, minutes, seconds)
    elif minutes > 0:
        res = '{}m{}s'.format(minutes, seconds)
    else:
        res = '{}s'.format(seconds)
    return '{}{}'.format('-' if negative else '', res)


def is_dev_sandbox():
    return Registry().common.installation == Installation.LOCAL


def html_cut(summary, text):
    return u'<details><summary>{}</summary>{}</details>'.format(summary, text)


def extract_zip_resource(current_task, resource_data_path):
    """
    Unpacks the contents of the resource archives into the current task directory
    :param current_task:
    :param resource_path:
    :return: unpacked resource data path
    """

    base_path = str(current_task.path("unpacked_resources"))
    result_path_name = os.path.join(base_path, str(uuid.uuid4().hex))
    if not os.path.exists(base_path):
        os.mkdir(base_path)
    os.mkdir(result_path_name)

    for dirpath, dnames, fnames in os.walk(os.path.join(resource_data_path)):
        for _f in fnames:
            if _f.endswith('.zip'):
                with zipfile.ZipFile(os.path.join(resource_data_path, _f),
                                     compression=zipfile.ZIP_DEFLATED) as myzipfile:
                    myzipfile.extractall(result_path_name)

    return result_path_name
