# -*- encoding: utf-8 -*-

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.projects.browser.booking import common
from sandbox.projects.browser.booking.estimator import Estimation
from sandbox.projects.browser.booking.estimator import Estimator
from sandbox.projects.browser.booking.task import BaseBrowserBookingTaskWithConfig


class BrowserEstimateBookingAssessorTesting(BaseBrowserBookingTaskWithConfig):
    """
    Estimates assessor testing bookings and warns responsible if it is needed.
    See https://wiki.yandex-team.ru/browser/dev/infra/docs/browserbookassessortesting/
    """

    def on_execute(self):
        super(BrowserEstimateBookingAssessorTesting, self).on_execute()

        with self.memoize_stage.start_estimations:
            estimator = Estimator(
                self.logger, self.cache, self.teamcity_client,
                self.now_msk)
            estimations = estimator.estimate(self.booking_config)
            for estimation in estimations:
                estimation.start_estimation_task(self)
            self.Context.estimations = [e.__dict__ for e in estimations]
            raise sdk2.WaitTask(
                [e.sandbox_task_id for e in estimations],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage.review_estimations:
            estimations = [Estimation(**e) for e in self.Context.estimations]
            for estimation in estimations:
                sandbox_task = sdk2.Task.find(id=estimation.sandbox_task_id).limit(1).first()
                if sandbox_task.status not in ctt.Status.Group.SUCCEED:
                    raise TaskFailure("Subtask #{} is failed with status {}".format(
                        sandbox_task.id, sandbox_task.status))

                estimation_amount = sandbox_task.Context.assessors_new_jobs_estimate / (60 * 60 * 1000)

                booking_info = self.cache.get_booking_info(estimation.booking_id)
                ratio = booking_info.amount / estimation_amount
                if ratio < 0.95 or ratio > 1.05:
                    release = self.cache.find_release(estimation.project_key, estimation.release_version)
                    event = self.cache.find_event(estimation.project_key, release, estimation.event_id)

                    message = (
                        'Проект {project}: Квота бронирования #{booking_id} ({booking_url}) для вехи "{event}" '
                        'релиза "{release}" назначеного на {booking_start} не соответствует оценке:\n'
                        '  - забронировано {booking_amount}\n'
                        '  - оценка {estimation_amount:.3f} (https://sandbox.yandex-team.ru/task/{sandbox_task_id})\n'
                    ).format(
                        project=estimation.project_key,
                        booking_id=booking_info.booking_id,
                        booking_url=common.BookingClient.get_booking_url(booking_info.booking_id),
                        event=event.title,
                        release=release.version,
                        booking_start=booking_info.start.astimezone(common.MSK_TIMEZONE),
                        booking_amount=booking_info.amount,
                        estimation_amount=estimation_amount,
                        sandbox_task_id=sandbox_task.id)

                    self.set_info(message)
                    self.notify_responsible(estimation.project_key, release, message)
