# coding=utf-8

import collections
import logging

from sandbox.projects.browser.booking.common import BookingClient
from sandbox.projects.browser.booking.common import BookingInfo
from sandbox.projects.browser.booking.common import BookingParams
from sandbox.projects.browser.booking.common import format_msk_datetime
from sandbox.projects.browser.booking.processor import Processor

_ACTION_BOOKING_CANCEL = 'CANCEL'
_ACTION_BOOKING_CREATE_LINK = 'CREATE-LINK'
_ACTION_EVENT_UPDATE_PARAMS = 'UPDATE-PARAMS'

_KEY_ACTION = 'action'
_KEY_BOOKINGS = 'bookings'
_KEY_BOOKING_START_MSK = 'booking-start-msk'
_KEY_BOOKING_TITLE = 'booking-title'
_KEY_DATE_MSK = 'date-msk'
_KEY_EVENT_BOOKING_KIND = 'event-booking-kind'
_KEY_EVENT_BOOKING_READONLY = 'event-booking-readonly'
_KEY_EVENT_DATE_MSK = 'event-date-msk'
_KEY_EVENT_ID = 'event-id'
_KEY_EVENT_TITLE = 'event-title'
_KEY_ID = 'id'
_KEY_KIND = 'kind'
_KEY_LOGIN = 'login'
_KEY_LINKS = 'links'
_KEY_PARAMS = 'params'
_KEY_PROJECT = 'project'
_KEY_READONLY = 'readonly'
_KEY_RELEASE = 'release'
_KEY_RELEASES = 'releases'
_KEY_STATUS = 'status'
_KEY_START_FROM_MSK = 'start-from-msk'
_KEY_START_MSK = 'start-msk'
_KEY_TITLE = 'title'
_KEY_URL = 'url'

_COMMAND_DUMP_SCHEMA = {
    'definitions': {
        'event': {
            'type': 'object',
            'properties': {
                _KEY_ID: {'type': 'string'},
                _KEY_TITLE: {'type': 'string'},
                _KEY_DATE_MSK: {'type': 'string'},
                _KEY_PARAMS: {
                    'type': 'object',
                    'properties': {
                        _KEY_KIND: {'type': ['null', 'string']},
                        _KEY_ID: {'type': ['null', 'integer']},
                        _KEY_URL: {'type': ['null', 'string']},
                        _KEY_READONLY: {'type': ['null', 'boolean']},
                    },
                },
                _KEY_BOOKING_TITLE: {'type': 'string'},
                _KEY_BOOKING_START_MSK: {'type': 'string'},
                _KEY_ACTION: {'enum': [_ACTION_EVENT_UPDATE_PARAMS]},
            },
            'required': [_KEY_ID],
        },
        'booking': {
            'type': 'object',
            'properties': {
                _KEY_ID: {'type': 'integer'},
                _KEY_STATUS: {'type': 'string'},
                _KEY_LOGIN: {'type': 'string'},
                _KEY_TITLE: {'type': 'string'},
                _KEY_START_MSK: {'type': 'string'},
                _KEY_START_FROM_MSK: {'type': 'string'},
                _KEY_LINKS: {
                    'type': 'array',
                    'items': {
                        'type': 'object',
                        'properties': {
                            _KEY_PROJECT: {'type': 'string'},
                            _KEY_RELEASE: {'type': 'string'},
                            _KEY_EVENT_ID: {'type': 'string'},
                            _KEY_EVENT_TITLE: {'type': 'string'},
                            _KEY_EVENT_DATE_MSK: {'type': 'string'},
                            _KEY_EVENT_BOOKING_KIND: {'type': 'string'},
                            _KEY_EVENT_BOOKING_READONLY: {'type': 'boolean'},
                        },
                    },
                },
                _KEY_ACTION: {'enum': [_ACTION_BOOKING_CANCEL, _ACTION_BOOKING_CREATE_LINK]},
            },
            'required': [_KEY_ID],
        },
    },

    'type': 'object',
    'properties': {
        _KEY_RELEASES: {
            'type': 'object',
            'patternProperties': {'^.*$': {
                'type': 'object',
                'patternProperties': {'^.*$': {
                    'type': 'array',
                    'items': {'$ref': '#/definitions/event'},
                }},
            }},
        },
        _KEY_BOOKINGS: {
            'type': 'array',
            'items': {'$ref': '#/definitions/booking'},
        },
    },
}


class ProjectReleaseEvent(object):
    def __init__(self, project_key, release, event):
        """
        :type project_key: str
        :type release: shuttle_client.model.Release
        :type event: shuttle_client.model.Event
        """
        self.project_key = project_key
        self.release = release
        self.event = event


class Dumper(object):
    def __init__(self, cache, now_msk, include_old):
        """
        :type cache: Cache
        :type now_msk: datetime.datetime
        :type include_old: bool
        """
        self.cache = cache
        self.now_msk = now_msk
        self.include_old = include_old

    def dump(self, username, project_key_list):
        """
        :type username: str
        :type project_key_list: list[str]
        """
        return {
            _KEY_RELEASES: {
                project_key: self.dump_project(project_key)
                for project_key in sorted(project_key_list)
            },
            _KEY_BOOKINGS: self.dump_bookings(username, project_key_list),
        }

    def dump_project(self, project_key):
        """
        :type project_key: str
        """
        releases = self.cache.get_releases(project_key)
        releases = sorted(releases, key=lambda r: r.version)

        dump = {}
        for release in releases:
            events = self.cache.get_events(project_key, release.id)
            if Processor.are_events_in_the_distant_past(self.now_msk, events) and not self.include_old:
                continue

            release_dump = self.dump_release(project_key, release)
            if release_dump:
                dump[release.version] = release_dump
        return dump

    def dump_release(self, project_key, release):
        """
        :type project_key: str
        :type release: shuttle_client.model.Release
        """
        events = self.cache.get_events(project_key, release.id)
        events = sorted(events, key=lambda e: e.date)

        dump = []
        for event in events:
            if not BookingParams.does_exist_in(event):
                continue

            booking_id = BookingParams.get_booking_id(event)
            params = {
                _KEY_KIND: BookingParams.get_booking_kind(event),
                _KEY_ID: booking_id,
                _KEY_URL: BookingParams.get_booking_url(event),
                _KEY_READONLY: BookingParams.get_booking_readonly(event),
            }
            params = {k: v for k, v in params.iteritems() if v is not None}

            event_dump = {
                _KEY_ID: event.id,
                _KEY_TITLE: event.title,
                _KEY_DATE_MSK: format_msk_datetime(event.date),
                _KEY_PARAMS: params,
            }

            booking_info = self.cache.get_booking_info(booking_id)
            if booking_info:
                event_dump[_KEY_BOOKING_TITLE] = booking_info.title
                event_dump[_KEY_BOOKING_START_MSK] = format_msk_datetime(booking_info.start)

            dump.append(event_dump)

        return dump

    def collect_bookings(self, project_key_list):
        """
        :type project_key_list: list[str]
        :rtype: dict[int, list[ProjectReleaseEvent]]
        """
        booking_links = collections.defaultdict(list)

        for project_key in project_key_list:
            for release in self.cache.get_releases(project_key):
                for event in self.cache.get_events(project_key, release.id):
                    if not BookingParams.does_exist_in(event):
                        continue
                    booking_id = BookingParams.get_booking_id(event)
                    if booking_id:
                        booking_links[booking_id].append(
                            ProjectReleaseEvent(project_key, release, event))

        return booking_links

    def dump_bookings(self, username, project_key_list):
        """
        :type username: str
        :type project_key_list: list[str]
        """
        booking_links = self.collect_bookings(project_key_list)
        booking_id_set = set(booking_links.keys())

        my_bookings = self.cache.get_bookings_by_username(username)
        booking_id_set.update(b.booking_id for b in my_bookings)

        dump = []
        for booking_id in sorted(booking_id_set):
            booking_info = self.cache.get_booking_info(booking_id)
            if not booking_info:
                continue

            booking_dump = {
                _KEY_ID: booking_info.booking_id,
                _KEY_STATUS: booking_info.status,
                _KEY_LOGIN: booking_info.login,
                _KEY_TITLE: booking_info.title,
                _KEY_START_MSK: format_msk_datetime(booking_info.start),
                _KEY_START_FROM_MSK: format_msk_datetime(booking_info.start_from),
            }

            link_dump_list = []
            for booking_link in booking_links.get(booking_id, []):
                link_dump = {
                    _KEY_PROJECT: booking_link.project_key,
                    _KEY_RELEASE: booking_link.release.version,
                    _KEY_EVENT_ID: booking_link.event.id,
                    _KEY_EVENT_TITLE: booking_link.event.title,
                    _KEY_EVENT_DATE_MSK: format_msk_datetime(booking_link.event.date),
                    _KEY_EVENT_BOOKING_KIND: BookingParams.get_booking_kind(booking_link.event),
                    _KEY_EVENT_BOOKING_READONLY: BookingParams.get_booking_readonly(booking_link.event),
                }
                link_dump = {k: v for k, v in link_dump.iteritems() if v is not None}
                link_dump_list.append(link_dump)

            if link_dump_list:
                booking_dump[_KEY_LINKS] = link_dump_list

            dump.append(booking_dump)

        return dump

    def restore_missed_bookings(self, dump):
        for booking_dump in dump[_KEY_BOOKINGS]:
            booking_id = booking_dump[_KEY_ID]
            status = booking_dump[_KEY_STATUS]
            title = booking_dump[_KEY_TITLE]
            links = booking_dump.get(_KEY_LINKS, [])
            if status == BookingInfo.STATUS_ACTIVE and not links:
                booking_dump[_KEY_ACTION] = _ACTION_BOOKING_CANCEL

                logging.info('Restore #%s: %s', booking_id, title)
                try:
                    project_key, release_version, event_title = title.split(',', 2)
                    project_key = project_key.strip().lower()
                    release_version = release_version.strip()
                    event_title = event_title.strip()
                    logging.info(
                        ' - Project: "%s", Release: "%s", Event: "%s".',
                        project_key, release_version, event_title)
                except ValueError as e:
                    logging.info(' - Cannot parse: %s', e.message)
                    continue

                if project_key not in dump[_KEY_RELEASES]:
                    logging.info(' - Project not found.')
                    continue

                if release_version not in dump[_KEY_RELEASES][project_key]:
                    logging.info(' - Release not found.')
                    continue

                event_dumps = dump[_KEY_RELEASES][project_key][release_version]
                event_dumps = [d for d in event_dumps if d[_KEY_TITLE] == event_title]
                if not event_dumps:
                    logging.info(' - Event not found.')
                    continue
                if len(event_dumps) > 1:
                    logging.info(' - Two or more events with same title: %s.', ', '.join(
                        [d[_KEY_ID] for d in event_dumps]))
                    continue
                event_dump = event_dumps[0]

                # Restore booking <-> event link.
                booking_dump[_KEY_LINKS] = [{
                    _KEY_PROJECT: project_key,
                    _KEY_RELEASE: release_version,
                    _KEY_EVENT_ID: event_dump[_KEY_ID],
                    _KEY_EVENT_TITLE: event_dump[_KEY_TITLE],
                    _KEY_EVENT_DATE_MSK: event_dump[_KEY_DATE_MSK],
                    _KEY_EVENT_BOOKING_KIND: event_dump[_KEY_PARAMS][_KEY_KIND],
                }]
                booking_dump[_KEY_ACTION] = _ACTION_BOOKING_CREATE_LINK


def execute_commands(cache, dump, dry_run, notifications):
    """
    :type cache: Cache
    :type dump: dict
    :type dry_run: bool
    :type notifications: list[str]
    """
    import jsonschema
    jsonschema.validate(dump, _COMMAND_DUMP_SCHEMA)

    execute_releases_commands(cache, dump, dry_run, notifications)
    execute_bookings_commands(cache, dump, dry_run, notifications)


def execute_releases_commands(cache, dump, dry_run, notifications):
    """
    :type cache: Cache
    :type dump: dict
    :type dry_run: bool
    :type notifications: list[str]
    """
    if _KEY_RELEASES not in dump:
        return
    for project_key, dump_project in dump[_KEY_RELEASES].iteritems():
        releases = cache.get_releases(project_key)
        for release_version, dump_release in dump_project.iteritems():
            release = next((r for r in releases if r.version == release_version), None)
            if not release:
                raise ValueError('Release "{}" not found in {}'.format(
                    release_version, [r.version for r in releases]))
            events = cache.get_events(project_key, release.id)

            for dump_event in dump_release:
                action = dump_event.get(_KEY_ACTION)
                if not action:
                    continue

                event_id = dump_event[_KEY_ID]
                event = next((e for e in events if e.id == event_id), None)
                if not event:
                    raise ValueError('Event #{} not found in {}'.format(
                        event_id, [e.id for e in events]))

                if action == _ACTION_EVENT_UPDATE_PARAMS:
                    action_params = {
                        _KEY_KIND: dump_event[_KEY_PARAMS].get(_KEY_KIND),
                        _KEY_ID: dump_event[_KEY_PARAMS].get(_KEY_ID),
                        _KEY_URL: dump_event[_KEY_PARAMS].get(_KEY_URL),
                    }
                    action_event_update_params(
                        cache, dry_run, project_key, release, event, action_params, notifications)


def action_event_update_params(cache, dry_run, project_key, release, event, action_params, notifications):
    """
    :type cache: Cache
    :type dry_run: bool
    :type project_key: str
    :type release: shuttle_client.model.Release
    :type event: shuttle_client.model.Event
    :type action_params: dict[str, any]
    :type notifications: list[str]
    """
    logging.info('ACTION: Update parameters of event #%s with: %s', event.id, action_params)

    new_event_params = dict(event.parameters)
    new_booking_params = new_event_params.get(BookingParams.PARAM_BOOKING, {})
    new_booking_params.update(action_params)
    new_booking_params = {k: v for k, v in new_booking_params.iteritems() if v is not None}
    new_event_params[BookingParams.PARAM_BOOKING] = new_booking_params
    if not dry_run:
        cache.shuttle.set_event_parameters(project_key, event.id, new_event_params)

    notifications.append((
        '{dry}Проект {project_key}: Обновлены параметры вехи "{event_title}" '
        'релиза "{release_version}" ({event_date}): {old_event_params} -> {new_event_params}.'
    ).format(
        dry=('DRY: ' if dry_run else ''),
        project_key=project_key,
        event_title=event.title,
        release_version=release.version,
        event_date=format_msk_datetime(event.date),
        old_event_params=event.parameters,
        new_event_params=new_event_params,
    ))


def execute_bookings_commands(cache, dump, dry_run, notifications):
    """
    :type cache: Cache
    :type dump: dict
    :type dry_run: bool
    :type notifications: list[str]
    """
    if _KEY_BOOKINGS not in dump:
        return
    for dump_booking in dump[_KEY_BOOKINGS]:
        action = dump_booking.get(_KEY_ACTION)
        if not action:
            continue

        booking_id = dump_booking[_KEY_ID]
        booking_info = cache.get_booking_info(booking_id)

        if action == _ACTION_BOOKING_CANCEL:
            action_cancel_booking(cache, dry_run, booking_id, booking_info, notifications)
            continue

        if action == _ACTION_BOOKING_CREATE_LINK:
            for link_dump in dump_booking[_KEY_LINKS]:
                project_key = link_dump[_KEY_PROJECT]
                release_version = link_dump[_KEY_RELEASE]
                event_id = link_dump[_KEY_EVENT_ID]
                event_booking_readonly = link_dump.get(_KEY_EVENT_BOOKING_READONLY)

                release = cache.find_release(project_key, release_version)
                event = cache.find_event(project_key, release, event_id)

                action_params = {BookingParams.PARAM_BOOKING_ID: booking_id}
                if event_booking_readonly is not None:
                    action_params[BookingParams.PARAM_BOOKING_READONLY] = (
                        BookingParams.VALUE_BOOKING_READONLY_YES if event_booking_readonly
                        else BookingParams.VALUE_BOOKING_READONLY_NO)
                action_event_update_params(cache, dry_run, project_key, release, event, action_params, notifications)
            continue

        raise ValueError('Unknown action: {}'.format(action))


def action_cancel_booking(cache, dry_run, booking_id, booking_info, notifications):
    logging.info('ACTION: Cancel booking #%s', booking_id)
    if not dry_run:
        cache.booking.cancel(booking_id)

    notifications.append((
        '{dry}Отменено бронирование #{booking_id} ({booking_url}) '
        'назначенное на {booking_start}.'
    ).format(
        dry=('DRY: ' if dry_run else ''),
        booking_id=booking_info.booking_id,
        booking_url=BookingClient.get_booking_url(booking_info.booking_id),
        booking_start=format_msk_datetime(booking_info.start),
    ))
