# coding=utf-8

import copy
import datetime
import json
import logging

import pytz
import requests

from sandbox.projects.common import decorators

MSK_TIMEZONE = pytz.timezone('Europe/Moscow')
DATETIME_FORMAT = '%Y-%m-%d %H:%M:%S'


def format_msk_datetime(date):
    """
    :type date: datetime.datetime
    :rtype: str
    """
    return date.astimezone(MSK_TIMEZONE).strftime(DATETIME_FORMAT)


class BookingParams(object):
    PARAM_BOOKING = 'booking'
    PARAM_BOOKING_ID = 'id'
    PARAM_BOOKING_KIND = 'kind'
    PARAM_BOOKING_URL = 'url'
    PARAM_BOOKING_READONLY = 'readonly'
    VALUE_BOOKING_READONLY_YES = 'yes'
    VALUE_BOOKING_READONLY_NO = 'no'
    _BOOKING_PARAMS_SCHEMA = {
        'type': 'object',
        'properties': {
            PARAM_BOOKING_KIND: {'type': 'string'},
            PARAM_BOOKING_ID: {'type': 'integer'},
            PARAM_BOOKING_URL: {'type': 'string'},
            PARAM_BOOKING_READONLY: {'enum': [
                VALUE_BOOKING_READONLY_YES,
                VALUE_BOOKING_READONLY_NO,
            ]},
        },
    }

    @classmethod
    def does_exist_in(cls, event):
        """
        :type event: shuttle_client.Event
        :rtype: bool
        """
        return cls.PARAM_BOOKING in event.parameters

    @classmethod
    def validate_booking_params(cls, event):
        """
        :type event: shuttle_client.Event
        :rtype: bool
        """
        import jsonschema
        jsonschema.validate(event.parameters[cls.PARAM_BOOKING], cls._BOOKING_PARAMS_SCHEMA)

    @classmethod
    def get_booking_kind(cls, event):
        """
        :type event: shuttle_client.Event
        :rtype: str | None
        """
        return event.parameters.get(cls.PARAM_BOOKING, {}).get(cls.PARAM_BOOKING_KIND)

    @classmethod
    def get_booking_id(cls, event):
        """
        :type event: shuttle_client.Event
        :rtype: int | None
        """
        return event.parameters.get(cls.PARAM_BOOKING, {}).get(cls.PARAM_BOOKING_ID)

    @classmethod
    def get_booking_url(cls, event):
        """
        :type event: shuttle_client.Event
        :rtype: str | None
        """
        return event.parameters.get(cls.PARAM_BOOKING, {}).get(cls.PARAM_BOOKING_URL)

    @classmethod
    def get_booking_readonly(cls, event):
        """
        :type event: shuttle_client.Event
        :rtype: bool | None
        """
        params = event.parameters.get(cls.PARAM_BOOKING, {})
        readonly_value = params.get(cls.PARAM_BOOKING_READONLY)
        return (None if readonly_value is None
                else readonly_value == cls.VALUE_BOOKING_READONLY_YES)

    @classmethod
    def add_booking(cls, event, booking_id, booking_readonly=None):
        """
        :type event: shuttle_client.Event
        :type booking_id: int
        :type booking_readonly: bool
        """
        params = event.parameters[cls.PARAM_BOOKING]
        params[cls.PARAM_BOOKING_ID] = booking_id
        params[cls.PARAM_BOOKING_URL] = BookingClient.get_booking_url(booking_id)
        if booking_readonly is not None:
            params[cls.PARAM_BOOKING_READONLY] = (
                cls.VALUE_BOOKING_READONLY_YES if booking_readonly
                else cls.VALUE_BOOKING_READONLY_NO)

    @classmethod
    def del_booking(cls, event):
        """
        :type event: shuttle_client.Event
        """
        params = event.parameters[cls.PARAM_BOOKING]
        params.pop(cls.PARAM_BOOKING_ID, None)
        params.pop(cls.PARAM_BOOKING_URL, None)
        params.pop(cls.PARAM_BOOKING_READONLY, None)


class BookingError(BaseException):
    CODE_FORBIDDEN = 'FORBIDDEN'
    CODE_NOT_FOUND = 'NOT_FOUND'
    CODE_QUOTA_LIMIT_EXCEEDED = 'QUOTA_LIMIT_EXCEEDED'
    CODE_RESOURCE_LIMIT_EXCEEDED = 'RESOURCE_LIMIT_EXCEEDED'

    MESSAGES = {
        CODE_QUOTA_LIMIT_EXCEEDED: 'Недостаточно квоты для запуска',
        CODE_RESOURCE_LIMIT_EXCEEDED: 'Недостаточно ресурсов Бронировщика для запуска',
    }

    def __init__(self, response_json):
        self.response_json = response_json

    @property
    def code(self):
        return self.response_json.get('code')

    @property
    def message(self):
        return self.response_json.get('message')

    def get_human_readable_description(self):
        if self.code in BookingError.MESSAGES:
            return '{} ({})'.format(BookingError.MESSAGES[self.code], self.message)
        else:
            return '[{}] ({})'.format(self.code, self.message)


class BookingInfo(object):
    STATUS_ACTIVE = 'ACTIVE'
    STATUS_CANCELLED = 'CANCELLED'
    STATUS_CLOSED = 'CLOSED'

    def __init__(self, booking_info_json):
        """
        :type booking_info_json: dict[str, any]
        """
        self._data = booking_info_json

    @property
    def booking_id(self):
        """
        :rtype: int
        """
        return self._data['bookingId']

    @property
    def title(self):
        """
        :rtype: str
        """
        return self._data['title']

    @property
    def status(self):
        """
        :rtype: str
        """
        return self._data['status']

    @property
    def login(self):
        """
        :rtype: str
        """
        return self._data['login']

    @property
    def start(self):
        """
        :rtype: datetime.datetime
        """
        start_timestamp = self._data['estimate']['startTs']
        return pytz.UTC.localize(datetime.datetime.utcfromtimestamp(start_timestamp / 1000.0))

    @property
    def start_from(self):
        """
        :rtype: datetime.datetime
        """
        start_from_timestamp = self._data['params']['startTsFrom']
        return pytz.UTC.localize(datetime.datetime.utcfromtimestamp(start_from_timestamp / 1000.0))

    @property
    def deadline(self):
        """
        :rtype: datetime.datetime
        """
        deadline_timestamp = self._data['estimate']['deadlineTs']
        return pytz.UTC.localize(datetime.datetime.utcfromtimestamp(deadline_timestamp / 1000.0))

    @property
    def amount(self):
        """
        :rtype: int
        """
        return self._data['params']['volumeDescription']['customVolume']['amount']


class BookingClient(object):
    @classmethod
    def get_booking_url(cls, booking_id):
        return 'https://booking.yandex-team.ru/#/bookings/{}'.format(booking_id)

    def __init__(self, booking):
        """
        :type booking: ya_booking_client.YaBookingClient
        """
        self.booking = booking
        self.logger = logging.getLogger('booking')

    def _make_request(self, *args, **kwargs):
        try:
            return self.booking.make_request(*args, **kwargs)
        except requests.HTTPError as e:
            self.logger.error('booking response: %s', e.response.text)

            try:
                response_json = e.response.json()
                raise BookingError(response_json)
            except ValueError:
                raise e

    def create(self, title, booking_config_params, booking_config_meta_data, start_timestamp):
        """
        :type title: str
        :type booking_config_params: dict[str, any]
        :type booking_config_meta_data: dict[str, any] | None
        :type start_timestamp: int
        :rtype: BookingInfo
        """
        booking_params = copy.deepcopy(booking_config_params)
        booking_params['startTsFrom'] = start_timestamp
        self.logger.info('booking params: %s', booking_params)
        self.logger.info('booking meta data: %s', booking_config_meta_data)
        estimate_response = self._make_request(
            method='POST', path='bookings/assessor-testing/default/estimate',
            data=json.dumps(booking_params))
        self.logger.info('estimate response: %s', estimate_response)
        request_data = {
            'title': title,
            'params': booking_params,
            'estimate': estimate_response,
        }
        if booking_config_meta_data:
            request_data['metaData'] = booking_config_meta_data
        self.logger.info('booking request data: %s', request_data)
        confirm_response = self._make_request(
            method='POST', path='bookings/assessor-testing/default',
            data=json.dumps(request_data))
        self.logger.info('confirm response: %s', confirm_response)
        return BookingInfo(confirm_response)

    def cancel(self, booking_id):
        """
        :type booking_id: int
        """
        cancel_response = self._make_request(
            method='PUT', path='bookings/{}/cancel'.format(booking_id))
        self.logger.info('cancel response: %s', cancel_response)

    def get_info(self, booking_id):
        """
        :type booking_id: int
        :rtype: BookingInfo
        """
        try:
            info_response = self._make_request(
                method='GET', path='bookings/{}'.format(booking_id))
            self.logger.info('get_info response: %s', info_response)
            return BookingInfo(info_response)
        except requests.HTTPError as e:
            self.logger.info('get_info response:\n%s', e.response.text)
            raise

    def filter(self, **filter_params):
        """
        :rtype: list[BookingInfo]
        """
        response_json = self._make_request(
            method='POST', path='bookings/assessor-testing/default/filter',
            data=json.dumps(filter_params))
        return [BookingInfo(booking_info_json) for booking_info_json in response_json]


class Cache(object):
    def __init__(self, shuttle_client, ya_booking_client):
        """
        :type shuttle_client: shuttle_client.ShuttleClient
        :type ya_booking_client: ya_booking_client.YaBookingClient
        """
        self.shuttle = shuttle_client
        self.booking = BookingClient(ya_booking_client)

    @decorators.memoize
    def get_booking_info(self, booking_id):
        """
        :type booking_id: int
        :rtype: sandbox.projects.browser.booking.processor.BookingInfo | None
        """
        try:
            return self.booking.get_info(booking_id)
        except BookingError as e:
            if e.code in [BookingError.CODE_FORBIDDEN, BookingError.CODE_NOT_FOUND]:
                logging.exception('Cannot get booking info.')
                return None
            else:
                raise

    @decorators.memoize
    def get_bookings_by_username(self, username):
        """
        :type username: str
        :rtype: list[sandbox.projects.browser.booking.processor.BookingInfo]
        """
        return self.booking.filter(status=BookingInfo.STATUS_ACTIVE, login=username)

    @decorators.memoize
    def get_releases(self, project_key):
        """
        :type project_key: str
        :rtype: list[shuttle_client.model.Release]
        """
        return self.shuttle.get_releases(project_key)

    @decorators.memoize
    def get_events(self, project_key, release_id):
        """
        :type project_key: str
        :type release_id: str
        :rtype: list[shuttle_client.model.Event]
        """
        return self.shuttle.get_events(project_key, release_id)

    def find_release(self, project_key, release_version):
        """
        :type project_key: str
        :type release_version: str
        :rtype: shuttle_client.model.Release
        """
        releases = self.get_releases(project_key)
        return next((r for r in releases if r.version == release_version), None)

    def find_event(self, project_key, release, event_id):
        """
        :type project_key: str
        :type release: shuttle_client.model.Release
        :type event_id: int
        :rtype: shuttle_client.model.Event
        """
        events = self.get_events(project_key, release.id)
        return next((e for e in events if e.id == event_id), None)
