# coding=utf-8

import collections
import copy
import math
import datetime
import json
import os
import re

import yaml


class ConfigLoader(object):
    _JSON_EXT = '.json'
    _YAML_EXT = '.yaml'
    _CONFIG_IGNORE_LIST = [
        '.revision',
    ]

    _MERGED_CONFIG_SCHEMA = {
        'type': 'object',
        'propertyNames': {
            'enum': [
                'desktop',
                'android',
                'ios',
                'pp-android',
                'pp-ios',
            ],
        },
        'patternProperties': {
            '^.*$': {
                'type': 'object',
                'properties': {
                    'default-config': {'$ref': '#/definitions/config'},
                    'booking-kinds': {
                        'type': 'object',
                        'patternProperties': {
                            '^.*$': {'$ref': '#/definitions/config'},
                        },
                    },
                },
                'required': ['booking-kinds'],
            },
        },
        'definitions': {
            'config': {
                'type': 'object',
                'additionalProperties': False,
                'properties': {
                    'create-days': {'type': 'integer'},
                    'veto-days': {'type': 'integer'},
                    'notification-hours-threshold': {'type': 'integer'},
                    'time-msk': {'type': 'string', 'pattern': '\\d\\d:\\d\\d'},
                    'estimation': {'$ref': '#/definitions/estimation'},
                    'params': {'$ref': '#/definitions/params'},
                    'meta-data': {'$ref': '#/definitions/meta-data'},
                },
                'required': ['create-days', 'veto-days', 'notification-hours-threshold', 'time-msk', 'params'],
            },
            'estimation': {
                'type': 'object',
                'additionalProperties': False,
                'properties': {
                    'before-hours': {'type': 'integer'},
                    'method': {
                        'type': 'string',
                        'enum': ['alpha', 'beta', 'rc'],
                    },
                },
                'required': ['before-hours', 'method'],
            },
            'params': {
                'type': 'object',
                'additionalProperties': False,
                'properties': {
                    'availableDays': {'type': 'integer'},
                    'quotaSource': {'type': 'string'},
                    'speedMode': {'enum': ['SLOW', 'NORMAL', 'URGENT']},
                    'volumeDescription': {
                        'type': 'object',
                        'additionalProperties': False,
                        'properties': {
                            'volumeSources': {'type': 'array', 'maxItems': 0},
                            'customVolume': {
                                'type': 'object',
                                'additionalProperties': False,
                                'properties': {
                                    'amount': {'type': 'string', 'pattern': '^\\d+$'},
                                    'availableForFarm': {'type': 'boolean'},
                                    'environmentDistribution': {
                                        'type': 'object',
                                        'patternProperties': {
                                            '^\\d+( \\|\\| \\d+)*$': {
                                                'type': 'number',
                                                'minimum': 0,
                                                'maximum': 1,
                                            },
                                            '^\\d+( && \\d+)*$': {
                                                'type': 'number',
                                                'minimum': 0,
                                                'maximum': 1,
                                            },
                                        },
                                        'additionalProperties': False,
                                    },
                                },
                                'required': ['amount', 'environmentDistribution'],
                            },
                        },
                        'required': ['volumeSources', 'customVolume'],
                    },
                },
                'required': ['quotaSource', 'speedMode', 'volumeDescription'],
            },
            'meta-data': {
                'type': 'object',
                'additionalProperties': False,
                'properties': {
                    'subscribers': {'type': 'array', 'items': {'type': 'string'}},
                    'customData': {'type': 'object'},
                },
                'required': ['subscribers', 'customData'],
            },
        },
    }

    @classmethod
    def available_configs(cls, config_dir=None):
        """
        :type config_dir: str | None
        :return: pairs of strings: config name and config path.
        :rtype: dict[str, str]
        """
        if not config_dir:
            config_dir = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'config')
        try:
            config_names = [n for n in os.listdir(config_dir) if n not in cls._CONFIG_IGNORE_LIST]
        except OSError:
            return {}

        configs = {}
        for config_name in config_names:
            if config_name.endswith(cls._JSON_EXT):
                choice_name = config_name[:-len(cls._JSON_EXT)]
            elif config_name.endswith(cls._YAML_EXT):
                choice_name = config_name[:-len(cls._YAML_EXT)]
            else:
                raise ValueError('Wrong config extension: {}'.format(config_name))
            if choice_name in configs:
                raise ValueError('Duplicated name of config: {}'.format(config_name))
            configs[choice_name] = os.path.join(config_dir, config_name)

        return configs

    @classmethod
    def _merge_dicts(cls, dct, merge_dct):
        """
        :type dct: dict[str, any]
        :type merge_dct: dict[str, any] or NoneType
        """
        for k, v in merge_dct.iteritems():
            if isinstance(dct.get(k), dict) and isinstance(merge_dct[k], collections.Mapping):
                cls._merge_dicts(dct[k], merge_dct[k])
            else:
                dct[k] = v

    @classmethod
    def _get_n(cls, dct, *keys):
        assert len(keys) >= 1
        if keys[0] not in dct:
            return None

        if len(keys) == 1:
            return dct[keys[0]]
        else:
            return cls._get_n(dct[keys[0]], *keys[1:])

    @classmethod
    def _set_n(cls, dct, value, *keys):
        assert len(keys) >= 1
        if len(keys) == 1:
            dct[keys[0]] = value
        else:
            if keys[0] not in dct:
                dct[keys[0]] = {}
            cls._set_n(dct[keys[0]], value, *keys[1:])

    @classmethod
    def _merge_config(cls, basic, update):
        """
        :type basic: dict[str, any] or NoneType
        :type update: dict[str, any] or NoneType
        :rtype: dict[str, any]
        """
        if not basic:
            return copy.deepcopy(update)

        merged = copy.deepcopy(basic)
        if update:
            cls._merge_dicts(merged, update)

        # Something must be overwritten.
        distribution = cls._get_n(update, 'params', 'volumeDescription', 'customVolume', 'environmentDistribution')
        if distribution:
            cls._set_n(merged, distribution, 'params', 'volumeDescription', 'customVolume', 'environmentDistribution')

        return merged

    @classmethod
    def _validate_merged_config(cls, config_json):
        """
        :type config_json: dict[str, any]
        """
        import jsonschema
        jsonschema.validate(config_json, cls._MERGED_CONFIG_SCHEMA)

        for project_key, project_config in config_json.iteritems():
            for booking_kind, booking_config in project_config['booking-kinds'].iteritems():
                params = booking_config['params']
                custom_volume = params['volumeDescription']['customVolume']
                distribution = custom_volume['environmentDistribution']
                distribution_sum = sum([
                    value for name, value in distribution.iteritems()])
                if math.fabs(1.0 - distribution_sum) > 1e-8:
                    raise ValueError(
                        'Суммарная нагрузка для "{}"->"{}" не равна 1.0 ({})'.format(
                            project_key, booking_kind, distribution_sum))

    @classmethod
    def _load_config_json(cls, config_path):
        """
        :type config_path: str
        :rtype: dict[str, any]
        """
        with open(config_path, 'r') as f:
            config_name = os.path.basename(config_path)
            if config_name.endswith(cls._JSON_EXT):
                # Remove comments.
                json_text = re.sub(r'(?m)^\s*//.*$', '', f.read())
                config = json.loads(json_text)
            elif config_name.endswith(cls._YAML_EXT):
                config = yaml.safe_load(f.read())
            else:
                raise ValueError('Wrong config extension: {}'.format(config_name))

            # Merge booking configs with defaults.
            for project_key, project_config in config.iteritems():
                default_booking_config = project_config.pop('default-config', None)
                for booking_kind, booking_config in project_config['booking-kinds'].iteritems():
                    merged_booking_config = cls._merge_config(default_booking_config, booking_config)
                    project_config['booking-kinds'][booking_kind] = merged_booking_config

            return config

    @classmethod
    def load_config(cls, config_path):
        """
        :type config_path: str
        :rtype: (dict[str, any], dict[str, dict[str, BookingConfig]])
        """
        merged_config_json = cls._load_config_json(config_path)
        cls._validate_merged_config(merged_config_json)
        return merged_config_json, {
            project_key: {
                booking_kind: BookingConfig(project_key, booking_kind, config_json)
                for booking_kind, config_json in project_config['booking-kinds'].iteritems()
            }
            for project_key, project_config in merged_config_json.iteritems()
        }


class EstimationConfig(object):
    METHOD_ALPHA = 'alpha'
    METHOD_BETA = 'beta'
    METHOD_RC = 'rc'

    def __init__(self, estimation_config_json):
        self.before_hours = estimation_config_json['before-hours']
        """ :type: int """
        self.method = estimation_config_json['method']
        """ :type: str """


class BookingConfig(object):
    def __init__(self, project_key, booking_kind, config_json):
        """
        :type project_key: str
        :type booking_kind: str
        :type config_json: dict[str, any]
        """
        self.project_key = project_key
        self.booking_kind = booking_kind
        self.create_days = config_json['create-days']
        """ :type: int """
        self.veto_days = config_json['veto-days']
        """ :type: int """

        time_matcher = re.match(r'^(\d\d):(\d\d)$', config_json['time-msk'])
        self.time_msk_hour = int(time_matcher.group(1))
        self.time_msk_minute = int(time_matcher.group(2))

        self.params = config_json['params']
        """ :type: dict[str, any] """

        self.meta_data = config_json.get('meta-data')
        """ :type: dict[str, any] | None """

        estimation_config_json = config_json.get('estimation')
        self.estimation = EstimationConfig(estimation_config_json) if estimation_config_json else None

        self.notification_threshold = datetime.timedelta(hours=config_json['notification-hours-threshold'])
