# coding=utf-8

import datetime

import pytz

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.browser.booking.config import EstimationConfig
from sandbox.projects.browser.booking.processor import BookingParams
from sandbox.projects.browser.booking.processor import Processor

BRANDED_BUILD_TYPE_ID = 'BrowserBrandedDistributives_BrandedBetaWin'
EVENT_TYPE_BUILD_BETA = 'milestone:build:beta'
EVENT_TYPE_BUILD_RC = 'milestone:build:rc'
ARTIFACT_TYPE_DEPLOY_ALPHA = 'deploy_alpha_win_x32'
ARTIFACT_TYPE_DEPLOY_BETA = 'deploy_beta_win_x32'
BRANCH_MASTER = 'master'


class Estimation(object):
    @classmethod
    def create(cls, project_key, release, event, booking_info, sandbox_task_params):
        """
        :type project_key: str
        :type release: shuttle_client.Release
        :type event: shuttle_client.Event
        :type booking_info: sandbox.projects.browser.booking.common.BookingInfo
        :type sandbox_task_params: dict[str, any]
        """
        return Estimation(
            project_key=project_key,
            release_version=release.version,
            event_id=event.id,
            booking_id=booking_info.booking_id,
            sandbox_task_params=sandbox_task_params,
            sandbox_task_id=None)

    def __init__(self, project_key, release_version, event_id, booking_id,
                 sandbox_task_params, sandbox_task_id):
        self.project_key = project_key
        self.release_version = release_version
        self.event_id = event_id
        self.booking_id = booking_id
        self.sandbox_task_params = sandbox_task_params
        self.sandbox_task_id = sandbox_task_id

    def start_estimation_task(self, parent_task):
        """
        :type parent_task: sdk2.Task
        :rtype: sdk2.Task
        """
        task = sdk2.Task['BROWSER_CALCULATE_FUNCTIONALITIES_PROFIT'](
            parent=parent_task,
            owner=parent_task.owner,
            priority=parent_task.Parameters.priority,
            **self.sandbox_task_params)
        task.enqueue()
        self.sandbox_task_id = task.id
        return task


class Estimator(object):
    @classmethod
    def get_regression_type(cls, method, last_number):
        if method == EstimationConfig.METHOD_ALPHA:
            name = 'Alpha'
        elif method == EstimationConfig.METHOD_BETA:
            name = 'Beta'
        elif method == EstimationConfig.METHOD_RC:
            name = 'RC'
        else:
            raise ValueError('Unknown method: {}'.format(method))
        return 'Win_{name}{number}_Diff.yaml'.format(
            name=name, number=last_number)

    @classmethod
    def get_scope_filter(cls, method):
        if method == EstimationConfig.METHOD_ALPHA:
            return 'Queue: "Наш Браузер"'
        elif method in [EstimationConfig.METHOD_BETA, EstimationConfig.METHOD_RC]:
            return ''
        else:
            raise ValueError('Unknown method: {}'.format(method))

    @classmethod
    def split_release_version(cls, release_version):
        """
        :type release_version: str
        :rtype: str, int
        """
        version_head, last_number = release_version.rsplit('.', 1)
        last_number = int(last_number)
        return version_head, last_number

    @classmethod
    def get_branch_name(cls, release_version):
        """
        :type release_version: str
        :rtype: str
        """
        version_head, _ = cls.split_release_version(release_version)
        return 'master-{}/rc'.format(release_version)

    @classmethod
    def get_estimation_task_params(cls, project_key, release, event,
                                   estimation_method, previous_build_id, browser_build_id):
        """
        :type project_key: str
        :type release: shuttle_client.Release
        :type event: shuttle_client.Event
        :type estimation_method: str
        :type previous_build_id: int
        :type browser_build_id: int
        :rtype: dict[str, any]
        """
        _, last_number = cls.split_release_version(release.version)
        return dict(
            description=(
                '- project: "{}"\n'
                '- release: "{}"\n'
                '- event  : "{}"\n'
            ).format(
                project_key,
                release.version,
                event.title),
            old_build_id=previous_build_id,
            build_id=browser_build_id,
            regression_type=cls.get_regression_type(estimation_method, last_number),
            scope_filter=cls.get_scope_filter(estimation_method),
        )

    def __init__(self, logger, cache, teamcity, now_msk):
        """
        :type logger: logging.Logger
        :type cache: sandbox.projects.browser.booking.common.Cache
        :type teamcity: teamcity_client.TeamcityClient
        :type now_msk: datetime.datetime
        """
        self.logger = logger
        self.cache = cache
        self.teamcity = teamcity
        self.now_msk = now_msk

    def estimate(self, booking_config):
        """
        :type booking_config: dict[str, dict[str, config.BookingConfig]]
        :rtype: list[Estimation]
        """
        estimation_tasks = []
        for project_key in sorted(booking_config.keys()):
            booking_config_by_kind = booking_config[project_key]
            self.logger.info('Project "%s"', project_key)

            releases = {r.version: r for r in self.cache.get_releases(project_key)}
            for version in sorted(releases.keys()):
                estimation_tasks.extend(self.process_release(
                    project_key, releases, releases[version], booking_config_by_kind))
        return estimation_tasks

    def process_release(self, project_key, releases, release, booking_config_by_kind):
        """
        :type project_key: str
        :type releases: dict[str, shuttle_client.Release]
        :type release: shuttle_client.Release
        :type booking_config_by_kind: dict[str, sandbox.projects.browser.booking.config.BookingConfig]
        :rtype: list[Estimation]
        """
        self.logger.info('Project "%s": release #%s "%s"', project_key, release.id, release.version)
        self.logger.info('- responsible: "%s"', release.responsible)
        events = self.cache.get_events(project_key, release.id)
        events.sort(key=lambda e: e.date)
        if Processor.are_events_in_the_distant_past(self.now_msk, events):
            self.logger.info('- all events are in the past - skip release')
            return []

        estimations = []
        for event in events:
            if not BookingParams.does_exist_in(event):
                continue

            self.logger.info('Event #%s "%s" at "%s"', event.id, event.title, event.date)
            self.logger.info('- params: %s', event.parameters)
            BookingParams.validate_booking_params(event)

            booking_info = Processor.resolve_booking_info(
                self.logger, self.cache.booking, project_key, release, event)
            if not booking_info:
                continue

            booking_kind = BookingParams.get_booking_kind(event)
            if not booking_kind:
                continue

            booking_config = booking_config_by_kind.get(booking_kind)
            if not booking_config:
                self.logger.warning('Unknown booking kind "%s"', booking_kind)
                continue

            if not booking_config.estimation:
                self.logger.info('- no estimation config.')
                continue

            estimation = self.get_estimation(
                project_key, releases, release, events, event,
                booking_info, booking_config)
            if estimation:
                estimations.append(estimation)

        return estimations

    def get_neighbors(self, releases, version):
        """
        :type releases: dict[str, shuttle_client.Release]
        :type version: str
        :rtype: (shuttle_client.Release, shuttle_client.Release)
        """
        try:
            release_versions = sorted(releases.keys(), key=lambda v: [int(p) for p in v.split('.')])
            version_index = release_versions.index(version)
            if version_index > 0:
                prev_release = releases[release_versions[version_index - 1]]
            else:
                prev_release = None
            if version_index < len(release_versions) - 1:
                next_release = releases[release_versions[version_index + 1]]
            else:
                next_release = None
        except ValueError:
            prev_release = next_release = None

        self.logger.info('- prev release: {}'.format(
            prev_release.version if prev_release else None))
        self.logger.info('- next release: {}'.format(
            next_release.version if next_release else None))

        return prev_release, next_release

    def get_tc_build_id(self, branch_name, index, release_version=None):
        """
        :type branch_name: str
        :type index: int
        :type release_version: str | None
        :rtype: int | None
        """
        builds = list(self.teamcity.Builds(
            buildType=BRANDED_BUILD_TYPE_ID,
            branch=branch_name, state='finished', status='SUCCESS',
        ).paginate())
        if release_version is not None:
            builds = [build for build in builds
                      if build.number.startswith(release_version)]
        try:
            return builds[index].id
        except IndexError:
            return None

    def get_branded_build(self, deploy_build_id):
        """
        :type deploy_build_id: int | None
        :rtype: teamcity_client.client.build.Build
        """
        if deploy_build_id is None:
            return None
        deploy_build = self.teamcity.Builds()[deploy_build_id]
        for build in deploy_build.artifact_dependencies:
            if build.build_type.id == BRANDED_BUILD_TYPE_ID:
                return build.id
        raise ValueError('Cannot find branded build for #{}'.format(deploy_build_id))

    def get_deploy_build_id(self, project_key, release, slug):
        """
        :type project_key: str
        :type release: shuttle_client.Release
        :type slug: str
        :rtype: int | None
        """
        # TODO[BYIN-11458] move getting artifacts to the shuttle_client module.
        artifacts_json = self.cache.shuttle.connection.get(
            '/rest/projects/{}/releases/{}/artifacts/'.format(
                project_key, release.id))
        for artifact_json in artifacts_json['items']:
            artifact_type = artifact_json['type']
            if artifact_type != 'TEAMCITY_BUILD':
                continue
            artifact_is_active = artifact_json['is_active']
            if not artifact_is_active:
                continue
            artifact_slug = artifact_json['slug']
            if artifact_slug == slug:
                return artifact_json['data']['build_id']
        return None

    # TODO[BYIN-11855] fix this method.
    @classmethod
    def find_estimation_event(cls, estimation_method, events):
        if estimation_method == EstimationConfig.METHOD_ALPHA:
            event_type = EVENT_TYPE_BUILD_BETA
            name_parts = [u'Альфы', u'Alpha']
        elif estimation_method == EstimationConfig.METHOD_BETA:
            event_type = EVENT_TYPE_BUILD_BETA
            name_parts = [u'Беты', u'Beta']
        elif estimation_method == EstimationConfig.METHOD_RC:
            event_type = EVENT_TYPE_BUILD_RC
            name_parts = [u'WIN']
        else:
            raise ValueError('Unknown estimation method: {}'.format(estimation_method))

        filtered_events = [e for e in events if e.type == event_type and
                           any(p in e.title for p in name_parts)]
        if not filtered_events:
            raise ValueError('No events found')
        if len(filtered_events) > 1:
            raise ValueError('Several events found: {}'.format(', '.join(
                '"{}"'.format(e.title) for e in filtered_events)))
        return filtered_events[0]

    def get_estimation(self, project_key, releases, release, events, event,
                       booking_info, booking_config):
        """
        :type project_key: str
        :type releases: dict[str, shuttle_client.Release]
        :type release: shuttle_client.Release
        :type events: list[shuttle_client.Event]
        :type event: shuttle_client.Event
        :type booking_info: sandbox.projects.browser.booking.common.BookingInfo
        :type booking_config: sandbox.projects.browser.booking.config.BookingConfig
        :rtype: Estimation
        """
        self.logger.info('- estimation:')
        self.logger.info('  - method: %s', booking_config.estimation.method)
        self.logger.info('  - before hours: %s', booking_config.estimation.before_hours)

        estimation_event = self.find_estimation_event(booking_config.estimation.method, events)
        if not estimation_event:
            raise TaskError(
                'Проект "{}", релиз "{}", событие "{}":\n'
                'Не удалось найти событие, запускающее оценку по типу: "{}"'
                ' - отсутствует в конфигурации.'.format(
                    project_key, release.version, event.title,
                    booking_config.estimation.method))
        # Do nothing if time hasn't come.
        now_datetime = self.now_msk.astimezone(pytz.UTC)
        before_hours_timedelta = datetime.timedelta(
            hours=booking_config.estimation.before_hours)
        action_datetime = estimation_event.date - before_hours_timedelta
        self.logger.info('  - event id: %s', estimation_event.id)
        self.logger.info('  - event title: %s', estimation_event.title)
        self.logger.info('  - event datetime: %s', estimation_event.date)
        self.logger.info('  - action datetime: %s', action_datetime)
        self.logger.info('  - now datetime: %s', now_datetime)
        if now_datetime < action_datetime or now_datetime > estimation_event.date:
            return None

        prev_release, next_release = self.get_neighbors(releases, release.version)
        release_branch_name = self.get_branch_name(release.version)

        if booking_config.estimation.method == EstimationConfig.METHOD_ALPHA:
            last_version_number = int(release.version.rsplit('.', 1)[-1])
            if last_version_number == 0:
                # Previous build id - первая branded сборка в ветке master с нашей версией.
                previous_build_id = self.get_tc_build_id(BRANCH_MASTER, -1, release.version)
                # Browser build id - самая свежая сборка в ветке master с нашей версией.
                browser_build_id = self.get_tc_build_id(BRANCH_MASTER, 0, release.version)
            else:
                # Previous build id - c альфой предыдущей версии
                previous_build_id = self.get_branded_build(
                    self.get_deploy_build_id(project_key, prev_release, ARTIFACT_TYPE_DEPLOY_ALPHA))
                # Browser build id - самая свежая сборка в ветке master с нашей версией.
                browser_build_id = self.get_tc_build_id(BRANCH_MASTER, 0, release.version)
        elif booking_config.estimation.method == EstimationConfig.METHOD_BETA:
            # Previous build id - соответствующая альфа этой же цифровой версии
            previous_build_id = self.get_branded_build(
                self.get_deploy_build_id(project_key, release, ARTIFACT_TYPE_DEPLOY_ALPHA))
            # Browser build id - самая свежая сборка в ветке master с нашей версией.
            browser_build_id = self.get_tc_build_id(BRANCH_MASTER, 0, release.version)
        elif booking_config.estimation.method == EstimationConfig.METHOD_RC:
            # Previous build id - соответствующая бета этой же цифровой версии
            previous_build_id = self.get_branded_build(
                self.get_deploy_build_id(project_key, release, ARTIFACT_TYPE_DEPLOY_BETA))
            # Browser build id - самая свежая ночная сборка из канала branded.beta соответствующей версии
            browser_build_id = self.get_tc_build_id(release_branch_name, 0)
        else:
            raise TaskError(
                'Проект "{}", релиз "{}", событие "{}":\n'
                'Неизвестый метод оценки: "{}".'.format(
                    project_key, release.version, event.title,
                    booking_config.estimation.method))

        self.logger.info('- previous build id: #%s', previous_build_id)
        self.logger.info('-  browser build id: #%s', browser_build_id)
        if previous_build_id is None or browser_build_id is None:
            raise TaskError(
                'Проект "{}", релиз "{}", событие "{}":\n'
                'Не удалось получить сборки для оценки нагрузки ('
                'previous_build_id={}, browser_build_id={})".'.format(
                    project_key, release.version, event.title,
                    previous_build_id, browser_build_id))

        return Estimation.create(
            project_key, release, event, booking_info,
            self.get_estimation_task_params(
                project_key, release, event,
                booking_config.estimation.method,
                previous_build_id, browser_build_id))
