# coding=utf-8

import abc
import calendar
import datetime
import sys
import traceback

import pytz
import requests

from sandbox.common.errors import TaskError
from sandbox.projects.browser.booking.common import BookingClient
from sandbox.projects.browser.booking.common import BookingError
from sandbox.projects.browser.booking.common import BookingInfo
from sandbox.projects.browser.booking.common import BookingParams
from sandbox.projects.browser.booking.common import MSK_TIMEZONE


class Processor(object):
    __metaclass__ = abc.ABCMeta

    def __init__(self, shuttle, ya_booking, now_msk, dry_run):
        """
        :type shuttle: shuttle_client.ShuttleClient
        :type ya_booking: ya_booking_client.YaBookingClient
        :type now_msk: datetime.datetime
        :type dry_run: bool
        """
        self.shuttle = shuttle
        self.booking = BookingClient(ya_booking)
        self.now_msk = now_msk
        self.dry_run = dry_run
        self.errors = []

    def _clear_errors(self):
        self.errors = []

    def _report_error(self, exc_info):
        self.errors.append(exc_info)
        error_traceback = traceback.format_exception(*exc_info)
        self.set_info(''.join(error_traceback))

    def _raise_errors_if_necessary(self):
        if self.errors:
            raise TaskError('Бронировщик сломался при обработке событий - см. описание и логи задачи')

    @abc.abstractproperty
    def logger(self):
        """
        :rtype: logging.Logger
        """

    @abc.abstractmethod
    def set_info(self, message):
        """
        :type message: str
        """

    @abc.abstractmethod
    def notify_responsible(self, project_key, release, message):
        """
        :type project_key: str
        :type release: shuttle_client.Release
        :type message: str
        """

    def cancel_booking(self, project_key, event, booking_info):
        """
        :type project_key: str
        :type event: shuttle_client.Event
        :type booking_info: BookingInfo
        :rtype: BookingInfo
        """
        if self.dry_run:
            return None

        self.booking.cancel(booking_info.booking_id)

        try:
            BookingParams.del_booking(event)
            self.shuttle.set_event_parameters(project_key, event.id, event.parameters)
        except requests.HTTPError:
            self.logger.exception('Booking is canceled but not synced with shuttle.')
            self.set_info('Бронирование #{} было отменено, но не удалено из '
                          'параметров вехи.'.format(booking_info.booking_id))
            raise TaskError('Ошибка при отмене бронирования')

        return None

    def create_booking(self, project_key, release, event, booking_datetime,
                       booking_params, booking_meta_data):
        """
        :type project_key: str
        :type release: shuttle_client.Release
        :type event: shuttle_client.Event
        :type booking_datetime: datetime.datetime
        :type booking_params: dict[str, any]
        :type booking_meta_data: dict[str, any] | None
        :rtype: BookingInfo
        """
        booking_title = '{project}, {release}, {event}'.format(
            project=project_key.capitalize(), release=release.version, event=event.title)
        booking_timestamp = calendar.timegm(booking_datetime.utctimetuple()) * 1000

        if self.dry_run:
            deadline_timestamp = (booking_timestamp +
                                  datetime.timedelta(days=4).total_seconds() * 1000)
            return BookingInfo({
                'bookingId': 'FAKE',
                'title': booking_title,
                'status': BookingInfo.STATUS_ACTIVE,
                'login': '???',
                'params': booking_params,
                'estimate': {
                    'startTs': booking_timestamp,
                    'startTsFrom': booking_timestamp,
                    'deadlineTs': deadline_timestamp,
                },
            })

        booking_info = self.booking.create(
            booking_title, booking_params, booking_meta_data, booking_timestamp)

        try:
            BookingParams.add_booking(event, booking_info.booking_id)
            self.shuttle.set_event_parameters(project_key, event.id, event.parameters)
        except requests.HTTPError:
            self.logger.exception('Booking is created but not synced with shuttle.')
            self.set_info('Бронирование #{} было создано, но не сохранено в '
                          'параметры вехи.'.format(booking_info.booking_id))
            raise TaskError('Ошибка при создании бронирования')

        return booking_info

    def calculate_booking_datetime(self, event_date, booking_config):
        """
        :type event_date: datetime.datetime
        :type booking_config: config.BookingConfig
        :rtype: datetime.datetime
        """
        event_msk_datetime = event_date.astimezone(MSK_TIMEZONE)
        booking_msk_datetime = event_msk_datetime.replace(
            hour=booking_config.time_msk_hour,
            minute=booking_config.time_msk_minute)
        return booking_msk_datetime.astimezone(pytz.UTC)

    def calculate_action_datetime(self, event_date, delta_days):
        """
        :type event_date: datetime.datetime
        :type delta_days: int
        :rtype: datetime.datetime
        """
        event_msk_datetime = event_date.astimezone(MSK_TIMEZONE)
        action_msk_datetime = event_msk_datetime - datetime.timedelta(days=delta_days)
        # Round to days to compare.
        action_msk_datetime = action_msk_datetime.replace(hour=0, minute=0, second=0, microsecond=0)
        return action_msk_datetime.astimezone(pytz.UTC)

    @classmethod
    def resolve_booking_info(cls, logger, booking, project_key, release, event):
        """
        :type logger: logging.Logger
        :type booking: BookingClient
        :type project_key: str
        :type release: shuttle_client.Release
        :type event: shuttle_client.Event
        :rtype: BookingInfo | None
        """
        booking_id = BookingParams.get_booking_id(event)
        if not booking_id:
            logger.info('- booking_info: None')
            return None
        booking_info = booking.get_info(booking_id)
        logger.info('- booking_info:')
        logger.info('  - booking_id = %s', booking_info.booking_id)
        logger.info('  - start = %s', booking_info.start)
        logger.info('  - start_from = %s', booking_info.start_from)

        if booking_info.status == BookingInfo.STATUS_CANCELLED:
            logger.info('  - booking is canceled')
            return None
        if booking_info.status not in [BookingInfo.STATUS_ACTIVE, BookingInfo.STATUS_CLOSED]:
            raise TaskError(
                'Проект {project}: Неожиданное изменение статуса бронирования #{booking}'
                ' вехи "{event}" релиза "{release}" - сейчас "{status}"'.format(
                    project=project_key, release=release.version, event=event.title,
                    booking=booking_info.booking_id, status=booking_info.status))
        return booking_info

    def process_event(self, project_key, release, event,
                      booking_config, booking_info,
                      notifications):
        """
        :type project_key: str
        :type release: shuttle_client.Release
        :type event: shuttle_client.Event
        :type booking_config: config.BookingConfig
        :type booking_info: BookingInfo | None
        :type notifications: list[str]
        :rtype: BookingInfo | None
        """
        if BookingParams.get_booking_readonly(event):
            self.logger.info('Skip: Booking is readonly.')
            return booking_info

        booking_datetime = self.calculate_booking_datetime(event.date, booking_config)
        create_datetime = self.calculate_action_datetime(event.date, booking_config.create_days)
        now_datetime = self.now_msk.astimezone(pytz.UTC)
        veto_datetime = now_datetime + datetime.timedelta(days=booking_config.veto_days)
        self.logger.info('- booking datetime %s', booking_datetime)
        self.logger.info('- create datetime %s', create_datetime)
        self.logger.info('- now datetime %s', now_datetime)
        self.logger.info('- veto datetime %s', veto_datetime)

        if booking_datetime <= now_datetime and (not booking_info or booking_info.start <= now_datetime):
            self.logger.info('Skip: All in the past.')
            # Event and booking are in the past - skip event.
            return booking_info

        if booking_info and booking_datetime == booking_info.start_from:
            self.logger.info('Skip: Booking is already created.')
            if booking_info.start != booking_info.start_from:
                message = (
                    'Проект {project}: Бронирование #{booking_id} ({booking_url}) для вехи "{event}" '
                    'релиза "{release}" назначено на {booking_start} (а не на {booking_start_from}).\n'
                    'Дата выкатки: {event_date}\n'
                    'Окончание дедлайна: {deadline_date}'
                ).format(
                    project=project_key,
                    booking_id=booking_info.booking_id,
                    booking_url=BookingClient.get_booking_url(booking_info.booking_id),
                    event=event.title,
                    release=release.version,
                    booking_start=booking_info.start.astimezone(MSK_TIMEZONE),
                    booking_start_from=booking_info.start_from.astimezone(MSK_TIMEZONE),
                    event_date=event.date.astimezone(MSK_TIMEZONE),
                    deadline_date=booking_info.deadline.astimezone(MSK_TIMEZONE),
                )
                self.set_info(message)
                if booking_info.start - event.date > booking_config.notification_threshold:
                    notifications.append(message)
            # Booking is already created - skip event.
            return booking_info

        need_cancel = bool(booking_info) and now_datetime <= booking_info.start
        need_create = create_datetime <= now_datetime <= booking_datetime
        veto_cancel = need_cancel and booking_info.start <= veto_datetime
        veto_create = need_create and booking_datetime <= veto_datetime
        self.logger.info('need_cancel = %s, need_create = %s, veto_cancel = %s, veto_create = %s',
                         need_cancel, need_create, veto_cancel, veto_create)

        if need_cancel:
            if need_create:
                if veto_cancel or veto_create:
                    message = (
                        'Проект {project}: Бронирование #{booking_id} ({booking_url}) '
                        'для вехи "{event}" релиза "{release}", '
                        'назначенное на {old_date}, должно быть перенесено на {new_date}, '
                        'но на операцию уже действует вето ({veto_days} дней).'
                    ).format(
                        project=project_key,
                        booking_id=booking_info.booking_id,
                        booking_url=BookingClient.get_booking_url(booking_info.booking_id),
                        event=event.title,
                        release=release.version,
                        old_date=booking_info.start.astimezone(MSK_TIMEZONE),
                        new_date=booking_datetime.astimezone(MSK_TIMEZONE),
                        veto_days=booking_config.veto_days,
                    )
                    self.set_info(message)
                    self.logger.info('Skip: Need move but veto.')
                    # Cannot move due to veto - skip event.
                    return booking_info
            else:
                if veto_cancel:
                    message = (
                        'Проект {project}: Бронирование #{booking_id} ({booking_url}) '
                        'для вехи "{event}" релиза "{release}", '
                        'назначенное на {old_date}, должно быть отменено, '
                        'но на операцию уже действует вето ({veto_days} дней).'
                    ).format(
                        project=project_key,
                        booking_id=booking_info.booking_id,
                        booking_url=BookingClient.get_booking_url(booking_info.booking_id),
                        event=event.title,
                        release=release.version,
                        old_date=booking_info.start.astimezone(MSK_TIMEZONE),
                        veto_days=booking_config.veto_days,
                    )
                    self.set_info(message)
                    self.logger.info('Skip: Need cancel but veto.')
                    # Cannot cancel due to veto - skip event.
                    return booking_info
        else:
            if need_create:
                if veto_create:
                    message = (
                        'Проект {project}: Бронирование для вехи "{event}" релиза "{release}" '
                        'должно быть назначено на {new_date}, '
                        'но на операцию уже действует вето ({veto_days} дней).'
                    ).format(
                        project=project_key,
                        event=event.title,
                        release=release.version,
                        new_date=booking_datetime.astimezone(MSK_TIMEZONE),
                        veto_days=booking_config.veto_days,
                    )
                    self.set_info(message)
                    self.logger.info('Skip: Need create but veto.')
                    # Cannot create due to veto - skip event.
                    return booking_info
            else:
                self.logger.info('Skip: Need nothing.')
                # Need nothing - skip event.
                return booking_info

        if need_cancel:
            self.logger.info('Action: Cancel booking.')
            old_booking_info = booking_info
            try:
                booking_info = self.cancel_booking(
                    project_key, event, old_booking_info)
            except BookingError as booking_error:
                message = (
                    'Проект {project}: Не получилось отменить бронирование #{booking_id} ({booking_url}) '
                    'для вехи "{event}" релиза "{release}", назначенное на {date}: {error_message}'
                ).format(
                    project=project_key,
                    booking_id=old_booking_info.booking_id,
                    booking_url=BookingClient.get_booking_url(old_booking_info.booking_id),
                    event=event.title,
                    release=release.version,
                    date=old_booking_info.start.astimezone(MSK_TIMEZONE),
                    error_message=booking_error.get_human_readable_description(),
                )
                self.set_info(message)
                notifications.append(message)
                raise

            message = (
                '{dry_run}Проект {project}: Отменено бронирование #{booking_id} ({booking_url}) '
                'для вехи "{event}" релиза "{release}", назначенное на {date}'
            ).format(
                dry_run='[DRY] ' if self.dry_run else '',
                project=project_key,
                booking_id=old_booking_info.booking_id,
                booking_url=BookingClient.get_booking_url(old_booking_info.booking_id),
                event=event.title,
                release=release.version,
                date=old_booking_info.start.astimezone(MSK_TIMEZONE),
            )
            self.set_info(message)

        if need_create:
            self.logger.info('Action: Create booking.')
            try:
                booking_info = self.create_booking(
                    project_key, release, event, booking_datetime,
                    booking_config.params, booking_config.meta_data)
            except BookingError as booking_error:
                message = (
                    'Проект {project}: Не получилось создать бронирование '
                    'для вехи "{event}" релиза "{release}" '
                    'на {date}: {error_message}'
                ).format(
                    project=project_key,
                    event=event.title,
                    release=release.version,
                    date=booking_datetime.astimezone(MSK_TIMEZONE),
                    error_message=booking_error.get_human_readable_description(),
                )
                self.set_info(message)
                notifications.append(message)
                raise

            message = (
                '{dry_run}Проект {project}: Создано бронирование #{booking_id} ({booking_url}) '
                'для вехи "{event}" релиза "{release}" на {booking_start}\n'
                'Дата выкатки: {event_date}\n'
                'Окончание дедлайна: {deadline_date}'
            ).format(
                dry_run='[DRY] ' if self.dry_run else '',
                project=project_key,
                booking_id=booking_info.booking_id,
                booking_url=BookingClient.get_booking_url(booking_info.booking_id),
                event=event.title,
                release=release.version,
                booking_start=booking_info.start.astimezone(MSK_TIMEZONE),
                event_date=event.date.astimezone(MSK_TIMEZONE),
                deadline_date=booking_info.deadline.astimezone(MSK_TIMEZONE),
            )
            self.set_info(message)
            if booking_info.start - event.date > booking_config.notification_threshold:
                notifications.append(message)

        return booking_info

    @classmethod
    def are_events_in_the_distant_past(cls, now_msk, events):
        """
        :type events: list[shuttle_client.Event]
        :type now_msk: datetime.datetime
        :rtype: bool
        """
        return all(e.date < now_msk - datetime.timedelta(weeks=1) for e in events)

    def process_release(self, project_key, release, booking_config_by_kind, notifications):
        """
        :type project_key: str
        :type release: shuttle_client.Release
        :type booking_config_by_kind: dict[str, config.BookingConfig]
        :type notifications: list[str]
        """
        self.logger.info('Project "%s": release #%s "%s"', project_key, release.id, release.version)
        self.logger.info('- responsible: "%s"', release.responsible)
        events = self.shuttle.get_events(project_key, release.id)
        events.sort(key=lambda e: e.date)
        if self.are_events_in_the_distant_past(self.now_msk, events):
            self.logger.info('- all events are in the past - skip release')
            return

        all_booking_kinds = set(booking_config_by_kind.keys())
        non_processed_booking_kinds = all_booking_kinds.copy()

        for event in events:
            if not BookingParams.does_exist_in(event):
                continue

            self.logger.info('Event #%s "%s" at "%s"', event.id, event.title, event.date)
            self.logger.info('- params: %s', event.parameters)
            BookingParams.validate_booking_params(event)

            booking_info = self.resolve_booking_info(self.logger, self.booking, project_key, release, event)
            booking_kind = BookingParams.get_booking_kind(event)
            if not booking_kind:
                continue

            booking_config = booking_config_by_kind.get(booking_kind)
            if not booking_config:
                self.logger.warning('Unknown booking kind "%s"', booking_kind)
                self.set_info(
                    'Проект "{}", релиз "{}", событие "{}":\n'
                    'Неизвестное значение параметра booking.kind: "{}"'
                    ' - отсутствует в конфигурации.'.format(
                        project_key, release.version, event.title, booking_kind))
                continue

            try:
                self.process_event(
                    project_key, release, event,
                    booking_config, booking_info,
                    notifications)
            except BookingError:
                exc_info = sys.exc_info()
                self._report_error(exc_info)
                self.logger.exception('Event processing failed', exc_info=exc_info)

            if booking_kind in non_processed_booking_kinds:
                non_processed_booking_kinds.remove(booking_kind)
            else:
                self.set_info(
                    'Проект "{}", релиз "{}", событие "{}":\n'
                    'Дублирование значения параметра booking.kind: "{}"'.format(
                        project_key, release.version, event.title, booking_kind))

        # Print "missed kinds" message if some booking kind is found only.
        if non_processed_booking_kinds and len(non_processed_booking_kinds) != len(all_booking_kinds):
            missed_booking_kinds = sorted(non_processed_booking_kinds)
            self.logger.warning('Non processed booking kinds: %s', missed_booking_kinds)
            self.set_info(
                'Проект "{}", релиз "{}": не найден(ы) booking kind: {}'.format(
                    project_key, release.version, missed_booking_kinds))

    def process(self, booking_config):
        """
        :type booking_config: dict[str, dict[str, config.BookingConfig]]
        """
        self._clear_errors()

        for project_key in sorted(booking_config.keys()):
            booking_config_by_kind = booking_config[project_key]
            self.logger.info('Project "%s"', project_key)
            releases = self.shuttle.get_releases(project_key)
            releases = sorted(releases, key=lambda r: r.version)
            for release in releases:
                notifications = []
                try:
                    self.process_release(project_key, release, booking_config_by_kind, notifications)
                finally:
                    if notifications:
                        notification_message = '\n\n'.join(notifications)
                        self.notify_responsible(project_key, release, notification_message)

        self._raise_errors_if_necessary()
