# -*- encoding: utf-8 -*-

import datetime
import json
import logging

import pytz

from sandbox import sdk2
from sandbox.common.utils import singleton_classproperty
from sandbox.projects.browser.booking.common import Cache
from sandbox.projects.browser.booking.common import DATETIME_FORMAT
from sandbox.projects.browser.booking.common import MSK_TIMEZONE
from sandbox.projects.browser.booking.config import ConfigLoader
from sandbox.projects.common import decorators
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.environments import PipEnvironment

BOOKING_PROD_URL = 'https://booking.yandex-team.ru'
BOOKING_TEST_URL = 'https://sandbox.booking.yandex-team.ru'


class Config(sdk2.parameters.String):
    @singleton_classproperty
    def default_value(cls):
        config_names = ConfigLoader.available_configs().keys()
        return config_names[0] if config_names else '---'


def now_datetime_msk():
    return pytz.UTC.localize(datetime.datetime.utcnow()).astimezone(MSK_TIMEZONE)


class BaseBrowserBookingTask(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('pyrsistent', '0.16.0'),
            PipEnvironment('jsonschema', '3.0.2', custom_parameters=['--upgrade-strategy', 'only-if-needed']),
            PipEnvironment('shuttle-client', '0.0.3'),
            PipEnvironment('teamcity-client', '4.9.0'),
            PipEnvironment('ya-booking-client', '1.0.2'),
            PipEnvironment('ya-messenger-client', '0.0.1'),
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 20 * 60  # 20 minutes.
        test_servers = sdk2.parameters.Bool(
            'Test servers.', description='Use test servers.',
            default=True)
        test_shuttle = sdk2.parameters.Bool(
            'Test shuttle server.', description='Use test shuttle server (only).',
            default=True)
        test_booking = sdk2.parameters.Bool(
            'Test booking server.', description='Use sandbox booking server (only).',
            default=True)
        now_datetime_msk = sdk2.parameters.String(
            'Now datetime (MSK)',
            description='Current datetime (MSK) to use in format {}.'.format(DATETIME_FORMAT))
        dry_run = sdk2.parameters.Bool(
            'Dry run.', description='Print log messages instead of modifying data.',
            default=True)
        skip_sending_messages = sdk2.parameters.Bool('Skip sending messages', default=True)
        booking_secrets_uuid = sdk2.parameters.YavSecret(
            'robot-bro-booking secrets',
            description='YAV UUID of booking secrets.',
            default='sec-01e57mzdbev4qt3edfsfv9x3hm')

    @decorators.memoized_property
    def now_msk(self):
        return MSK_TIMEZONE.localize(datetime.datetime.strptime(
            self.Parameters.now_datetime_msk, DATETIME_FORMAT))

    @decorators.memoized_property
    def logger(self):
        return logging.getLogger('booking')

    @decorators.memoized_property
    def user_agent(self):
        return '{} #{}'.format(self.type.name, self.id)

    @decorators.memoized_property
    def username(self):
        return self.Parameters.booking_secrets_uuid.data()['username']

    @decorators.memoized_property
    def booking_client(self):
        """ :rtype: ya_booking_client.YaBookingClient """
        oauth_token = self.Parameters.booking_secrets_uuid.data()['oauth_token']
        test_server = self.Parameters.test_servers or self.Parameters.test_booking
        server_url = BOOKING_TEST_URL if test_server else BOOKING_PROD_URL

        import ya_booking_client
        return ya_booking_client.YaBookingClient(oauth_token, server_url)

    @decorators.memoized_property
    def shuttle_client(self):
        """ :rtype: shuttle_client.ShuttleClient """
        oauth_token = self.Parameters.booking_secrets_uuid.data()['oauth_token']
        test_server = self.Parameters.test_servers or self.Parameters.test_shuttle

        import shuttle_client
        return shuttle_client.ShuttleClient(
            url=shuttle_client.get_server_url(test_server=test_server),
            token=oauth_token, logger=self.logger, user_agent=self.user_agent)

    @decorators.memoized_property
    def cache(self):
        """ :rtype: Cache """
        return Cache(self.shuttle_client, self.booking_client)

    @decorators.memoized_property
    def teamcity_client(self):
        """ :rtype: teamcity_client.TeamcityClient """
        username = self.Parameters.booking_secrets_uuid.data()['teamcity_username']
        password = self.Parameters.booking_secrets_uuid.data()['teamcity_password']
        import teamcity_client
        return teamcity_client.TeamcityClient(
            server_url='https://teamcity.browser.yandex-team.ru/',
            auth=(username, password))

    @decorators.memoized_property
    def msgr_registry_client(self):
        oauth_token = self.Parameters.booking_secrets_uuid.data()['oauth_token']
        import ya_messenger_client
        return ya_messenger_client.RegistryClient(
            token=oauth_token, logger=self.logger, user_agent=self.user_agent)

    @decorators.memoized_property
    def msgr_botplatform_client(self):
        oauth_token = self.Parameters.booking_secrets_uuid.data()['botplatform_oauthteam_token']
        import ya_messenger_client
        return ya_messenger_client.BotplatformClient(
            token=oauth_token, logger=self.logger, user_agent=self.user_agent)

    @decorators.memoized_property
    def msgr_my_guid(self):
        return self.msgr_botplatform_client.team.me().guid

    def send_message(self, responsible, chat_info, message):
        """
        :type responsible: str
        :type chat_info: shuttle_client.model.Chat
        :type message: str
        :rtype: bool
        """
        # TODO: Add chat type property to the shuttle_client
        chat_type = chat_info._data.get('type', 'YACHAT')  # noqa
        if chat_type != 'YACHAT':
            return False

        chat = self.msgr_registry_client.get_chat_by_id(chat_info.chat_id)

        if self.msgr_my_guid not in chat.members:
            logging.debug('Inviting user %s to chat %s', self.msgr_my_guid, chat.id)
            if not self.Parameters.skip_sending_messages:
                self.msgr_registry_client.invite(chat.invite_hash)

        logging.debug('Sending message to chat %s', chat.id)
        if self.Parameters.skip_sending_messages:
            logging.debug('Skip sending message')
        else:
            self.msgr_botplatform_client.bot.send_message(chat.id, '@{}:\n{}'.format(responsible, message))
        return True

    def send_email(self, responsible, message):
        email_message = (
            '{message}\n\n\nВы получили это письмо от {task_url}. Обычно такие '
            'сообщения приходят в релизном чате, но что-то пошло не так.').format(
            message=message, task_url='https://sandbox.yandex-team.ru/task/{}/view'.format(self.id))
        if self.Parameters.skip_sending_messages:
            logging.debug('Skip sending email')
        else:
            channel.sandbox.send_email(
                '{}@yandex-team.ru'.format(responsible), None,
                'Робот-Бронировщик', email_message,
                content_type='text/plain', charset='utf-8')

    def find_release_event(self, project_key, release):
        events = self.shuttle_client.get_events(project_key, release.id)
        import shuttle_client
        release_event = next((e for e in events if e.type == shuttle_client.TYPE_RELEASE), None)
        return release_event

    def notify_responsible(self, project_key, release, message):
        release_event = self.find_release_event(project_key, release)
        chat_info = release_event.chat if release_event else None

        addressees = []
        if release_event:
            addressees.extend([release_event.tester, release_event.responsible])
        addressees.append(release.responsible)
        addressee = next((a for a in addressees if a), None)

        self.set_info('Сообщение для {}:\n\n{}'.format(addressee, message))

        if chat_info:
            try:
                sent_to_chat = self.send_message(addressee, chat_info, message)
            except Exception:
                self.logger.exception('Cannot notify responsible by chats.')
                sent_to_chat = False
            if sent_to_chat:
                return

        try:
            self.send_email(release.responsible, message)
        except Exception:
            self.logger.exception('Cannot notify responsible by email.')

    def on_save(self):
        if not self.Parameters.now_datetime_msk:
            self.Parameters.now_datetime_msk = now_datetime_msk().strftime(DATETIME_FORMAT)


class BaseBrowserBookingTaskWithConfig(BaseBrowserBookingTask):
    class Parameters(BaseBrowserBookingTask.Parameters):
        with Config('Booking config') as config:
            for choice_name in ConfigLoader.available_configs().keys():
                config.values[choice_name] = choice_name

    def on_execute(self):
        self.logger.info('booking config name: %s', self.Parameters.config)
        config_path = ConfigLoader.available_configs()[self.Parameters.config]
        self.logger.info('booking config path: %s', config_path)
        merged_booking_config_json, self.booking_config = ConfigLoader.load_config(config_path)
        self.logger.info('merged booking config:\n%s', json.dumps(merged_booking_config_json, indent=4))
