# encoding: utf-8
from sandbox import common
from sandbox.common.errors import TaskError
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2


TEAMCITY_SERVER_URL = 'https://teamcity.browser.yandex-team.ru/'
TEAMCITY_BARE_REQUIRED_TAG = 'kraken'
TEAMCITY_BUILDS_BY_PLATFORM = {
    'mac': {
        'bare': 'BrowserBareUnsignedBuilds_StableMac',
        'branded': 'BrowserBrandedDistributives_PartnersStableMac',
    },
    'win': {
        'bare': 'BrowserBareUnsignedBuilds_StableWin',
        'branded': 'BrowserBrandedDistributives_PartnersStableWin',
    }
}
BRANDINGS_BRANCH = 'master'
ISSUE_COMMENT_TEMPLATE = u'Поставлены сборки браузера:\n{builds}'


class BrowserBrandedPartnersBuildsTrigger(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 100
        environments = (
            PipEnvironment('startrek-client', '2.3', custom_parameters=['--upgrade-strategy only-if-needed']),
            PipEnvironment('teamcity-client', '4.8.2', custom_parameters=['--upgrade-strategy only-if-needed'])
        )

    class Parameters(sdk2.Task.Parameters):
        partner_id = sdk2.parameters.String('Partner id for starting builds', required=True)
        issue = sdk2.parameters.String('Related issue key for sending reports', required=True)
        platforms = sdk2.parameters.List('Platforms titles', required=True)
        issue_comment_summonees = sdk2.parameters.List('Issue comment summonees list', default=['distrib-fellow'])
        with sdk2.parameters.Group('Credentials') as credentials_group:
            secrets_uid = sdk2.parameters.YavSecret(
                'distrib-fellow secrets',
                description=('YAV UUID of secrets. Must contain secrets: '
                             'TEAMCITY_TOKEN, STARTREK_TOKEN'),
                default='sec-01ehq7r4efw3dmztwx27e5gqwh')

    class Context(sdk2.Task.Context):
        build_ids = {}

    @property
    @common.utils.singleton
    def startrek(self):
        import startrek_client
        return startrek_client.Startrek(
            token=self.Parameters.secrets_uid.data()['STARTREK_TOKEN'],
            useragent=self.__class__.name,
        )

    @property
    @common.utils.singleton
    def teamcity(self):
        import teamcity_client
        return teamcity_client.TeamcityClient(
            server_url=TEAMCITY_SERVER_URL,
            auth=self.Parameters.secrets_uid.data()['TEAMCITY_TOKEN']
        )

    def trigger_build(self, platform):
        bare_builds = self.teamcity.Builds(
            buildType__id=TEAMCITY_BUILDS_BY_PLATFORM[platform]['bare'], status='SUCCESS',
            branch__default='any', tags=TEAMCITY_BARE_REQUIRED_TAG, count=1,
        )
        last_bare_build = next(iter(bare_builds), None)
        if last_bare_build is None:
            raise TaskError('Last successful bare build not found for platform {}'.format(platform))

        build = self.teamcity.BuildType(id=TEAMCITY_BUILDS_BY_PLATFORM[platform]['branded']).run_build(
            branch=last_bare_build.branch_name,
            change=last_bare_build.last_changes[0],
            properties={
                'partner-names': self.Parameters.partner_id,
                'brandings-branch': BRANDINGS_BRANCH,
            },
            comment='Triggered from sandbox task {}'.format(self.id),
            snapshot_dependencies=[last_bare_build]
        )
        self.Context.build_ids[platform] = build.web_url
        self.set_info('Build started: <a href="{}">{}</a>'.format(build.web_url, build.id), do_escape=False)

    def report_issue(self):
        issue = self.startrek.issues[self.Parameters.issue]
        text = ISSUE_COMMENT_TEMPLATE.format(
            builds='\n'.join('* {} - {}'.format(platform, url) for platform, url in self.Context.build_ids.items())
        )
        issue.comments.create(text=text, summonees=self.Parameters.issue_comment_summonees)

    def on_enqueue(self):
        unknown_platforms = set(self.Parameters.platforms) - set(TEAMCITY_BUILDS_BY_PLATFORM)
        if unknown_platforms:
            raise TaskError('Unknown platforms: {}. Available platforms: {}'.format(
                ','.join(unknown_platforms),
                ','.join(TEAMCITY_BUILDS_BY_PLATFORM.keys())
            ))

    def on_execute(self):
        for platform in self.Parameters.platforms:
            self.trigger_build(platform)
        self.report_issue()
