# encoding: utf-8
from collections import OrderedDict, namedtuple
import copy
import json
import logging
import os
import time
from xml.dom import minidom
from xml.etree import ElementTree as ET

from sandbox import common
from sandbox.common import errors
from sandbox.common.types.task import Semaphores
from sandbox.projects.browser.common.git import ConfigureGitEnvironment, GitEnvironment, repositories
from sandbox.projects.common import decorators
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2
from sandbox.sdk2.helpers import process


BitBucketData = namedtuple('BitBucketData', 'project, repo')
BITBUCKET_PROD = BitBucketData(project='stardust', repo='browser-brandings')
BITBUCKET_DEV = BitBucketData(project='~distrib-fellow', repo='browser-brandings-test')
WORKING_BRANCH = 'master'
DEFAULT_REVIEWERS = ['hmstr']

ISSUE_COMMENT_ERROR_TEMPLATE = (
    u'В процессе создания партнера ##{partner_id}## произошла ошибка. '
    u'Подробности: https://sandbox.yandex-team.ru/task/{task_id}'
)
ISSUE_COMMENT_CREATE_PARTNER_TEMPLATE = u'Партнер ##{partner_id}## был успешно создан.'
ISSUE_COMMENT_UPDATE_PARTNER_TEMPLATE = u'Партнер ##{partner_id}## был успешно обновлен.'
ISSUE_COMMENT_NO_CHANGE_PARTNER_TEMPLATE = u'Партнер ##{partner_id}## уже существует и содержит актуальные данные.'


@decorators.retries(3, delay=30, backoff=2, exceptions=process.subprocess.CalledProcessError)
def retried_git(*args, **kwargs):
    git_call(*args, **kwargs)


def git_call(*args, **kwargs):
    command_line = ['git'] + list(args)
    with process.ProcessLog(task=sdk2.task.Task.current, logger='git') as pl:
        process.subprocess.check_call(command_line, stdout=pl.stdout, stderr=pl.stderr, **kwargs)


class BrowserBrandedPartnersUpdate(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 3000  # now repo weight 1.4GB + reserve for future growth
        semaphores = Semaphores(
            acquires=(
                Semaphores.Acquire(name='BROWSER_BRANDED_PARTNER_UPDATE_SEMAPHORE', weight=1, capacity=1),
            ),
        )
        environments = (
            PipEnvironment('startrek-client', '2.3', custom_parameters=['--upgrade-strategy only-if-needed']),
            PipEnvironment(
                'bitbucket-server-client', '2.11.1', custom_parameters=['--upgrade-strategy only-if-needed']),
            PipEnvironment('natsort', '6.0.0'),
            GitEnvironment('2.24.1'),
            ConfigureGitEnvironment(email='distrib-fellow@yandex-team.ru', username='distrib-fellow'),
        )

    class Parameters(sdk2.Task.Parameters):
        partner_id = sdk2.parameters.String('Partner id for creating or updating', required=True)
        brand = sdk2.parameters.String('Partner brand title', required=True)
        clids = sdk2.parameters.JSON('JSON with clids', required=True, default={})
        issue = sdk2.parameters.String('Related issue key for sending reports', required=True)
        platforms = sdk2.parameters.List('Platforms titles', required=True)
        reviewers = sdk2.parameters.List('Reviewers list for PR', default=DEFAULT_REVIEWERS)
        issue_comment_summonees = sdk2.parameters.List('Issue comment summonees list', default=['distrib-fellow'])
        skip_force_merge = sdk2.parameters.Bool('Start script without force-merge created PR')
        use_test_repo = sdk2.parameters.Bool('Use test repo for development purpose')
        with sdk2.parameters.Group('Credentials') as credentials_group:
            secrets_uid = sdk2.parameters.YavSecret(
                'distrib-fellow secrets',
                description=('YAV UUID of secrets. Must contain secrets: '
                             'BITBUCKET_TOKEN, STARTREK_TOKEN, SSH_PRIVATE_KEY'),
                default='sec-01ehq7r4efw3dmztwx27e5gqwh')

    class Context(sdk2.Task.Context):
        changes_branch = None
        new = False
        updated = False

    @property
    @common.utils.singleton
    def bitbucket_client(self):
        import bitbucket
        return bitbucket.BitBucket(
            url='https://bitbucket.browser.yandex-team.ru/',
            token=self.Parameters.secrets_uid.data()['BITBUCKET_TOKEN'],
        )

    @property
    @common.utils.singleton
    def repo(self):
        if self.Parameters.use_test_repo:
            return repositories.Personal.browser_brandings_test()
        return repositories.Stardust.browser_brandings()

    @property
    @common.utils.singleton
    def startrek(self):
        import startrek_client
        return startrek_client.Startrek(
            token=self.Parameters.secrets_uid.data()['STARTREK_TOKEN'],
            useragent=self.__class__.name,
        )

    def browser_brandings_repo_path(self, *args):
        repo_path = 'browser-brandings-test' if self.Parameters.use_test_repo else 'browser-brandings'
        return str(self.path(repo_path, *args))

    def partners_dir(self):
        return self.browser_brandings_repo_path('partners_distrib')

    def partner_dir(self):
        return self.browser_brandings_repo_path('partners_distrib', self.Parameters.partner_id)

    def prepare_repo(self):
        self.Context.changes_branch = 'wp/robots/{issue_key}/{ts}'.format(
            issue_key=self.Parameters.issue, ts=int(time.time())
        )
        self.repo.clone(self.browser_brandings_repo_path(), WORKING_BRANCH)

        git_call('checkout', '-b', self.Context.changes_branch,
                 cwd=self.browser_brandings_repo_path())

    def update_partner(self):
        if not os.path.exists(self.partner_dir()):
            self._create_partner_dir()
            self.Context.updated = True
        self._update_partners_file_if_needed()

    def commit_changes_and_create_pr(self):
        if not (self.Context.new or self.Context.updated):
            return

        with sdk2.ssh.Key(self, private_part=self.Parameters.secrets_uid.data()['SSH_PRIVATE_KEY']):
            git_call('add', self.partners_dir(),
                     cwd=self.browser_brandings_repo_path())
            git_call('commit', '-m', self._get_commit_message(),
                     cwd=self.browser_brandings_repo_path())

            retried_git('push', 'origin', self.Context.changes_branch,
                        cwd=self.browser_brandings_repo_path())

        bitbucket_data = BITBUCKET_DEV if self.Parameters.use_test_repo else BITBUCKET_PROD
        pr = self.bitbucket_client.projects[
            bitbucket_data.project
        ].repos[
            bitbucket_data.repo
        ].pull_requests.create_pull_request(
            title=self._get_commit_message(),
            source_branch=self.Context.changes_branch,
            dest_branch=WORKING_BRANCH,
            reviewers=self.Parameters.reviewers,
        )
        self.set_info('PR created: <a href="{}">{}</a>'.format(pr.web_url, pr.id), do_escape=False)
        if self.Parameters.skip_force_merge:
            logging.info('Skip force-merge')
            return

        pr.force_merge(delete_branch=True)

    def report_issue(self, error=False):
        issue = self.startrek.issues[self.Parameters.issue]
        if error:
            text = ISSUE_COMMENT_ERROR_TEMPLATE
        elif self.Context.new:
            text = ISSUE_COMMENT_CREATE_PARTNER_TEMPLATE
        elif self.Context.updated:
            text = ISSUE_COMMENT_UPDATE_PARTNER_TEMPLATE
        else:
            text = ISSUE_COMMENT_NO_CHANGE_PARTNER_TEMPLATE
        text = text.format(partner_id=self.Parameters.partner_id, task_id=self.id)
        issue.comments.create(text=text, summonees=self.Parameters.issue_comment_summonees)

    def _create_partner_dir(self):
        partner_dir_path = self.partner_dir()
        os.mkdir(partner_dir_path)
        for clid_name, clid_content in self.Parameters.clids.items():
            with open(os.path.join(partner_dir_path, '{}.xml'.format(clid_name)), 'wb') as output:
                output.write(self._prepare_xml(clid_content))

    def _prepare_xml(self, clid_content):
        def prettify(root):
            """
            Return a pretty-printed XML string for the Element.
            """
            parsed = minidom.parseString(ET.tostring(root))
            return parsed.toprettyxml(encoding='utf8')

        import natsort
        root = ET.Element('vendor', {'name': self.Parameters.partner_id})
        for clid_tag in natsort.natsorted(clid_content.keys()):
            child = ET.SubElement(root, clid_tag)
            child.text = str(clid_content[clid_tag])
        return prettify(root)

    def _update_partners_file_if_needed(self):
        with open(os.path.join(self.partners_dir(), 'partners.json'), 'r') as partners_file:
            partners = json.load(partners_file, object_pairs_hook=OrderedDict)

        if self.Parameters.partner_id in partners['partners']:
            current_partner_data = copy.deepcopy(partners['partners'][self.Parameters.partner_id])
            partners['partners'][self.Parameters.partner_id]['brand_name'] = self.Parameters.brand
            partners['partners'][self.Parameters.partner_id]['platforms'] = sorted(self.Parameters.platforms)
            self.Context.updated = (current_partner_data != partners['partners'][self.Parameters.partner_id])
        else:
            partners['partners'][self.Parameters.partner_id] = OrderedDict()
            partners['partners'][self.Parameters.partner_id]['brand_name'] = self.Parameters.brand
            partners['partners'][self.Parameters.partner_id]['platforms'] = sorted(self.Parameters.platforms)
            self.Context.new = True

        if self.Context.updated or self.Context.new:
            with open(os.path.join(self.partners_dir(), 'partners.json'), 'w') as partners_file:
                json.dump(partners, partners_file, indent=2, separators=(',', ': '))

    def _get_commit_message(self):
        return '{issue_key}: {action} partner with id {partner_id}'.format(
            issue_key=self.Parameters.issue,
            action='Create' if self.Context.new else 'Update',
            partner_id=self.Parameters.partner_id,
        )

    def on_save(self):
        if not isinstance(self.Parameters.clids, dict):
            raise errors.TaskError('Param clids must be dict')

    def on_execute(self):
        error = False
        try:
            self.prepare_repo()
            self.update_partner()
            self.commit_changes_and_create_pr()
        except Exception:
            error = True
            raise
        finally:
            self.report_issue(error=error)
