import abc
import multiprocessing
import os
import shlex

import jinja2

from sandbox.common import errors
from sandbox.common.config import Registry
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

from sandbox.projects.browser.common.bitbucket import BitBucket, DEFAULT_BITBUCKET_URL
from sandbox.projects.browser.common.chromium_releases import Release
from sandbox.projects.browser.common.RunBrowserScript import RunBrowserScript
from sandbox.projects.browser.common.timeout import GracefulKillBeforeTimeoutMixin
from sandbox.projects.common.teamcity import TeamcityArtifacts

from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox import sdk2
from sandbox.sdk2.service_resources import TaskLogs

DEFAULT_TAGS = ctc.Tag.BROWSER


def _host_ncpu(host):
    return host.info['system'].get('ncpu', 0)


class BuildBrowserTargets(GracefulKillBeforeTimeoutMixin, RunBrowserScript):
    class Requirements(RunBrowserScript.Requirements):
        client_tags = DEFAULT_TAGS

        class Caches(RunBrowserScript.Requirements.Caches):
            pass

    class Parameters(RunBrowserScript.Parameters):
        max_restarts = 2
        kill_timeout = 7 * 60 * 60

        with RunBrowserScript.Parameters.general_settings() as general_settings:
            teamcity_build_id = sdk2.parameters.String('ID of parent teamcity build', required=True)
            target_cpu = sdk2.parameters.String('Target CPU')
            channel = sdk2.parameters.String('Channel')
            extra_gn_args = sdk2.parameters.String('Extra GN args')
            extra_args = sdk2.parameters.String('Extra script arguments')

        with sdk2.parameters.Group('Task settings') as task_settings:
            additional_tags = sdk2.parameters.CustomClientTags(
                'Additional client tags (will be intersected with default)')

            may_increase_requirements = sdk2.parameters.Bool(
                'Increase requirements (disk space, CPU, RAM) if they are lower than default for specified platform',
                default=True)

            lxc_container_resource_id = sdk2.parameters.Integer(
                'ID of LXC container resource to use on linux hosts', default=None, required=False)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            yav_token_vault = sdk2.parameters.String(
                'Vault item with yav token', default='robot-browser-infra_yav_token')
            bitbucket_token_vault = sdk2.parameters.String(
                'Vault item with oauth token for bitbucket', default='robot-bro-infra_bitbucket_token')

        with sdk2.parameters.Group('Crosscompilation') as crosscompilation:
            allow_crosscompilation_on_linux = sdk2.parameters.Bool(
                'Allow crosscompilation on linux', default=False)

    class Context(RunBrowserScript.Context):
        # Will be true if task should run on Linux host
        force_linux_host = False

    TARGET_MODULE = abc.abstractproperty()

    secret_envvars = RunBrowserScript.secret_envvars + ('YAV_TOKEN',)

    DEFAULT_DISK_SPACE = {
        'android': 110 * 1024,
        'ios': 150 * 1024,
        'linux': 140 * 1024,
        'mac': 100 * 1024,
        ('mac', 'arm64'): 140 * 1024,
        'win': 110 * 1024,
    }
    DEFAULT_CORES = {
        'android': 15,
        'ios': 4,
        'linux': 15,
        'mac': 4,
        'win': 16,
    }
    DEFAULT_RAM = {
        'android': 31 * 1024,
        'ios': 16 * 1024,
        'linux': 31 * 1024,
        'mac': 16 * 1024,
        'win': 32 * 1024,
    }

    PLATFORMS_RUN_ON_LINUX = ('android', 'linux')

    @sdk2.footer()
    def footer(self):
        logs_resource = TaskLogs.find(task=self).first()
        logs_url = logs_resource.http_proxy if logs_resource else None

        teamcity_log_resource = self._find_script_tc_log()
        teamcity_log_url = teamcity_log_resource.http_proxy if teamcity_log_resource else None

        artifacts_resource = TeamcityArtifacts.find(task=self, state=ctr.State.READY).first()
        artifacts_url = artifacts_resource.http_proxy if artifacts_resource else None

        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
        return env.get_template('footer.html').render(
            logs_url=logs_url,
            teamcity_log_url=teamcity_log_url,
            browser_dir_url=artifacts_url or 'https://proxy.sandbox.yandex-team.ru/task/{}/browser'.format(self.id),
        )

    def on_enqueue(self):
        host_is_linux = self.Parameters.platform in self.PLATFORMS_RUN_ON_LINUX
        if self.Context.force_linux_host:
            self.Requirements.client_tags = DEFAULT_TAGS & ctc.Tag.Group.LINUX
            host_is_linux = True

        if self.Parameters.may_increase_requirements:
            def get_default_requirement(requirements_dict, fallback=0):
                platform = 'linux' if host_is_linux else self.Parameters.platform
                key1 = (platform, self.Parameters.target_cpu)
                key2 = platform
                return requirements_dict.get(key1, requirements_dict.get(key2, fallback))

            default_disk_space = get_default_requirement(self.DEFAULT_DISK_SPACE)
            self.Requirements.disk_space = max(self.Requirements.disk_space, default_disk_space)

            default_cores = get_default_requirement(self.DEFAULT_CORES)
            self.Requirements.cores = max(self.Requirements.cores, default_cores)

            default_ram = get_default_requirement(self.DEFAULT_RAM)
            self.Requirements.ram = max(self.Requirements.ram, default_ram)

        if self.Parameters.additional_tags:
            self.Requirements.client_tags = DEFAULT_TAGS & self.Parameters.additional_tags

        if host_is_linux and self.Parameters.lxc_container_resource_id:
            self.Requirements.container_resource = self.Parameters.lxc_container_resource_id

        if self.Parameters.binary_platform != 'none':
            binary_platform = self.Parameters.binary_platform
            if host_is_linux:
                binary_platform = 'linux'
            resource = self.find_binary_tasks_resource(binary_platform)
            if resource:
                self.Requirements.tasks_resource = resource
            else:
                raise RuntimeError('No tasks resource found')

        super(BuildBrowserTargets, self).on_enqueue()

    def script_cmd(self, python_executable):
        cmd = [str(python_executable), '-m', self.TARGET_MODULE, self.platform]
        if self.Parameters.target_cpu:
            cmd += ['--target-cpu', self.Parameters.target_cpu]
        if self.Parameters.channel:
            cmd += ['--channel', self.Parameters.channel]
        if self.Parameters.extra_gn_args:
            cmd += ['--extra-gn-args', self.Parameters.extra_gn_args]
        cmd += shlex.split(self.Parameters.extra_args)
        return cmd

    def script_extra_env(self):
        cpu_per_slot = multiprocessing.cpu_count() / Registry().client.max_job_slots
        # Multislot clients have |max_job_slots| equal to their number of cpu cores, so each slot can occupy one
        # CPU. But such clients may run tasks that require, for example, 5 CPU, in that case client would dedicate
        # 5 slots to such task.
        # On the other hand client may contain only one slot with say 12 CPU, and task that requires 5 CPU would
        # occupy slot with 12 CPU, so let it use all of them, to make it perform faster.
        number_of_cores_to_use = max(self.Requirements.cores, cpu_per_slot)
        env = super(BuildBrowserTargets, self).script_extra_env()
        env.update({
            'CIPD_CACHE_DIR': SandboxEnvironment.exclusive_build_cache_dir('cipd_cache'),
            'DC_CACHE_TYPE': 'none',
            'DC_LOCAL_THREADS_NUMBER': str(number_of_cores_to_use),
            'GCLIENT_CACHE_DIR': self.gclient_cache_path,
            'GCLIENT_RUNHOOKS_CACHE_DIR': SandboxEnvironment.exclusive_build_cache_dir('gclient_hooks'),
            'GIT_CACHE_PATH': self.gclient_cache_path,
            'LANG': 'en_US.UTF-8',
            'SLOT_CPU_NUMBER': str(number_of_cores_to_use),
            'SLOT_RAM_MB': str(self.Requirements.ram),
            'TEAMCITY_BUILD_ID': str(self.Parameters.teamcity_build_id),
            'YAV_TOKEN': sdk2.Vault.data(self.Parameters.yav_token_vault),
        })
        if self.Parameters.platform == 'android':
            env.update({
                'GRADLE_USER_HOME': SandboxEnvironment.exclusive_build_cache_dir('gradle'),
            })
        elif self.Parameters.platform == 'win':
            env.update({
                'DEPOT_TOOLS_WIN_TOOLCHAIN_ROOT': SandboxEnvironment.exclusive_build_cache_dir('win_toolchain_cache'),
            })
        return env

    def maybe_rebounce_to_linux_host(self):
        bb = BitBucket(DEFAULT_BITBUCKET_URL, 'x-oauth-token', sdk2.Vault.data(self.Parameters.bitbucket_token_vault))
        version_file = bb.load_file(project='STARDUST', repo='browser', path='src/chrome/VERSION',
                                    at=self.Parameters.commit)
        release = Release.fromversionfile(version_file, set())
        can_run_on_linux = release.major_version > 93 or (
            release.major_version == 93 and release.build_version >= 4577)
        if can_run_on_linux:
            self.Context.force_linux_host = True
            self.Context.save()
            raise errors.TemporaryError('Rebouncing build to linux host')

    def on_execute(self):
        if (self.Parameters.allow_crosscompilation_on_linux and not self.Context.force_linux_host):
            self.maybe_rebounce_to_linux_host()

        return super(BuildBrowserTargets, self).on_execute()
