import os
import platform

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox.projects.browser.common import binary_tasks
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.hermetic_xcode import HermeticXcodeEnvironment
from sandbox.projects.browser.common.swift_toolchain import SwiftToolchainEnvironment
from sandbox.projects.common.teamcity import TeamcityArtifactsContext

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessRegistry, subprocess


DEFAULT_TAGS = ctc.Tag.BROWSER


class BuildHomeAPIGenerator(binary_tasks.CrossPlatformBinaryTaskMixin, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        cores = 8
        ram = 4 * 1024
        disk_space = 20 * 1024
        dns = ctm.DnsType.DNS64
        environments = (
            GitEnvironment('2.32.0'),
        )

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Repositories settings') as repositories_settings:
            branch = sdk2.parameters.String('Branch to checkout on', default='master')
            commit = sdk2.parameters.String('Commit to checkout on')
        kill_timeout = 4 * 60 * 60

        with sdk2.parameters.Group('General settings') as general_settings:
            with sdk2.parameters.String('Target platform') as platform:
                platform.values.default = platform.Value('default (host OS)', default=True)
                platform.values.linux = platform.Value('linux')
                platform.values.mac = platform.Value('mac')

        with sdk2.parameters.Group('Task settings') as task_settings:
            additional_tags = sdk2.parameters.CustomClientTags(
                'Additional client tags (will be intersected with default)')

            lxc_container_resource_id = sdk2.parameters.Integer(
                'ID of LXC container resource to use on linux hosts', default=None, required=False)

        _binary_task_params = binary_tasks.cross_platform_binary_task_parameters()

    def checkout_path(self, *args):
        return str(self.path('home-api-generator', *args))

    @property
    def home_api_generator_dir(self):
        return self.checkout_path('HomeAPIGenerator')

    def script_arguments(self, swift_path):
        common_arguments = [swift_path, 'build', '-c', 'release']
        if self.platform == 'mac':
            return common_arguments + ['--arch', 'x86_64']
        else:
            return common_arguments + ['-v', '-Xswiftc', '-v', '-Xswiftc', '-static-executable', '-Xlinker', '-x',
                                       '-Xlinker', '-s']

    def provide_swift_toolchain(self):
        platform = {
            'mac': 'darwin',
        }.get(self.platform, self.platform)
        return SwiftToolchainEnvironment(platform, '5.5.2', '2').prepare()

    @property
    def platform(self):
        if self.Parameters.platform == 'default':
            return {
                'Darwin': 'mac',
                'Linux': 'linux',
            }[platform.system()]
        else:
            return self.Parameters.platform

    def checkout_repository(self):
        repositories.ML.mobile_alice_library_crossplatform(filter_branches=False).clone(
            self.checkout_path(), self.Parameters.branch, self.Parameters.commit,
        )

    def on_enqueue(self):
        if self.Parameters.additional_tags:
            self.Requirements.client_tags = DEFAULT_TAGS & self.Parameters.additional_tags

        if self.Parameters.lxc_container_resource_id:
            self.Requirements.container_resource = self.Parameters.lxc_container_resource_id

    def on_prepare(self):
        super(BuildHomeAPIGenerator, self).on_prepare()
        if self.Parameters.platform == 'mac':
            hermetic_xcode = HermeticXcodeEnvironment('12.4')
            hermetic_xcode.prepare()

    def on_execute(self):
        self.checkout_repository()
        swift_path = self.provide_swift_toolchain()
        tac = TeamcityArtifactsContext(
            path_suffix='build.log', log_name='build.log', tc_service_messages_description='Build HomeAPIGenerator')
        with tac, ProcessRegistry:
            try:
                tac.logger.info('Building using swift compiler at %s', swift_path)
                subprocess.check_output(
                    self.script_arguments(swift_path),
                    cwd=self.home_api_generator_dir)
            except subprocess.CalledProcessError as e:
                tac.logger.exception('Error occured while building HomeAPIGenerator')
                tac.logger.info('Build output: %s', e.output)
                tac.logger.info("##teamcity[buildProblem description='Failed to build HomeAPIGenerator']")
                raise
            else:
                artifact_path = os.path.join(self.home_api_generator_dir, '.build', 'release', 'HomeAPIGenerator')
                tac.logger.info("##teamcity[publishArtifacts '{}']".format(artifact_path))
