import json
import logging
import os
import platform
import shutil
import zipfile

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox.projects.browser.common import binary_tasks
from sandbox.projects.browser.common.chromium_releases import Release
from sandbox.projects.browser.common.contextmanagers import TempfileContext, ChdirContext
from sandbox.projects.browser.common.depot_tools import DepotToolsEnvironment
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.teamcity_step import teamcity_step
from sandbox.projects.browser.common.timeout import GracefulKillBeforeTimeoutMixin

from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, ProcessRegistry, subprocess

DEFAULT_TAGS = ctc.Tag.BROWSER

GCLIENT_SPEC = '''
solutions=[
  {{
    'custom_deps': {{}},
    'custom_hooks': [
        {{
            'action': ['python3', 'src/build/linux/sysroot_scripts/install-sysroot.py', '--arch=i386'],
            'pattern': '.',
            'name': 'i386-sysroot',
            'condition': 'checkout_android',
        }},
        {{
            'action': ['python3', 'src/build/linux/sysroot_scripts/install-sysroot.py', '--arch=amd64'],
            'pattern': '.',
            'name': 'amd64-sysroot',
            'condition': 'checkout_android',
        }},
    ],
    'custom_vars': {{ {gclient_vars} }},
    'deps_file': '.DEPS.snapshot',
    'managed': False,
    'name': 'src',
    'url': None
  }}
]
target_os=['{target_os}']
verify_syntax=True
'''

BUILD_TIMESTAMP_SCRIPT = 'ya_compute_build_timestamp.py'


def _to_release(version_string):
    return Release(version_string, None, None)


class BuildChromiumSnapshot(binary_tasks.CrossPlatformBinaryTaskMixin, GracefulKillBeforeTimeoutMixin, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = DEFAULT_TAGS
        disk_space = 140 * 1024
        dns = ctm.DnsType.DNS64
        cores = 12
        environments = (
            GitEnvironment('2.32.0'),
        )

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 2
        kill_timeout = 24 * 60 * 60

        with sdk2.parameters.Group('Repositories settings') as repositories_settings:
            branch = sdk2.parameters.String('Branch to checkout on', default='upstream/dev')
            commit = sdk2.parameters.String('Commit to checkout on')

        with sdk2.parameters.Group('General settings') as general_settings:
            with sdk2.parameters.String('Target platform') as platform:
                platform.values.default = platform.Value('default (host OS)', default=True)
                platform.values.android = platform.Value('android')
                platform.values.linux = platform.Value('linux')
                platform.values.mac = platform.Value('mac')
                platform.values.win = platform.Value('win')

            teamcity_build_id = sdk2.parameters.String('ID of parent teamcity build', required=True)

            targets = sdk2.parameters.String('Targets to build', required=True)
            gclient_variables = sdk2.parameters.String('Gclient variables')
            gn_args = sdk2.parameters.String('GN arguments', required=True)
            build_timestamp_script = sdk2.parameters.String('Contents of build_timestamp.py script')

        with sdk2.parameters.Group('Task settings') as task_settings:
            additional_tags = sdk2.parameters.CustomClientTags(
                'Additional client tags (will be intersected with default)')

            lxc_container_resource_id = sdk2.parameters.Integer(
                'ID of LXC container resource to use on linux hosts', default=None, required=False)

        with sdk2.parameters.Group('Secrets') as secrets:
            robot_secrets = sdk2.parameters.YavSecret('YAV secret with Teamcity access token',
                                                      default='sec-01cn4cz7j3tryd6n540b5ebgkj#TEAMCITY_TOKEN')

        _binary_task_params = binary_tasks.cross_platform_binary_task_parameters()

    PATCHES = {
        'patches/dawn.patch': {
            'from': _to_release('105.0.0.0'),
        },
    }

    def on_enqueue(self):
        if self.Parameters.additional_tags:
            self.Requirements.client_tags = DEFAULT_TAGS & self.Parameters.additional_tags

        if self.Parameters.lxc_container_resource_id:
            self.Requirements.container_resource = self.Parameters.lxc_container_resource_id

        super(BuildChromiumSnapshot, self).on_enqueue()

    def chromium_path(self, *args):
        return self.path('chromium', *args)

    @property
    def src_path(self):
        return self.chromium_path('src')

    @property
    def ninja_dir(self):
        return self.chromium_path('src', 'out', 'Default')

    @property
    def platform(self):
        if self.Parameters.platform == 'default':
            return {
                'Darwin': 'mac',
                'Linux': 'linux',
                'Windows': 'win',
            }[platform.system()]
        else:
            return self.Parameters.platform

    @property
    def targets(self):
        return self.Parameters.targets.split()

    @property
    def source_deps_path(self):
        return self.ninja_dir.joinpath('source_deps.json')

    def get_current_release(self):
        with open(str(self.chromium_path('src', 'chrome', 'VERSION'))) as chrome_version:
            return Release.fromversionfile(chrome_version.read(), None)

    def checkout_chromium(self):
        repositories.Stardust.browser(filter_branches=False).clone(
            str(self.chromium_path()), self.Parameters.branch, self.Parameters.commit)

    def provide_depot_tools(self):
        depot_tools_env = DepotToolsEnvironment(
            deps_file=str(self.chromium_path('src', '.DEPS.snapshot')),
            dep_name='src/third_party/depot_tools',
        )
        depot_tools_env.prepare()
        return depot_tools_env.depot_tools_folder

    def clean_files(self):
        logging.debug('Cleaning %s', self.chromium_path())
        # Remove repository manually, because Sandbox does it too slowly
        # (it iterates through all files and checks if file is a resource).
        shutil.rmtree(str(self.chromium_path()), ignore_errors=True)
        logging.debug('Cleaning finished')

    def gclient_extra_env(self):
        env = {
            'CIPD_CACHE_DIR': SandboxEnvironment.exclusive_build_cache_dir('cipd_cache'),
            'LANG': 'en_US.UTF-8',
            'USE_CIPD_PROXY': '1',
        }
        if self.platform == 'win':
            env.update({
                'DEPOT_TOOLS_WIN_TOOLCHAIN_ROOT': SandboxEnvironment.exclusive_build_cache_dir('win_toolchain_cache'),
            })
        return env

    def gclient_sync(self, depot_tools_dir):
        gclient_name = 'gclient.bat' if platform.system() == 'Windows' else 'gclient'
        gclient_path = depot_tools_dir.joinpath(gclient_name)
        target_os = 'unix' if self.platform == 'linux' else self.platform
        with open(str(self.chromium_path('.gclient')), 'w') as dot_gclient:
            dot_gclient.write(GCLIENT_SPEC.format(gclient_vars=self.Parameters.gclient_variables,
                                                  target_os=target_os))
        cmd = [str(gclient_path), 'sync', '--verbose', '--force', '--ignore_locks']
        extra_env = self.gclient_extra_env()
        logging.info('GClient sync extra environment:\n%r', extra_env)
        env = dict(os.environ, **extra_env)
        with ProcessRegistry, ProcessLog(self, logger='chromium_gclient_sync') as log:
            subprocess.check_call(cmd, cwd=str(self.chromium_path()), stdout=log.stdout, stderr=log.stdout, env=env)

    def apply_patches(self):
        from library.python import resource

        current_release = self.get_current_release()
        for patch_name, patch_releases in self.PATCHES.items():
            if 'from' in patch_releases and current_release < patch_releases['from']:
                logging.info('Skipping patch %s as current version is too young', patch_name)
                continue
            if 'until' in patch_releases and current_release > patch_releases['until']:
                logging.info('Skipping patch %s as current version is too old', patch_name)
                continue

            patch = resource.find(patch_name)
            with TempfileContext() as patch_path:
                with open(patch_path, 'wb') as patch_file:
                    patch_file.write(patch)
                cmd = ['git', 'apply', '--verbose', '--whitespace=fix', patch_path]
                log_name = 'apply_custom_patch_{}'.format(os.path.basename(patch_name))
                with ProcessRegistry, ProcessLog(self, logger=log_name) as log:
                    subprocess.check_call(cmd, cwd=str(self.chromium_path()), stdout=log.stdout, stderr=log.stdout)

    def run_gn(self, depot_tools_dir, log, *arguments):
        gn_name = 'gn.bat' if platform.system() == 'Windows' else 'gn'
        gn_path = depot_tools_dir.joinpath(gn_name)
        cmd = [str(gn_path)] + list(arguments)
        if log is None:
            return subprocess.check_output(cmd, cwd=str(self.src_path))
        return subprocess.check_call(cmd, cwd=str(self.src_path), stdout=log.stdout, stderr=log.stdout)

    def gn_gen(self, depot_tools_dir):
        os.makedirs(str(self.ninja_dir))
        with open(str(self.ninja_dir.joinpath('args.gn')), 'w') as gn_args:
            gn_args.write(self.Parameters.gn_args)
        with ProcessRegistry, ProcessLog(self, logger='gn_gen') as log:
            self.run_gn(depot_tools_dir, log, 'gen', '--check', str(self.ninja_dir))

    def build_targets(self, depot_tools_dir):
        ninja_name = 'ninja.exe' if platform.system() == 'Windows' else 'ninja'
        ninja_path = depot_tools_dir.joinpath(ninja_name)
        cmd = [str(ninja_path), '-C', str(self.ninja_dir), '-k1000'] + self.targets
        with ProcessRegistry, ProcessLog(self, logger='ninja') as log:
            subprocess.check_call(cmd, cwd=str(self.chromium_path()), stdout=log.stdout, stderr=log.stdout)

    def list_sources(self):
        cmd = ['git', 'ls-tree', '-rt', '--name-only', '--full-tree', 'HEAD']
        return subprocess.check_output(cmd, cwd=str(self.chromium_path())).splitlines()

    def write_source_deps(self, runtime_deps):
        sources = set(self.list_sources())
        source_deps = []
        for dep in runtime_deps:
            dep_relative_to_root = str(self.ninja_dir.joinpath(dep).resolve().relative_to(self.chromium_path()))
            dep_relative_to_root = dep_relative_to_root.replace('\\', '/')
            if dep_relative_to_root in sources:
                source_deps.append(dep_relative_to_root)

        with open(str(self.source_deps_path), 'w') as source_deps_file:
            json.dump(sorted(source_deps), source_deps_file, indent=2)

    def publish_archive(self, deps, archive_name, tac):
        with ChdirContext(str(self.ninja_dir)):
            with zipfile.ZipFile(archive_name, 'w', compression=zipfile.ZIP_DEFLATED, allowZip64=True) as archive:
                archive_files = set()

                def add_file(file_path):
                    archive_files.add(file_path)

                for path in set(deps):
                    if os.path.isdir(path):
                        for root, _, files in os.walk(path):
                            map(add_file, [os.path.join(root, file_path) for file_path in files])
                    else:
                        assert os.path.isfile(path)
                        add_file(path)

                for file_path in archive_files:
                    archive.write(file_path)

        tac.logger.info("##teamcity[publishArtifacts '{}']".format(str(self.ninja_dir.joinpath(archive_name))))

    def publish_apks(self, tac):
        archive_path = str(self.ninja_dir.joinpath('chrome.zip'))
        with zipfile.ZipFile(archive_path, 'w', compression=zipfile.ZIP_DEFLATED, allowZip64=True) as archive:
            for root, _, files in os.walk(str(self.ninja_dir.joinpath('apks'))):
                for path in files:
                    archive.write(os.path.join(root, path), arcname='apks/' + path)
        tac.logger.info("##teamcity[publishArtifacts '{}']".format(archive_path))

    def collect_artifacts(self, depot_tools_dir, tac):
        all_labels = set(self.run_gn(depot_tools_dir, None, 'ls', str(self.ninja_dir)).split())
        def find_matching_labels(target):
            return [label for label in all_labels if label.rsplit(':', 1)[-1] == target]
        target_labels = []
        for target in self.targets:
            matched_labels = find_matching_labels(target)
            assert len(matched_labels) == 1, 'Several gn labels match target {}: {}'.format(target, matched_labels)
            target_labels.append(matched_labels[0])

        runtime_deps = set()
        for label in target_labels:
            runtime_deps |= set(self.run_gn(depot_tools_dir, None, 'desc', str(self.ninja_dir),
                                            label, 'runtime_deps').splitlines())

        self.write_source_deps(runtime_deps)
        debug_deps, nondebug_deps = [], []
        for dep in runtime_deps:
            if dep.startswith('..') or os.path.basename(dep) == 'browser_tests.exe' or 'nacl_irt' in dep:
                continue
            (debug_deps if dep.endswith('.pdb') else nondebug_deps).append(dep)
        nondebug_deps.append(str(self.source_deps_path.relative_to(self.ninja_dir)))

        self.publish_archive(nondebug_deps, 'chrome.zip', tac)
        self.publish_archive(debug_deps, 'chrome_debug.zip', tac)

    def tag_build(self, tac):
        import teamcity_client

        chromium_release = self.get_current_release()
        tc_client = teamcity_client.client.TeamcityClient(
            server_url='teamcity.browser.yandex-team.ru',
            auth=self.Parameters.robot_secrets.data()[self.Parameters.robot_secrets.default_key])
        current_tc_build = tc_client.builds[self.Parameters.teamcity_build_id]
        tac.logger.info("##teamcity[buildNumber '{version} #{{build.number}}']".format(
            version=chromium_release.version))
        current_tc_build.add_tags(['chromium_{}'.format(chromium_release.version)])

    def on_execute(self):
        with teamcity_step(self, 'Checkout chromium snapshot', 'checkout'):
            self.checkout_chromium()
        with teamcity_step(self, 'Apply patches', 'apply-patches'):
            self.apply_patches()

        with teamcity_step(self, 'Provide depot_tools', 'depot-tools'):
            depot_tools_dir = self.provide_depot_tools()
        with teamcity_step(self, 'GClient sync', 'gclient-sync'):
            self.gclient_sync(depot_tools_dir)

        if self.Parameters.build_timestamp_script:
            with open(str(self.chromium_path('src', 'build', BUILD_TIMESTAMP_SCRIPT)), 'w') as build_timestamp_script:
                build_timestamp_script.write(self.Parameters.build_timestamp_script)

        with teamcity_step(self, 'GN gen', 'gn-gen'):
            self.gn_gen(depot_tools_dir)
        with teamcity_step(self, 'Ninja', 'ninja'):
            self.build_targets(depot_tools_dir)
        if self.platform == 'android':
            with teamcity_step(self, 'Publish apks', 'publish_apks') as tac:
                self.publish_apks(tac)
        else:
            with teamcity_step(self, 'Collect artifacts', 'collect_artifacts') as tac:
                self.collect_artifacts(depot_tools_dir, tac)
        with teamcity_step(self, 'Tag build', 'tag-build') as tac:
            self.tag_build(tac)

    def on_finish(self, prev_status, status):
        self.clean_files()
        super(BuildChromiumSnapshot, self).on_finish(prev_status, status)

    def on_break(self, prev_status, status):
        self.clean_files()
        super(BuildChromiumSnapshot, self).on_break(prev_status, status)
