import collections
import contextlib
import json
import logging
import multiprocessing
import os
import stat
import tempfile

import jinja2

from sandbox.common import errors, utils
from sandbox.common.config import Registry
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox.projects.browser.builds.ci import BrowserBuildStatus
from sandbox.projects.browser.common import binary_tasks
from sandbox.projects.browser.common.contextmanagers import TmpdirContext
from sandbox.projects.browser.common.depot_tools import DepotToolsEnvironment
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment
from sandbox.projects.browser.common.keychain import MacKeychainEnvironment
from sandbox.projects.common.teamcity import TeamcityArtifactsContext, TeamcityServiceMessagesLog

from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess


def _host_ncpu(host):
    return host.info['system'].get('ncpu', 0)


class BrowserIsolateOutput(sdk2.Resource):
    isolate_name = sdk2.Attributes.String('Isolate name', required=True)
    ttl = 3


class BrowserIsolateXmlReport(BrowserIsolateOutput):
    pass


class BrowserIsolateJsonSummary(BrowserIsolateOutput):
    pass


class BrowserIsolateLog(BrowserIsolateOutput):
    pass


class BrowserIsolateArtifacts(BrowserIsolateOutput):
    pass


class RunBrowserIsolates(binary_tasks.CrossPlatformBinaryTaskMixin, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.BROWSER
        disk_space = 5 * 1024
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 80 * 60

        with sdk2.parameters.Group('General settings') as general_settings:
            with sdk2.parameters.String('Platform') as platform:
                platform.values.android = platform.Value('android')
                platform.values.ios = platform.Value('ios')
                platform.values.linux = platform.Value('linux')
                platform.values.mac = platform.Value('mac')
                platform.values.win = platform.Value('win')

            params_json = sdk2.parameters.String('params.json url')

            params = sdk2.parameters.JSON('Isolate parameters')

            btr_requirement = sdk2.parameters.String('BTR requirement', required=True)

            extra_requirements = sdk2.parameters.String('Extra python requirements')

            depot_tools_revision = sdk2.parameters.String('Depot tools revision')

            use_ram_drive = sdk2.parameters.Bool('Use RAM drive for temp files')

            _binary_task_params = binary_tasks.cross_platform_binary_task_parameters()

        with sdk2.parameters.Group('Debug settings') as debug_settings:
            debug_mode = sdk2.parameters.Bool('Enable debug mode (suspend on finish)')
            publish_isolate_output = sdk2.parameters.Bool('Publish isolate output as resources')

        with sdk2.parameters.Group('Vault items') as vault_items:
            yav_token_vault = sdk2.parameters.String('Vault item with yav token',
                                                     default='robot-browser-btr_yav_token')

            with platform.value['mac']:
                default_keychain_password_vault = sdk2.parameters.String(
                    'Vault item with default keychain password',
                    default=MacKeychainEnvironment.DEFAULT_KEYCHAIN_PASSWORD_VAULT)
                browser_keychain_content_vault = sdk2.parameters.String(
                    'Vault item with browser keychain content',
                    default=MacKeychainEnvironment.BROWSER_KEYCHAIN_CONTENT_VAULT)
                safe_storage_secret_vault = sdk2.parameters.String(
                    'Vault item with safe storage secret',
                    default=MacKeychainEnvironment.SAFE_STORAGE_SECRET_VAULT)

        with sdk2.parameters.Group('Task settings') as task_settings:
            additional_tags = sdk2.parameters.CustomClientTags(
                'Additional client tags (will be intersected with default)'
            )

            may_increase_requirements = sdk2.parameters.Bool(
                'Increase requirements (CPU, RAM) if they are lower than default for specified platform',
                default=True)

        _container = sdk2.parameters.Container('LXC container', default=None, required=False)

    DEFAULT_CORES = {
        'android': 15,
        'ios': 4,
        'linux': 15,
        'mac': 4,
        'win': 16,
    }
    DEFAULT_RAM = {
        'android': 31 * 1024,
        'ios': 16 * 1024,
        'linux': 31 * 1024,
        'mac': 16 * 1024,
        'win': 32 * 1024,
    }

    @sdk2.footer()
    def footer(self):
        teamcity_log_resource = TeamcityServiceMessagesLog.find(task=self).first()
        teamcity_log_url = teamcity_log_resource.http_proxy if teamcity_log_resource else None

        resources = collections.defaultdict(list)
        if self.Parameters.publish_isolate_output:
            resource_classes = (
                BrowserIsolateXmlReport,
                BrowserIsolateJsonSummary,
                BrowserIsolateLog,
                BrowserIsolateArtifacts,
            )
            for resource_cls in resource_classes:
                # limit(0) returns all resources, assuming their number is not too large (< 3000).
                for resource in resource_cls.find(task=self).limit(0):
                    resources[resource.isolate_name].append(resource)

        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
        return env.get_template('footer.html').render(
            teamcity_log_url=teamcity_log_url,
            resources=resources,
        )

    @sdk2.header()
    def header(self):
        status_resource = BrowserBuildStatus.find(task=self).first()
        if status_resource:
            return '<a href="{}/index.html" style="font-size: 2em">Isolate results</a>'.format(
                status_resource.http_proxy
            )

    @utils.singleton_property
    def status_resource_dir(self):
        status_resource_dir = tempfile.mkdtemp(dir=str(self.path()))
        os.chmod(status_resource_dir, 0o755)
        with open(os.path.join(status_resource_dir, 'empty'), 'w') as f:
            f.write('empty file to avoid error on creating empty resource')

        return status_resource_dir

    def on_enqueue(self):
        if bool(self.Parameters.params_json) == bool(self.Parameters.params):
            raise errors.TaskError('Only one of `params_json` and `params` should be provided')
        super(RunBrowserIsolates, self).on_enqueue()

        if self.Parameters.may_increase_requirements:
            default_cores = self.DEFAULT_CORES.get(self.Parameters.platform, 0)
            self.Requirements.cores = max(self.Requirements.cores, default_cores)

            default_ram = self.DEFAULT_RAM.get(self.Parameters.platform, 0)
            self.Requirements.ram = max(self.Requirements.ram, default_ram)

        if self.Parameters.additional_tags:
            self.Requirements.client_tags &= self.Parameters.additional_tags

        # TODO: remove `or self.Parameters.platform == 'linux'` when 22.3.* will be obsoleted (Jun 2022).
        if self.Parameters.use_ram_drive or self.Parameters.platform == 'linux':
            self.Requirements.ramdrive = ctm.RamDrive(type=ctm.RamDriveType.TMPFS, size=6 * 1024, path=None)

    def on_prepare(self):
        super(RunBrowserIsolates, self).on_prepare()
        DepotToolsEnvironment(self.Parameters.depot_tools_revision).prepare()
        if self.Parameters.platform == 'mac':
            MacKeychainEnvironment(
                self.Parameters.default_keychain_password_vault, self.Parameters.browser_keychain_content_vault,
                self.Parameters.safe_storage_secret_vault).prepare()

    def get_params_json(self):
        if self.Parameters.params_json:
            return self.Parameters.params_json

        params_json = self.path('params.json')
        with params_json.open('wb') as params_file:
            json.dump(self.Parameters.params, params_file)
        return str(params_json)

    def script_cwd(self):
        return self.path()

    def btr_env(self):
        cpu_per_slot = multiprocessing.cpu_count() / Registry().client.max_job_slots
        # Multislot clients have |max_job_slots| equal to their number of cpu cores, so each slot can occupy one
        # CPU. But such clients may run tasks that require, for example, 5 CPU, in that case client would dedicate
        # 5 slots to such task.
        # On the other hand client may contain only one slot with say 12 CPU, and task that requires 5 CPU would
        # occupy slot with 12 CPU, so let it use all of them, to make it perform faster.
        number_of_cores_to_use = max(self.Requirements.cores, cpu_per_slot)

        env = os.environ.copy()
        env.update({
            'CIPD_CACHE_DIR': SandboxEnvironment.exclusive_build_cache_dir('cipd_cache'),
            'GCLIENT_RUNHOOKS_CACHE_DIR': SandboxEnvironment.exclusive_build_cache_dir('gclient_hooks'),
            'SLOT_CPU_NUMBER': str(number_of_cores_to_use),
            'SLOT_RAM_MB': str(self.Requirements.ram),
            'TEAMCITY_VERSION': '2017.1.1',  # Force publishing Teamcity artifacts in btr_fire_many.
            'YAV_TOKEN': sdk2.Vault.data(self.Parameters.yav_token_vault),
            'SANDBOX_CACHE_DIR': SandboxEnvironment.exclusive_build_cache_dir('common_cache'),
        })
        if self.Parameters.platform == 'android':
            env.update({
                # get-deps is a fast Android SDK downloader.
                # https://bitbucket.browser.yandex-team.ru/projects/MBROINFRA/repos/get-deps
                'GET_DEPS_CACHE_DIR': os.path.join(SandboxEnvironment.build_cache_dir, 'get-deps'),
                'GRADLE_USER_HOME': SandboxEnvironment.exclusive_build_cache_dir('gradle'),
            })
        return env

    @contextlib.contextmanager
    def redirect_tmpdir(self):
        if self.ramdrive is not None:
            with TmpdirContext(str(self.ramdrive.path)):
                yield
        else:
            yield

    @property
    def script_log_path(self):
        return self.log_path('btr_fire_many.out.log')

    def ensure_permissions(self, path):
        """
        Ensure that sandbox can publish file as resource.
        zomb-sandbox (in sandbox group) and skynet (in other group) users should have permissions
        to find and read file.
        TODO: remove it when SANDBOX-5499 will be done
        """
        while path != str(self.path()):
            os.chmod(path, os.stat(path).st_mode | stat.S_IRGRP | stat.S_IXGRP | stat.S_IROTH | stat.S_IXOTH)
            path = os.path.dirname(path)

    def publish_isolate_output_resource(self, path, resource_cls, description, isolate_name):
        if not os.path.exists(path):
            logging.warn('No such file: %s', path)
            return
        self.ensure_permissions(path)
        resource = resource_cls(self, description, path, isolate_name=isolate_name)
        sdk2.ResourceData(resource).ready()

    def publish_resources(self, results_json_path):
        if not results_json_path.exists():
            return
        with results_json_path.open('rb') as results_json:
            results = json.load(results_json)
        for isolate_name, isolate_results in results.iteritems():
            if isolate_results.get('xml_report_path'):
                self.publish_isolate_output_resource(isolate_results['xml_report_path'], BrowserIsolateXmlReport,
                                                     'GTest xml report', isolate_name)
            if isolate_results.get('json_summary_path'):
                self.publish_isolate_output_resource(isolate_results['json_summary_path'], BrowserIsolateJsonSummary,
                                                     'Test launcher summary', isolate_name)
            if isolate_results.get('log_file_path'):
                self.publish_isolate_output_resource(isolate_results['log_file_path'], BrowserIsolateLog,
                                                     'Isolate log', isolate_name)
            if isolate_results.get('artifacts_archive_path'):
                self.publish_isolate_output_resource(isolate_results['artifacts_archive_path'],
                                                     BrowserIsolateArtifacts,  'Artifacts archive', isolate_name)

    def run_isolate(self):
        cmd = ['btr_fire_many', '-v', 'local', self.get_params_json()]
        if self.Parameters.publish_isolate_output:
            results_json_path = self.path('results.json')
            cmd += ['--results-json-path', str(results_json_path)]
        else:
            results_json_path = None

        cmd += ['--tests-report-dir', self.status_resource_dir]

        env = self.btr_env()
        secret_tokens = [env['YAV_TOKEN']]

        with TeamcityArtifactsContext(self.path(), prepend_time=False,
                                      secret_tokens=secret_tokens) as artifacts_context:
            exit_code = subprocess.call(
                cmd, cwd=str(self.script_cwd()), env=env,
                stdout=artifacts_context.output, stderr=artifacts_context.output)

        if self.Parameters.publish_isolate_output:
            self.publish_resources(results_json_path)

        build_status_resource = BrowserBuildStatus(
            self, 'Tests results report', self.status_resource_dir)
        sdk2.ResourceData(build_status_resource).ready()

        if exit_code != 0:
            raise errors.TaskFailure('btr_fire_many exited with {}'.format(exit_code))

    def on_execute(self):
        os.environ['PYTHONDONTWRITEBYTECODE'] = '1'  # TODO: temporary fix for DEVTOOLSSUPPORT-15014
        hpe = HermeticPythonEnvironment(
            python_version='3.9.7',
            pip_version='20.3.4',
            packages=[self.Parameters.btr_requirement] + self.Parameters.extra_requirements.split(),
        )
        with hpe, self.redirect_tmpdir():
            try:
                self.run_isolate()
            finally:
                if self.Parameters.debug_mode:
                    self.suspend()
