import logging
import os
import shlex

import jinja2
import yaml

import sandbox.common.types.client as ctc
from sandbox.common import utils

from sandbox.projects.browser.common.RunBrowserScript import RunBrowserScript
from sandbox.projects.browser.common.bitbucket import (
    BitBucket, ResourceNotFound, DEFAULT_BITBUCKET_URL, TESTING_BITBUCKET_URL)

from sandbox import sdk2
from sandbox.sdk2.service_resources import TaskLogs


class RunBrowserLiteChecks(RunBrowserScript):
    class Requirements(RunBrowserScript.Requirements):
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        disk_space = 2 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(RunBrowserScript.Parameters):
        kill_timeout = 2 * 60 * 60

        with sdk2.parameters.Group('General settings') as general_settings:
            teamcity_build_id = sdk2.parameters.String('ID of parent teamcity build', required=True)

            extra_args = sdk2.parameters.String('Extra script arguments')

        with sdk2.parameters.Group('Credentials') as credentials_group:
            yav_token_vault = sdk2.parameters.String(
                'Vault item with yav token', default='robot-browser-infra_yav_token')
            bitbucket_token_vault = sdk2.parameters.String(
                'Vault item with oauth token for bitbucket', default='robot-bro-infra_bitbucket_token')

    git_browser_sparse_checkout_paths = [
        'src/build/config/*',
        'src/build/yandex/*',
        'src/.DEPS.snapshot',
        '.gclient',
        '.gclient_default',
        'requirements.txt',
    ]
    secret_envvars = RunBrowserScript.secret_envvars + ('YAV_TOKEN',)
    script_module = 'yandex.ci.run_lite_checks'

    @utils.singleton_property
    def bb(self):
        if self.Parameters.use_test_bitbucket:
            bitbucket_url = TESTING_BITBUCKET_URL
        else:
            bitbucket_url = DEFAULT_BITBUCKET_URL
        return BitBucket(bitbucket_url, 'x-oauth-token',
                         sdk2.Vault.data(self.Parameters.bitbucket_token_vault))

    @utils.singleton_property
    def bitbucket_browser_sparse_checkout_paths(self):
        try:
            sparse_checkout_file = self.bb.load_file(
                'STARDUST', 'browser',
                'src/build/yandex/ci/sparse_checkout_paths.yaml',
                at=self.checkout_ref)
            return yaml.safe_load(sparse_checkout_file)['RunBrowserLiteChecks']
        except ResourceNotFound:
            logging.warn('src/build/yandex/ci/sparse_checkout_paths.yaml not found! Will use git checkout')
            return None

    @sdk2.footer()
    def footer(self):
        logs_resource = TaskLogs.find(task=self).first()
        logs_url = logs_resource.http_proxy if logs_resource else None

        teamcity_log_resource = self._find_script_tc_log()
        teamcity_log_url = teamcity_log_resource.http_proxy if teamcity_log_resource else None

        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
        return env.get_template('footer.html').render(
            logs_url=logs_url,
            teamcity_log_url=teamcity_log_url,
        )

    def script_extra_env(self):
        env = super(RunBrowserLiteChecks, self).script_extra_env()
        env.update({
            'TEAMCITY_BUILD_ID': self.Parameters.teamcity_build_id,
            'YAV_TOKEN': sdk2.Vault.data(self.Parameters.yav_token_vault),
        })
        return env

    def script_cmd(self, python_executable):
        cmd = [str(python_executable), '-m', self.script_module, self.platform]
        cmd += shlex.split(self.Parameters.extra_args)
        return cmd
