import logging
import os
import shlex
import tempfile

import jinja2
import yaml

import sandbox.common.types.client as ctc
from sandbox.common import utils

from sandbox.projects.browser.builds.ci import BrowserBuildStatus
from sandbox.projects.browser.common.bitbucket import (
    BitBucket, ResourceNotFound, DEFAULT_BITBUCKET_URL, TESTING_BITBUCKET_URL)
from sandbox.projects.browser.common.RunBrowserScript import RunBrowserScript
from sandbox.projects.browser.util.BrowserStopSandboxTasks import BrowserStopSandboxTasks
from sandbox.projects.browser.util.BrowserStopTeamcityBuilds import BrowserStopTeamcityBuilds
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2
from sandbox.sdk2.service_resources import TaskLogs


class RunBrowserTests(RunBrowserScript):
    class Requirements(RunBrowserScript.Requirements):
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        disk_space = 2 * 1024
        cores = 1
        environments = RunBrowserScript.Requirements.environments.default + (
            PipEnvironment('watchdog', '0.10.2'),
            # Version here is like src/third_party/jdk in Browser project (for comfort)
            # but it should be OK to upgrade if required.
            SandboxJavaJdkEnvironment('11.0.2'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(RunBrowserScript.Parameters):
        kill_timeout = 7 * 60 * 60

        with RunBrowserScript.Parameters.general_settings() as general_settings:
            teamcity_build_id = sdk2.parameters.String('ID of parent teamcity build', required=True)
            test_flavor = sdk2.parameters.String('Test flavor')
            sanitizer = sdk2.parameters.String('Sanitizer type')
            extra_args = sdk2.parameters.String('Extra script arguments')

        with sdk2.parameters.Group('Credentials') as credentials_group:
            yav_token_vault = sdk2.parameters.String(
                'Vault item with yav token', default='robot-browser-infra_yav_token')
            bitbucket_token_vault = sdk2.parameters.String(
                'Vault item with oauth token for bitbucket', default='robot-bro-infra_bitbucket_token')

        _container = sdk2.parameters.Container('LXC container', default=None, required=False)

    class Context(RunBrowserScript.Context):
        teamcity_build_ids = []
        sandbox_task_ids = []

    git_browser_sparse_checkout_paths = [
        'src/build/config/*',
        'src/build/find_depot_tools.py',
        'src/build/yandex/*',
        'src/build/yatest_runner/*',
        'src/chrome/VERSION',
        'src/testing/buildbot/gn_isolate_map.pyl',
        'src/tools/pixel_tests/*',
        'src/.DEPS.snapshot',
        '.gclient',
        '.gclient_default',
        'requirements.txt',
    ]
    secret_envvars = RunBrowserScript.secret_envvars + ('YAV_TOKEN',)

    @utils.singleton_property
    def bb(self):
        if self.Parameters.use_test_bitbucket:
            bitbucket_url = TESTING_BITBUCKET_URL
        else:
            bitbucket_url = DEFAULT_BITBUCKET_URL
        return BitBucket(bitbucket_url, 'x-oauth-token',
                         sdk2.Vault.data(self.Parameters.bitbucket_token_vault))

    @utils.singleton_property
    def bitbucket_browser_sparse_checkout_paths(self):
        try:
            sparse_checkout_file = self.bb.load_file(
                'STARDUST', 'browser',
                'src/build/yandex/ci/sparse_checkout_paths.yaml',
                at=self.checkout_ref)
            return yaml.safe_load(sparse_checkout_file)['RunBrowserTests']
        except ResourceNotFound:
            logging.warn('src/build/yandex/ci/sparse_checkout_paths.yaml not found! Will use git checkout')
            return None

    @utils.singleton_property
    def status_resource_dir(self):
        status_resource_dir = tempfile.mkdtemp(dir=str(self.path()))
        os.chmod(status_resource_dir, 0o755)
        with open(os.path.join(status_resource_dir, 'empty'), 'w') as f:
            f.write('empty file to avoid error on creating empty resource')

        return status_resource_dir

    @sdk2.footer()
    def footer(self):
        logs_resource = TaskLogs.find(task=self).first()
        logs_url = logs_resource.http_proxy if logs_resource else None

        status_resource = BrowserBuildStatus.find(task=self).first()
        status_url = status_resource.http_proxy + '/index.html' if status_resource else None

        teamcity_log_resource = self._find_script_tc_log()
        teamcity_log_url = teamcity_log_resource.http_proxy if teamcity_log_resource else None

        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
        return env.get_template('footer.html').render(
            logs_url=logs_url,
            status_url=status_url,
            teamcity_log_url=teamcity_log_url,
        )

    def script_cmd(self, python_executable):
        cmd = [str(python_executable), '-m', 'yandex.ci.run_tests', self.platform]
        if self.Parameters.test_flavor:
            cmd += ['--test-flavor', self.Parameters.test_flavor]
            # TODO(BYIN-12282): remove it when all actual branches will use `--test-flavor` instead of `--os-flavor`.
            cmd += ['--os-flavor', self.Parameters.test_flavor]
        if self.Parameters.sanitizer:
            cmd += ['--san', self.Parameters.sanitizer]
        cmd += ['--isolates-status-dir', self.status_resource_dir]
        cmd += shlex.split(self.Parameters.extra_args)
        return cmd

    @property
    def communication_file_path(self):
        return str(self.browser_path('communication.log'))

    def script_extra_env(self):
        env = super(RunBrowserTests, self).script_extra_env()
        env.update({
            'SANDBOX_COMMUNICATION_FILE': self.communication_file_path,
            'TEAMCITY_BUILD_ID': self.Parameters.teamcity_build_id,
            'YAV_TOKEN': sdk2.Vault.data(self.Parameters.yav_token_vault),
        })
        return env

    def run_script(self, python_executable):
        from sandbox.projects.browser.builds.ci.RunBrowserTests import sandbox_communication
        with sandbox_communication.CommunicationFileObserver(self.communication_file_path, self):
            super(RunBrowserTests, self).run_script(python_executable)

    def on_execute(self):
        status_resource = BrowserBuildStatus(self, 'Build status page', self.status_resource_dir)
        # That allows to view resource in browser even if it is NOT_READY.
        self.server.resource[status_resource.id].source()

        try:
            return super(RunBrowserTests, self).on_execute()
        finally:
            sdk2.ResourceData(status_resource).ready()

    def spawn_stopping_task(self):
        if self.Context.teamcity_build_ids:
            BrowserStopTeamcityBuilds(
                self,
                description='Stop swarmed builds',
                teamcity_build_ids=self.Context.teamcity_build_ids,
                cancel_comment='Parent build stopped.',
            ).enqueue()
        if self.Context.sandbox_task_ids:
            BrowserStopSandboxTasks(
                self,
                description='Stop swarmed tasks',
                sandbox_task_ids=self.Context.sandbox_task_ids,
            ).enqueue()

    def on_finish(self, prev_status, status):
        super(RunBrowserTests, self).on_finish(prev_status, status)
        self.spawn_stopping_task()

    def on_break(self, prev_status, status):
        super(RunBrowserTests, self).on_break(prev_status, status)
        self.spawn_stopping_task()
