import json
import os

from watchdog.observers import Observer
from watchdog.events import PatternMatchingEventHandler


class CommandExecutor(object):
    def __init__(self, command, task):
        self.command_params = json.loads(command)
        self.type = self.command_params.pop('type')
        self.task = task

    def record_teamcity_build(self, build_id):
        self.task.Context.teamcity_build_ids.append(build_id)
        self.task.Context.save()

    def record_sandbox_task(self, task_id):
        self.task.Context.sandbox_task_ids.append(task_id)
        self.task.Context.save()

    def execute(self):
        {
            'RECORD_TEAMCITY_BUILD': self.record_teamcity_build,
            'RECORD_SANDBOX_TASK': self.record_sandbox_task,
        }[self.type](**self.command_params)


class ModificationHandler(PatternMatchingEventHandler):
    def __init__(self, task, *args, **kwargs):
        super(ModificationHandler, self).__init__(*args, **kwargs)
        self.task = task
        self.file_position = 0

    def on_modified(self, event):
        with open(event.src_path, 'rb') as communication_file:
            communication_file.seek(self.file_position)
            for command in communication_file:
                CommandExecutor(command, self.task).execute()
            self.file_position = communication_file.tell()


class CommunicationFileObserver(object):
    def __init__(self, communication_file_path, task):
        self.observer = Observer()
        self.observer.schedule(ModificationHandler(task, patterns=[communication_file_path]),
                               os.path.dirname(communication_file_path))

    def __enter__(self):
        self.observer.start()

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.observer.stop()
        self.observer.join()
