import platform
import shutil

import sandbox.common.types.client as ctc
from sandbox.common.types import resource as ctr
from sandbox.common import utils

from sandbox.projects.browser.common import binary_tasks
from sandbox.projects.browser.common.contextmanagers import ExitStack
from sandbox.projects.browser.common.depot_tools import DepotToolsEnvironment
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.git.git_cli import GitCli
from sandbox.projects.browser.common.hermetic_xcode import HermeticXcodeEnvironment

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessRegistry, subprocess


class BrowserClang(sdk2.Resource):
    """
    Gzipped tar archive with clang distribution used by browser builds.
    """
    any_arch = False
    auto_backup = True
    executable = False
    restart_policy = ctr.RestartPolicy.IGNORE
    browser_commit = sdk2.Attributes.String('Browser commit hash this distro was built at', required=True)
    platform = sdk2.Attributes.String('Platform', required=True)
    ttl = 356


class BuildBrowserClang(binary_tasks.CrossPlatformBinaryTaskMixin, sdk2.Task):
    """
    Build clang using package.py & build.py scripts in browser repo and publish archive with clang distribution
    as resource.
    """
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.BROWSER
        cores = 12
        disk_space = 40 * 1024
        environments = (
            GitEnvironment('2.24.1'),
        )
        ram = 15 * 1024

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 1
        kill_timeout = 5 * 60 * 60

        with sdk2.parameters.String('Target platform') as platform:
            platform.values.linux = platform.Value('linux', default=True)
            platform.values.mac = platform.Value('mac')
            platform.values.mac_arm64 = platform.Value('mac-arm64')
            platform.values.win = platform.Value('win')

        with sdk2.parameters.Group('Repositories settings') as repositories_settings:
            branch = sdk2.parameters.String('Branch to checkout browser repo on', default='master')
            commit = sdk2.parameters.String('Commit to checkout browser repo on')

            depot_tools_revision = sdk2.parameters.String('Depot tools revision', default='master')
            llvm_revision = sdk2.parameters.String('LLVM revision is used to scout final archive', required=True)

        with sdk2.parameters.Group('General settings') as general_settings:
            additional_tags = sdk2.parameters.CustomClientTags(
                'Additional client tags to select host platform')

            suspend_before_finish = sdk2.parameters.Bool('Suspend after building & publishing resource', default=False)
            lxc_container_resource_id = sdk2.parameters.Integer(
                'ID of LXC container resource to use on linux hosts', default=None, required=False)

            _binary_task_params = binary_tasks.cross_platform_binary_task_parameters()

    OVERRIDES = {
        'linux': {
            'cores': 31,
            'ram': 63,
        },
        'win': {
            'cores': 15,
            'ram': 31,
        },
    }

    @property
    def browser_sparse_checkout_paths(self):
        return [
            '.gclient_default',
            'src/.DEPS.snapshot',
            'src/buildtools/third_party/eu-strip',
            'src/build',
            'src/tools/clang',
        ]

    def on_enqueue(self):
        if self.Parameters.platform in self.OVERRIDES:
            self.Requirements.cores = max(
                self.Requirements.cores, self.OVERRIDES[self.Parameters.platform].get('cores', 0))
            self.Requirements.ram = max(
                self.Requirements.ram, self.OVERRIDES[self.Parameters.platform].get('ram', 0))

        if self.Parameters.additional_tags:
            self.Requirements.client_tags = self.Requirements.client_tags & self.Parameters.additional_tags

        if self.Parameters.platform == 'linux' and self.Parameters.lxc_container_resource_id:
            self.Requirements.container_resource = self.Parameters.lxc_container_resource_id

        super(BuildBrowserClang, self).on_enqueue()

    def repo_path(self, *args):
        return self.path('browser', *args)

    @property
    def package_script(self):
        return str(self.repo_path('src', 'tools', 'clang', 'scripts', 'package.py'))

    @property
    def clang_archive_name(self):
        return 'clang-{}.tgz'.format(self.Parameters.llvm_revision)

    @property
    def clang_archive_path(self):
        return self.repo_path(self.clang_archive_name)

    @utils.singleton_property
    def custom_logs(self):
        custom_log_resource = sdk2.service_resources.TaskCustomLogs(self, "Clang build logs", "clang_build_logs")
        custom_logs = sdk2.ResourceData(custom_log_resource)
        custom_logs.path.mkdir(0o755, parents=True, exist_ok=True)
        return custom_logs

    def checkout(self):
        repositories.Stardust.browser(filter_branches=False).clone(
            str(self.repo_path()), self.Parameters.branch, self.Parameters.commit,
            sparse_checkout_paths=self.browser_sparse_checkout_paths)

    def clean_files(self):
        # Remove repository manually, as Sandbox does it too slowly
        # (it iterates through all files and check if file is resource).
        # And distclang fetches & builds LLVM project that leads to lot of files.
        shutil.rmtree(str(self.repo_path()), ignore_errors=True)

    def provide_depot_tools(self):
        depot_tools_env = DepotToolsEnvironment(revision=self.Parameters.depot_tools_revision)
        depot_tools_env.prepare()
        return depot_tools_env.depot_tools_folder

    def provide_win_toolchain(self, python3_path, win_toolchain_log):
        toolchain_script = self.repo_path('src', 'build', 'vs_toolchain.py')
        cmd = [python3_path, str(toolchain_script), 'update', '--force']

        with ProcessRegistry:
            with open(str(win_toolchain_log), 'w') as log:
                subprocess.check_call(cmd, cwd=str(self.repo_path()), stdout=log, stderr=log)

    def gclient_sync(self, depot_tools_dir, gclient_sync_log):
        gclient_path = str(depot_tools_dir.joinpath('gclient.bat' if self.Parameters.platform == 'win' else 'gclient'))
        deps = {
            'linux': 'android',
        }.get(self.Parameters.platform, self.Parameters.platform)
        cmd = [gclient_path, 'sync', '--verbose', '--force', '--ignore_locks', '--deps={}'.format(deps), '--nohooks']
        with ProcessRegistry:
            with open(str(gclient_sync_log), 'w') as log:
                subprocess.check_call(cmd, cwd=str(self.repo_path()), stdout=log, stderr=log)

    def build_clang(self, python3_path, build_log):
        cmd = [python3_path, self.package_script, '--clang-only']
        if self.Parameters.platform == 'mac_arm64':
            cmd.append('--build-mac-arm')

        with ProcessRegistry:
            with open(str(build_log), 'w') as log:
                subprocess.check_call(cmd, cwd=str(self.repo_path()), stdout=log, stderr=log)

    def on_prepare(self):
        if self.Parameters.platform in ('mac', 'mac_arm64'):
            hermetic_xcode = HermeticXcodeEnvironment('13.4.1')
            hermetic_xcode.prepare()

    def on_execute(self):
        with ExitStack() as exit_stack:
            try:
                exit_stack.callback(self.clean_files)
                self.checkout()
                depot_tools_dir = self.provide_depot_tools()
                # Run 'gclient sync' without hooks to fetch deps line android ndk to build android runtime libs or
                # depot_tools to be able to fetch win toolchain to.
                self.gclient_sync(depot_tools_dir, self.custom_logs.path.joinpath('gclient_sync.log'))

                python3_path = str(depot_tools_dir.joinpath(
                    'python3.bat' if platform.system() == 'Windows' else 'python3'))

                if self.Parameters.platform == 'win':
                    self.provide_win_toolchain(python3_path, self.custom_logs.path.joinpath('win_toolchain.log'))

                self.build_clang(python3_path, self.custom_logs.path.joinpath('package.log'))

                browser_commit = self.Parameters.commit
                if not browser_commit:
                    git_repo = GitCli(str(self.repo_path()))
                    browser_commit = git_repo.rev_parse('HEAD').strip()
                clang_archive = BrowserClang(
                    self, 'Browser clang distribution in gzipped tar archive', self.clang_archive_name)
                clang_archive.browser_commit = browser_commit
                shutil.copy(str(self.clang_archive_path), str(clang_archive.path))
                sdk2.ResourceData(clang_archive).ready()
            finally:
                if self.Parameters.suspend_before_finish:
                    self.suspend()
