import contextlib
import multiprocessing
import os
import shutil

from sandbox.common.config import Registry
from sandbox.common import errors
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.common.types import resource as ctr

from sandbox.projects.browser.builds.compiler.BuildDistclangTests.gtest_xml_reader import GTestXmlReader
from sandbox.projects.browser.common import binary_tasks
from sandbox.projects.browser.common.contextmanagers import ExitStack
from sandbox.projects.browser.common.depot_tools import DepotToolsEnvironment
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.hermetic_xcode import HermeticXcodeEnvironment
from sandbox.projects.browser.common.teamcity_step import teamcity_step

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessRegistry, subprocess


class DistclangReleasePackage(sdk2.Resource):
    """
    Directory with Distclang .deb and .changes files.
    """
    any_arch = False
    auto_backup = True
    executable = False
    restart_policy = ctr.RestartPolicy.IGNORE
    version = sdk2.Attributes.String('Version', required=True)
    ttl = 356


class LLVMBuildLog(sdk2.Resource):
    """
    CMake build log of LLVM build.
    """
    any_arch = True
    auto_backup = False
    executable = False
    restart_policy = ctr.RestartPolicy.IGNORE
    flavor = sdk2.Attributes.String('Flavor', required=True)
    ttl = 90


class BuildDistclangTests(binary_tasks.CrossPlatformBinaryTaskMixin, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        cores = 4
        client_tags = ctc.Tag.BROWSER
        disk_space = 40 * 1024  # LLVM is built while building Distclang
        dns = ctm.DnsType.DNS64
        environments = (
            GitEnvironment('2.24.1'),
        )
        ram = 16 * 1024

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 2
        kill_timeout = 5 * 60 * 60

        with sdk2.parameters.Group('Repositories settings') as repositories_settings:
            branch = sdk2.parameters.String('Branch to checkout on', default='browser')
            commit = sdk2.parameters.String('Commit to checkout on')

            depot_tools_revision = sdk2.parameters.String('Depot tools revision', default='master')

        with sdk2.parameters.Group('General settings') as general_settings:
            with sdk2.parameters.String('Target platform') as platform:
                platform.values.linux = platform.Value('linux', default=True)
                platform.values.mac = platform.Value('mac')
                platform.values.mac_arm64 = platform.Value('mac-arm64')
                platform.values.win = platform.Value('win')

            additional_tags = sdk2.parameters.CustomClientTags(
                'Additional client tags to select host platform')

            suspend_before_finish = sdk2.parameters.Bool('Suspend after building & running tests', default=False)

            lxc_container_resource_id = sdk2.parameters.Integer(
                'ID of LXC container resource to use on linux hosts', default=None, required=False)

            _binary_task_params = binary_tasks.cross_platform_binary_task_parameters()

    DISK_SPACE_OVERRIDE = {
        'win': 80 * 1024,  # On win hosts we build LLVM twice: debug & release flavors, so double required space
    }
    CORES_OVERRIDE = {
        'linux': 15,
        'win': 15,
    }

    class Context(sdk2.Context):
        build_steps_statuses = []
        tests_skipped = dict()
        tests_failed = dict()
        tests_succeeded = dict()

    @property
    def templates_path(self):
        return os.path.dirname(os.path.abspath(__file__))

    @sdk2.report(title='Test results', label='test_results')
    def tests_report(self):
        import jinja2
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(self.templates_path), extensions=['jinja2.ext.do'])
        return env.get_template('test_report.html').render({
            'successful_tests': sorted(self.Context.tests_succeeded.keys()),
            'failed_tests': sorted(self.Context.tests_failed.items()),
            'skipped_tests': sorted(self.Context.tests_skipped.keys()),
        })

    @sdk2.footer(title='Build steps')
    def footer(self):
        import jinja2
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(self.templates_path), extensions=['jinja2.ext.do'])
        build_steps = []
        for name, status in self.Context.build_steps_statuses:
            status = {
                None: ('status_executing', 'EXECUTING'),
                False: ('status_exception', 'FAILURE'),
                True: ('status_success', 'SUCCESS'),
            }[status]
            build_steps.append((name, status))
        return env.get_template('footer.html').render({
            'build_steps': build_steps,
        })

    def on_enqueue(self):
        self.Requirements.disk_space = max(
            self.Requirements.disk_space, self.DISK_SPACE_OVERRIDE.get(self.Parameters.platform, 0))
        self.Requirements.cores = max(
            self.Requirements.cores, self.CORES_OVERRIDE.get(self.Parameters.platform, 0))

        if self.Parameters.additional_tags:
            self.Requirements.client_tags = self.Requirements.client_tags & self.Parameters.additional_tags

        if self.Parameters.platform == 'linux' and self.Parameters.lxc_container_resource_id:
            self.Requirements.container_resource = self.Parameters.lxc_container_resource_id

        super(BuildDistclangTests, self).on_enqueue()

    def repo_path(self, *args):
        return self.path('distclang', *args)

    def checkout_repo(self):
        repositories.DC.distclang(filter_branches=False).clone(
            str(self.repo_path()), self.Parameters.branch, self.Parameters.commit)

    def clean_files(self):
        # Remove repository manually, as Sandbox does it too slowly
        # (it iterates through all files and check if file is resource).
        shutil.rmtree(str(self.repo_path()), ignore_errors=True)

    def report_results(self):
        failure_messages = [
            'Build step {} has failed'.format(name) for
            name, succeeded in self.Context.build_steps_statuses if not succeeded
        ]
        if failure_messages:
            raise errors.TaskFailure('\n'.join(failure_messages))

    def collect_test_results(self, test_binary):
        gtest_xml_file = 'tests_results_{}.xml'.format(test_binary)
        abs_path = self.repo_path(gtest_xml_file)
        reader = GTestXmlReader(str(abs_path))
        passed_tests, failed_tests, skipped_tests = reader.get_tests()
        self.Context.tests_succeeded.update(passed_tests)
        self.Context.tests_failed.update(failed_tests)
        self.Context.tests_skipped.update(skipped_tests)

    def call_integrate(self, python3_executable, log, arguments):
        cmd = [python3_executable, str(self.repo_path('build', 'integrate.py'))] + arguments

        env = os.environ.copy()
        cpu_per_slot = multiprocessing.cpu_count() / Registry().client.max_job_slots
        # Multislot clients have |max_job_slots| equal to their number of cpu cores, so each slot can occupy one
        # CPU. But such clients may run tasks that require, for example, 5 CPU, in that case client would dedicate
        # 5 cores to such task.
        # On the other hand client may contain only one slot with say 12 CPU, and task that requires 5 CPU would
        # occupy slot with 12 CPU, so let it use all of them, to make it perform faster.
        number_of_cores_to_use = max(self.Requirements.cores, cpu_per_slot)
        env.update({
            'PATH': os.path.dirname(python3_executable) + os.path.pathsep + env['PATH'],
            'SLOT_CPU_NUMBER': str(number_of_cores_to_use),
            'SLOT_RAM_MB': str(self.Requirements.ram),
        })
        with ProcessRegistry:
            subprocess.check_call(cmd, stdout=log, stderr=subprocess.STDOUT, cwd=str(self.repo_path()), env=env)

    @contextlib.contextmanager
    def build_step(self, name, log_name, test_binary=None):
        self.Context.build_steps_statuses.append((name, None))
        self.Context.save()
        with teamcity_step(self, name, log_name) as tac:
            try:
                yield tac.output
            except subprocess.CalledProcessError:
                # Do not re-raise this exception to let other steps run
                self.Context.build_steps_statuses[-1] = (name, False)
            else:
                self.Context.build_steps_statuses[-1] = (name, True)
            finally:
                if test_binary is not None:
                    self.collect_test_results(test_binary)
            self.Context.save()

    def provide_depot_tools(self):
        depot_tools_env = DepotToolsEnvironment(revision=self.Parameters.depot_tools_revision)
        depot_tools_env.prepare()
        return str(depot_tools_env.depot_tools_folder)

    def python3_path(self, depot_tools_dir):
        python3_bin_reldir_file = os.path.join(depot_tools_dir, 'python3_bin_reldir.txt')
        with open(python3_bin_reldir_file, 'rb') as reldir_file:
            return os.path.join(depot_tools_dir, reldir_file.read().strip())

    def on_prepare(self):
        if self.Parameters.platform in ('mac', 'mac-arm64'):
            hermetic_xcode = HermeticXcodeEnvironment('12.4')
            hermetic_xcode.prepare()

    def checkout_and_sync(self, exit_stack):
        with teamcity_step(self, 'Provide depot tools', 'provide-depot-tools'):
            depot_tools_dir = self.provide_depot_tools()
            python3_path = self.python3_path(depot_tools_dir)
            # Also save direct path to python3 binary to call distclang integrate script with it
            python3_binary_name = 'python3.exe' if self.Parameters.platform == 'win' else 'python3'
            python3_binary = os.path.join(python3_path, python3_binary_name)

        with teamcity_step(self, 'Checkout distclang repo', 'distclang-checkout'):
            self.checkout_repo()
            exit_stack.callback(self.clean_files)

        with teamcity_step(self, 'Sync distclang dependencies', 'sync') as tac:
            self.call_integrate(python3_binary, tac.output, ['sync'])

        return python3_binary

    def publish_deb_package(self, python3_binary):
        version_script = self.repo_path('build', 'version.py')
        version = subprocess.check_output([python3_binary, str(version_script)],
                                          cwd=str(self.repo_path())).split()[0]
        resource = DistclangReleasePackage(self, 'Distclang debian package', 'distclang_release')
        resource.version = version
        data = sdk2.ResourceData(resource)
        data.path.mkdir(0o755, parents=True, exist_ok=True)
        for file_path in self.repo_path('out', 'Release').glob('dist-clang*'):
            shutil.copy(str(file_path), str(data.path))
        data.ready()

    def publish_llvm_build_log(self, flavor):
        log_name = 'llvm_build_{}.log'.format(flavor)
        log_file = self.repo_path('out', log_name)
        llvm_log = LLVMBuildLog(self, "LLVM build log", log_name)
        llvm_log.flavor = flavor
        llvm_log.path.write_bytes(log_file.read_bytes())

    def on_execute(self):
        with ExitStack() as exit_stack:
            exit_stack.callback(self.report_results)
            python3_binary = self.checkout_and_sync(exit_stack)
            build_command_arguments = ['build']
            if self.Parameters.platform == 'mac-arm64':
                build_command_arguments.append('--arm64-mac')

            with self.build_step('Build Tests', 'build-tests') as output:
                self.call_integrate(python3_binary, output, build_command_arguments + ['Test'])

                # MSVC C Runtime is different for debug and release builds. So to build Test build we LLVM is
                # being built second time and linked against debug runtime library. So publish log of that build.
                self.publish_llvm_build_log(flavor=('DEBUG' if self.Parameters.platform == 'win' else 'RELEASE'))

            # There're no arm64 mac clients for now, so we're unable to run tests built for arm64
            with self.build_step('Run tests', 'run-tests', test_binary='unit_tests') as output:
                self.call_integrate(python3_binary, output, ['test', '--publish-xml'])

            with self.build_step('Build Portable', 'build-portable') as output:
                self.call_integrate(python3_binary, output, build_command_arguments + ['Portable'])
                if self.Parameters.platform == 'win':
                    self.publish_llvm_build_log(flavor='RELEASE')

            with self.build_step('Run perf tests', 'run-perf-tests', test_binary='perf_tests') as output:
                self.call_integrate(python3_binary, output, ['test', '--perf', '--publish-xml'])

            if self.Parameters.platform == 'linux':
                with self.build_step('Build Release', 'build-release') as output:
                    self.call_integrate(python3_binary, output, build_command_arguments + ['Release'])
                    self.publish_deb_package(python3_binary)

            with self.build_step('Build Debug', 'build-debug') as output:
                self.call_integrate(python3_binary, output, build_command_arguments + ['Debug'])

            with teamcity_step(self, 'Publish artifacts', 'publish-artifacts') as tac:
                for artifacts_path in ('out/distclang/*.log', 'out/Release/dist-clang*', 'out/llvm_build_*.log'):
                    tac.logger.info("##teamcity[publishArtifacts '{repo_path}/{mask}']\n".format(
                        repo_path=str(self.repo_path()), mask=artifacts_path))

            if self.Parameters.suspend_before_finish:
                self.suspend()
