import os
import platform
import shutil

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.common.types import resource as ctr

from sandbox.projects.browser.common import binary_tasks
from sandbox.projects.browser.common.contextmanagers import ExitStack
from sandbox.projects.browser.common.depot_tools import DepotToolsEnvironment
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.git.git_cli import GitCli
from sandbox.projects.browser.common.hermetic_xcode import HermeticXcodeEnvironment
from sandbox.projects.browser.common.teamcity_step import teamcity_step

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess


BUILD_FLAVOR = 'Portable'


class Distclang(sdk2.Resource):
    """
    Directory with portable distclang executable files and their dynamic libs.
    """
    any_arch = False
    auto_backup = True
    executable = False
    restart_policy = ctr.RestartPolicy.IGNORE
    commit = sdk2.Attributes.String('Commit hash', required=True)
    platform = sdk2.Attributes.String('Platform', required=True)
    ttl = 356


class BuildPortableDistclang(binary_tasks.CrossPlatformBinaryTaskMixin, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        cores = 8
        client_tags = ctc.Tag.BROWSER
        disk_space = 40 * 1024  # LLVM is built while building Distclang
        dns = ctm.DnsType.DNS64
        environments = (
            GitEnvironment('2.24.1'),
        )
        ram = 15 * 1024

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 2
        kill_timeout = 5 * 60 * 60

        with sdk2.parameters.String('Target platform') as platform:
            platform.values.linux = platform.Value('linux', default=True)
            platform.values.mac = platform.Value('mac')
            platform.values.mac_arm64 = platform.Value('mac_arm64')
            platform.values.win = platform.Value('win')

        with sdk2.parameters.Group('Repositories settings') as repositories_settings:
            branch = sdk2.parameters.String('Branch to checkout Distclang repo on', default='browser')
            commit = sdk2.parameters.String('Commit to checkout Distclang repo on')

            depot_tools_revision = sdk2.parameters.String('Depot tools revision',
                                                          default='master')

        with sdk2.parameters.Group('General settings') as general_settings:
            additional_tags = sdk2.parameters.CustomClientTags(
                'Additional client tags to select host platform')

            suspend_before_finish = sdk2.parameters.Bool('Suspend after building & publishing resource', default=False)
            lxc_container_resource_id = sdk2.parameters.Integer(
                'ID of LXC container resource to use on linux hosts', default=None, required=False)

            _binary_task_params = binary_tasks.cross_platform_binary_task_parameters()

    CORES_OVERRIDE = {
        'linux': 15,
        'win': 15,
    }

    def on_enqueue(self):
        self.Requirements.cores = max(
            self.Requirements.cores, self.CORES_OVERRIDE.get(self.Parameters.platform, 0))

        if self.Parameters.additional_tags:
            self.Requirements.client_tags = self.Requirements.client_tags & self.Parameters.additional_tags

        if self.Parameters.platform == 'linux' and self.Parameters.lxc_container_resource_id:
            self.Requirements.container_resource = self.Parameters.lxc_container_resource_id

        super(BuildPortableDistclang, self).on_enqueue()

    def repo_path(self, *args):
        return self.path('distclang', *args)

    @property
    def files_to_publish(self):
        libraries_to_pack = ['libc++abi', 'libc++']
        exe_extension = ''

        if self.Parameters.platform == 'win':
            libraries_to_pack = []
            exe_extension = '.exe'
        else:
            lib_extension = '.so' if self.Parameters.platform == 'linux' else '.dylib'
            libraries_to_pack = [library + lib_extension for library in libraries_to_pack]

        return [
            'distclangd{}'.format(exe_extension),
            'distclang{}'.format(exe_extension),
            'distswift{}'.format(exe_extension),
            'populate_cache{}'.format(exe_extension),
            'stat.py',
        ] + libraries_to_pack

    def checkout(self):
        repositories.DC.distclang(filter_branches=False).clone(
            str(self.repo_path()), self.Parameters.branch, self.Parameters.commit)

    def clean_files(self):
        # Remove repository manually, as Sandbox does it too slowly
        # (it iterates through all files and check if file is resource).
        # And distclang fetches & builds LLVM project that leads to lot of files.
        shutil.rmtree(str(self.repo_path()), ignore_errors=True)

    def provide_depot_tools(self):
        depot_tools_env = DepotToolsEnvironment(revision=self.Parameters.depot_tools_revision)
        depot_tools_env.prepare()
        return str(depot_tools_env.depot_tools_folder)

    def python3_root(self, depot_tools_dir):
        python3_bin_reldir_file = os.path.join(depot_tools_dir, 'python3_bin_reldir.txt')
        with open(python3_bin_reldir_file, 'rb') as reldir_file:
            return os.path.join(depot_tools_dir, reldir_file.read().strip())

    def python3_path(self, depot_tools_dir):
        binary = 'python3.exe' if platform.system() == 'Windows' else 'python3'
        return os.path.join(self.python3_root(depot_tools_dir), binary)

    def on_prepare(self):
        if self.Parameters.platform in ('mac', 'mac_arm64'):
            hermetic_xcode = HermeticXcodeEnvironment('12.4')
            hermetic_xcode.prepare()

    def run_integrate(self, depot_tools_dir, logger_name, command):
        integrate_path = self.repo_path('build', 'integrate.py')
        env = os.environ.copy()
        env['PATH'] = os.pathsep.join([self.python3_root(depot_tools_dir), env['PATH']])
        with ProcessLog(self, logger=logger_name) as process_log:
            subprocess.check_call([self.python3_path(depot_tools_dir), str(integrate_path)] + command,
                                  stdout=process_log.stdout, stderr=process_log.stderr, cwd=str(self.repo_path()),
                                  env=env)

    def sync_dependencies(self, depot_tools_dir):
        self.run_integrate(depot_tools_dir, 'integrate-sync', ['sync'])

    def build_distclang(self, depot_tools_dir, is_arm64_mac):
        arguments = ['build', BUILD_FLAVOR]
        if is_arm64_mac:
            arguments.insert(1, '--arm64-mac')
        self.run_integrate(depot_tools_dir, 'integrate-build', arguments)

    def publish_distclang(self):
        build_directory = str(self.repo_path('out', BUILD_FLAVOR))
        resource = Distclang(self, 'Portable distclang', 'portable_distclang')
        current_commit = self.Parameters.commit
        if not current_commit:
            git_repo = GitCli(str(self.repo_path()))
            current_commit = git_repo.rev_parse('HEAD').strip()
        resource.commit = current_commit
        resource.platform = self.Parameters.platform

        data = sdk2.ResourceData(resource)
        data.path.mkdir(0o755, parents=True, exist_ok=True)
        for file_path in self.files_to_publish:
            shutil.copy(os.path.join(build_directory, file_path), str(data.path))
        data.ready()

    def on_execute(self):
        with ExitStack() as exit_stack:
            exit_stack.callback(self.clean_files)
            with teamcity_step(self, 'Checkout distclang', 'checkout-distclang'):
                self.checkout()
            with teamcity_step(self, 'Provide depot_tools', 'provide-depot-tools'):
                depot_tools_dir = self.provide_depot_tools()
            with teamcity_step(self, 'Sync dependencies', 'sync-dependencies'):
                self.sync_dependencies(depot_tools_dir)
            with teamcity_step(self, 'Build distclang', 'build-distclang'):
                self.build_distclang(depot_tools_dir, is_arm64_mac=self.Parameters.platform == 'mac_arm64')
            self.publish_distclang()

        if self.Parameters.suspend_before_finish:
            self.suspend()
