from datetime import date
import hashlib
import os
import platform

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.common import utils

from sandbox.projects.browser.common import binary_tasks
from sandbox.projects.browser.common.contextmanagers import TempfileContext
from sandbox.projects.browser.common.git import GitEnvironment
from sandbox.projects.browser.common.hermetic_xcode import HermeticXcodeEnvironment
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment
from sandbox.projects.browser.common.RunYinScript import RunYinScript
from sandbox.projects.browser.builds.compiler.BuildYakuza import YakuzaEnvironment
from sandbox.projects.browser.builds.compiler import LLVMToolchainEnvironment, CMakeEnvironment

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessRegistry, subprocess
from sandbox.sdk2.vcs.git import Git as SandboxGit


class BuildSwiftToolchain(binary_tasks.CrossPlatformBinaryTaskMixin, sdk2.Task):
    """
    Build and upload to s3 swift toolchain.

    Requires packages:
       libedit-dev
       libncurses5-dev
    """

    class Requirements(RunYinScript.Requirements):
        client_tags = ctc.Tag.BROWSER
        cores = 8
        ram = 15 * 1024
        disk_space = 60 * 1024
        dns = ctm.DnsType.DNS64
        environments = (
            GitEnvironment('2.24.1'),
            YakuzaEnvironment(),
        )

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 1
        kill_timeout = 8 * 60 * 60  # 8 hours

        with sdk2.parameters.Group('Credentials') as credentials:
            s3_access_key = sdk2.parameters.YavSecret(
                's3 access token', default='sec-01ckrgypb5aarnasyfp2b2bzjw#s3_infra_access_key')
            s3_secret_key = sdk2.parameters.YavSecret(
                's3 secret access token', default='sec-01ckrgypb5aarnasyfp2b2bzjw#s3_infra_access_secret_key')

        with sdk2.parameters.String('Target platform') as platform:
            platform.values.linux = platform.Value('linux', default=True)
            platform.values.mac = platform.Value('mac')

        tag = sdk2.parameters.String('Tag to build toolchain from', required=True)
        branch_scheme = sdk2.parameters.String('Branch scheme of swift repo checkout',
                                               description='Branch scheme to use, typically release/x.y')
        prefix = sdk2.parameters.String('Toolchain prefix', default='Yandex')
        subrevision = sdk2.parameters.String('Subrevision', default='0')

        with sdk2.parameters.Group('General settings') as general_settings:
            additional_tags = sdk2.parameters.CustomClientTags(
                'Additional client tags to select host platform')

            suspend_before_finish = sdk2.parameters.Bool('Suspend after building & publishing resource', default=False)
            lxc_container_resource_id = sdk2.parameters.Integer(
                'ID of LXC container resource to use on linux hosts', default=None, required=False)

            _binary_task_params = binary_tasks.cross_platform_binary_task_parameters()

    OVERRIDES = {
        'linux': {
            'cores': 31,
            'ram': 63,
        },
    }
    SWIFT_REPO_URL = 'https://github.com/apple/swift.git'
    S3_ENDPOINT = 'https://s3.mds.yandex.net'
    S3_BUCKET = 'broinfra-tools'

    def on_enqueue(self):
        if self.Parameters.platform in self.OVERRIDES:
            self.Requirements.cores = max(
                self.Requirements.cores, self.OVERRIDES[self.Parameters.platform].get('cores', 0))
            self.Requirements.ram = max(
                self.Requirements.ram, self.OVERRIDES[self.Parameters.platform].get('ram', 0))

        if self.Parameters.additional_tags:
            self.Requirements.client_tags = self.Requirements.client_tags & self.Parameters.additional_tags

        if self.Parameters.platform == 'linux' and self.Parameters.lxc_container_resource_id:
            self.Requirements.container_resource = self.Parameters.lxc_container_resource_id

        super(BuildSwiftToolchain, self).on_enqueue()

    def on_prepare(self):
        super(BuildSwiftToolchain, self).on_prepare()

        # One should manually call |prepare| method for conditional environments. See st/SANDBOX-7500
        LLVMToolchainEnvironment(self.Parameters.platform, '13.0.0').prepare()
        CMakeEnvironment(self.Parameters.platform, '3.20.0').prepare()
        if platform.system() == 'Darwin':
            hermetic_xcode = HermeticXcodeEnvironment('13.1')
            hermetic_xcode.prepare()

    @utils.singleton_property
    def custom_logs(self):
        custom_log_resource = sdk2.service_resources.TaskCustomLogs(self, "Swift build logs", "swift_build_logs")
        custom_logs = sdk2.ResourceData(custom_log_resource)
        custom_logs.path.mkdir(0o755, parents=True, exist_ok=True)
        return custom_logs

    @property
    def prefix(self):
        return '{prefix}.{subrevision}'.format(prefix=self.Parameters.prefix, subrevision=self.Parameters.subrevision)

    @property
    def toolchain_name(self):
        return 'swift-toolchain-{platform}-{tag}.{prefix}.tar.gz'.format(
            platform=platform.system(),
            tag=self.Parameters.tag,
            prefix=self.prefix
        ).lower()

    @utils.singleton_property
    def toolchain_archive_name(self):
        system = {
            'Linux': 'linux',
            'Darwin': 'osx',
        }[platform.system()]
        today = date.today()
        month = str(today.month).zfill(2)
        day = str(today.day).zfill(2)
        return 'swift-LOCAL-{year}-{month}-{day}-a-{system}.tar.gz'.format(
            year=today.year, month=month, day=day, system=system)

    @property
    def s3_prefix(self):
        return 'swift-toolchain-{}/'.format(platform.system().lower())

    @utils.singleton_property
    def s3_client(self):
        import boto3

        s3_session = boto3.session.Session(
            aws_access_key_id=self.Parameters.s3_access_key.data()[self.Parameters.s3_access_key.default_key],
            aws_secret_access_key=self.Parameters.s3_secret_key.data()[self.Parameters.s3_secret_key.default_key])
        return s3_session.client(service_name='s3', endpoint_url=self.S3_ENDPOINT)

    def checkout_path(self, *args):
        return str(self.path('swift-build', *args))

    def repo_path(self, *args):
        return self.checkout_path('swift', *args)

    def available_toolchains(self):
        objects = self.s3_client.list_objects_v2(Bucket=self.S3_BUCKET, Prefix=self.s3_prefix)
        return set(
            archive['Key'][len(self.s3_prefix):] for archive in objects.get('Contents', [])
        )

    def checkout(self):
        swift_repo = SandboxGit(url=self.SWIFT_REPO_URL, filter_branches=False)
        swift_repo.clone(self.repo_path(), branch='main')
        with ProcessRegistry, open(str(self.custom_logs.path.joinpath('git_checkout.log')), 'w') as log:
            subprocess.check_call(['git', 'fetch', 'origin', '--tags'], cwd=self.repo_path(),
                                  stdout=log, stderr=log)
            subprocess.check_call(['git', 'checkout', self.Parameters.tag], cwd=self.repo_path(),
                                  stdout=log, stderr=log)

    def update_checkout(self):
        args = [self.repo_path('utils', 'update-checkout'), '--clone']
        if self.Parameters.branch_scheme:
            args += ['--scheme', self.Parameters.branch_scheme]
        else:
            args += ['--tag', self.Parameters.tag]

        with ProcessRegistry, open(str(self.custom_logs.path.joinpath('update_checkout.log')), 'w') as log:
            subprocess.check_call(args, cwd=self.checkout_path(), stdout=log, stderr=log)

    def apply_patch(self):
        from library.python import resource
        patch = resource.find('swift_toolchain_patches/{}.patch'.format(self.Parameters.platform))
        with TempfileContext() as patch_path:
            with open(patch_path, 'wb') as patch_file:
                patch_file.write(patch)
            cmd = ['git', 'apply', '--verbose', '--whitespace=fix', patch_path]
            with ProcessRegistry, open(str(self.custom_logs.path.joinpath('apply_patch.log')), 'w') as log:
                subprocess.check_call(cmd, cwd=self.repo_path(), stdout=log, stderr=log)

    def build_toolchain(self):
        cmd = [
            self.repo_path('utils', 'build-toolchain'),
            self.prefix,
        ]
        hpe = HermeticPythonEnvironment(python_version='2.7.17', pip_version='9.0.2', packages=['six==1.15.0'])
        with hpe, ProcessRegistry, open(str(self.custom_logs.path.joinpath('build.log')), 'w') as log:
            subprocess.check_call(cmd, cwd=self.checkout_path(), stdout=log, stderr=log)

    def upload_toolchain(self):
        self.set_info('Uploading toolchain as {}'.format(self.toolchain_name))
        toolchain_path = self.checkout_path(self.toolchain_archive_name)
        with open(toolchain_path, 'rb') as toolchain_archive:
            archive_contens = toolchain_archive.read()
        sha1sum = hashlib.sha1(archive_contens).hexdigest()
        self.set_info('Toolchain sha1 sum is {}'.format(sha1sum))
        self.s3_client.upload_file(toolchain_path, self.S3_BUCKET, self.s3_prefix + self.toolchain_name)

    def on_execute(self):
        available_toolchains = self.available_toolchains()
        if self.toolchain_name in available_toolchains:
            self.set_info('Toolchain already exists, exiting')
        else:
            self.set_info('Toolchain {} is not found across existing: {}'.format(
                self.toolchain_name, ', '.join(available_toolchains)))

        os.environ['GIT_SSL_NO_VERIFY'] = '1'
        self.checkout()
        self.update_checkout()
        self.apply_patch()
        self.build_toolchain()
        self.upload_toolchain()
