import json
import logging
import os
import platform
import shutil
import sys

import sandbox.common.types.client as ctc
from sandbox.common.types import resource as ctr

from sandbox.projects.browser.common import binary_tasks
from sandbox.projects.browser.common.depot_tools import DepotToolsEnvironment
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.hermetic_xcode import HermeticXcodeEnvironment
from sandbox.projects.browser.common.teamcity_step import teamcity_step
from sandbox.projects.browser.common.win_toolchain import WindowsToolchainEnvironment

from sandbox import sdk2
from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox.sdk2.helpers import ProcessLog, subprocess


def _get_mac_sysroot():
    sdks = json.loads(subprocess.check_output(['/usr/bin/xcodebuild', '-showsdks', '-json']))
    for sdk in sdks:
        if sdk['platform'] == 'macosx' and sdk['canonicalName'].startswith('macosx'):
            assert float(sdk['platformVersion']) >= 11, 'Mac platforms prior to macosx11 do not support arm64 arch'
            return sdk['sdkPath']


class Yakuza(sdk2.Resource):
    """
    Compiled yakuza (browser's fork of ninja) binary.
    """
    any_arch = False
    auto_backup = True
    executable = True
    restart_policy = ctr.RestartPolicy.IGNORE
    commit = sdk2.Attributes.String('Ninja repo commit hash', required=True)
    ttl = 356


class YakuzaEnvironment(SandboxEnvironment):
    """
    Places most recent yakuza published as resource to PATH.
    """

    @property
    def task(self):
        return sdk2.Task.current

    @property
    def directory_with_binary(self):
        return str(self.task.current.path('_yakuza'))

    def prepare(self):
        if sys.platform.startswith("linux"):
            current_platform = "linux"
        elif sys.platform == 'darwin':
            current_platform = 'macos'
        else:
            current_platform = 'win'
        yakuza_resource = Yakuza.find(attrs={'platform': current_platform}).first()
        yakuza_data = sdk2.ResourceData(yakuza_resource)
        os.makedirs(self.directory_with_binary)
        shutil.copy(str(yakuza_data.path), self.directory_with_binary)
        os.environ['PATH'] = os.pathsep.join([os.environ['PATH'], self.directory_with_binary])


class BuildYakuza(binary_tasks.CrossPlatformBinaryTaskMixin, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        cores = 8
        client_tags = ctc.Tag.BROWSER
        # To build yakuza quite heavy additional environemnts like Xcode.app would be installed
        disk_space = 20 * 1024
        environments = (
            GitEnvironment('2.24.1'),
        )
        ram = 15 * 1024

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 1
        kill_timeout = 60 * 60  # 1 hr

        with sdk2.parameters.Group('Repositories settings') as repositories_settings:
            branch = sdk2.parameters.String('Branch to checkout on', default='yakuza')
            commit = sdk2.parameters.String('Commit to checkout on')

            depot_tools_revision = sdk2.parameters.String('Depot tools revision', default='master')

        with sdk2.parameters.Group('Task settings') as task_settings:
            with sdk2.parameters.String('Target platform') as platform:
                platform.values.linux = platform.Value('linux')
                platform.values.mac = platform.Value('mac')
                platform.values.win = platform.Value('win')

            additional_tags = sdk2.parameters.CustomClientTags(
                'Additional client tags (will be intersected with default)')

            suspend_at_end = sdk2.parameters.Bool('Suspend at end')

            _binary_task_params = binary_tasks.cross_platform_binary_task_parameters()

    @property
    def ninja_executable(self):
        return 'ninja.exe' if self.Parameters.platform == 'win' else 'ninja'

    @property
    def yakuza_executable(self):
        return 'yakuza.exe' if self.Parameters.platform == 'win' else 'yakuza'

    def provide_depot_tools(self):
        depot_tools_env = DepotToolsEnvironment(revision=self.Parameters.depot_tools_revision)
        depot_tools_env.prepare()
        return str(depot_tools_env.depot_tools_folder)

    def on_enqueue(self):
        super(BuildYakuza, self).on_enqueue()
        if self.Parameters.additional_tags:
            self.Requirements.client_tags &= self.Parameters.additional_tags

    def on_prepare(self):
        super(BuildYakuza, self).on_prepare()
        # One should manually call |prepare| method for conditional environments. See st/SANDBOX-7500
        if self.Parameters.platform == 'mac':
            hermetic_xcode = HermeticXcodeEnvironment('12.4')
            hermetic_xcode.prepare()
        elif self.Parameters.platform == 'win':
            win_toolchain = WindowsToolchainEnvironment('d1d6cc8eb1')
            win_toolchain.prepare()

    def ninja_repo_path(self, *args):
        return str(self.path('ninja', *args))

    @property
    def yakuza_path(self):
        return str(self.path(self.yakuza_executable))

    def checkout_repository(self):
        repositories.Stardust.ninja(filter_branches=False).clone(
            self.ninja_repo_path(), self.Parameters.branch, self.Parameters.commit)

    def publish(self):
        resource = Yakuza(self, 'Yakuza', self.yakuza_path, commit=self.Parameters.commit)
        sdk2.ResourceData(resource).ready()
        logging.debug('Published yakuza as resource %d.', resource.id)
        with teamcity_step(self, 'publish-yakuza-binary', 'publish_yakuza_binary') as tac:
            tac.logger.info("##teamcity[publishArtifacts '{}']\n".format(self.yakuza_path))

    def build(self, logger_name, depot_tools_dir, bootstrap, cflags, ldflags):
        if cflags is not None:
            os.environ['CFLAGS'] = cflags
        if ldflags is not None:
            os.environ['LDFLAGS'] = ldflags

        logging.info('Building yakuza with cflags: %s', cflags)
        logging.info('Building yakuza with ldlags: %s', ldflags)
        python3_path = os.path.join(depot_tools_dir, 'python3.bat' if platform.system() == 'Windows' else 'python3')
        with ProcessLog(self, logger=logger_name) as log:
            configure_args = [python3_path, 'configure.py', '--verbose']
            if bootstrap:
                configure_args.append('--bootstrap')

            subprocess.check_call(configure_args, cwd=self.ninja_repo_path(), stdout=log.stdout, stderr=log.stderr)
            if bootstrap:
                subprocess.check_call([self.ninja_repo_path(self.ninja_executable), 'all'], cwd=self.ninja_repo_path(),
                                      stdout=log.stdout, stderr=log.stderr)
                test_executable = 'ninja_test.exe' if self.Parameters.platform == 'win' else 'ninja_test'
                subprocess.check_call([self.ninja_repo_path(test_executable)], cwd=self.ninja_repo_path(),
                                      stdout=log.stdout, stderr=log.stderr)
                shutil.copy(self.ninja_repo_path(self.ninja_executable), self.yakuza_path)
            else:
                subprocess.check_call([self.yakuza_path, 'all'], cwd=self.ninja_repo_path())

    def on_execute(self):
        depot_tools_dir = self.provide_depot_tools()
        self.checkout_repository()
        common_cflags = ''
        common_ldflags = ''
        if self.Parameters.platform == 'mac':
            common_cflags = '-stdlib=libc++ -isysroot {}'.format(_get_mac_sysroot())
            common_ldflags = '-Wl,-s'

        native_cflags = common_cflags
        native_ldflags = common_ldflags
        if self.Parameters.platform == 'mac':
            native_cflags += ' -mmacosx-version-min=10.14'

        self.build('build_x64_binary', depot_tools_dir, True, native_cflags, native_ldflags)
        if self.Parameters.platform == 'mac':
            cflags = common_cflags + ' -target arm64-apple-macosx11'
            logger_name = 'build_arm64_binary'
            # We cannot bootstrap & test on x86_64 machine arm64 binariy :(
            self.build(logger_name, depot_tools_dir, False, cflags, common_ldflags)

            with ProcessLog(self, logger=logger_name) as log:
                subprocess.check_call([
                    '/usr/bin/lipo', '-create', '-output', self.yakuza_path, self.ninja_repo_path(self.ninja_executable),
                    self.yakuza_path], stdout=log.stdout, stderr=log.stderr)

        self.publish()
        if self.Parameters.suspend_at_end:
            self.suspend()
