import sandbox.common.types.client as ctc
from sandbox.common import utils

from sandbox.projects.browser.common.bitbucket import BitBucket, DEFAULT_BITBUCKET_URL
from sandbox.projects.browser.common.RunBrowserScript import RunBrowserScript

from sandbox import sdk2


class CreateHermeticXcode(RunBrowserScript):
    class Requirements(RunBrowserScript.Requirements):
        # Ask for latest version as new Xcode may fail to run first launch on old versions.
        client_tags = ctc.Tag.BROWSER & ctc.Tag.OSX_MONTEREY & (ctc.Tag.INTEL_8700B | ctc.Tag.M1)
        disk_space = 40 * 1024

    class Parameters(RunBrowserScript.Parameters):
        max_restarts = 0
        kill_timeout = 1 * 60 * 60

        with sdk2.parameters.Group('Task settings') as task_settings:
            dry_run = sdk2.parameters.Bool('Do not upload hermetic xcode and SDKs')

            with sdk2.parameters.String('Select Hermetic Xcode type') as hermetic_xcode_type:
                hermetic_xcode_type.values.xcode_binaries = hermetic_xcode_type.Value('xcode-binaries', default=True)
                hermetic_xcode_type.values.ios = hermetic_xcode_type.Value('ios')
                hermetic_xcode_type.values.mac = hermetic_xcode_type.Value('mac')

            xcode_version = sdk2.parameters.String('Xcode version in X.Y[.Z] format', required=True)
            xcode_build_version = sdk2.parameters.String('Xcode build version (use to override)')

        with sdk2.parameters.Group('Credentials') as credentials_group:
            bitbucket_token_vault = sdk2.parameters.String(
                'Vault item with oauth token for bitbucket', default='robot-bro-commiter-bitbucket-token')
            yav_token = sdk2.parameters.String(
                'Vault item with yav.yandex-team.ru OAuth token', default='robot-bro-commiter_yav_token')

    secret_envvars = RunBrowserScript.secret_envvars + ('YAV_TOKEN', )

    @utils.singleton_property
    def bb(self):
        return BitBucket(DEFAULT_BITBUCKET_URL, 'x-oauth-token',
                         sdk2.Vault.data(self.Parameters.bitbucket_token_vault))

    @utils.singleton_property
    def bitbucket_browser_sparse_checkout_paths(self):
        return ['requirements.txt', 'src/.DEPS.snapshot', 'src/build']

    def script_cmd(self, python_executable):
        cmd = [
            str(python_executable), '-m', 'yandex.tools.upload_mac_toolchain',
            '--platform', self.Parameters.hermetic_xcode_type
        ]
        if self.Parameters.xcode_build_version:
            cmd += ['--xcode-build-version', self.Parameters.xcode_build_version]
        if self.Parameters.dry_run:
            cmd += ['--dry-run']
        cmd += [self.Parameters.xcode_version]
        return cmd

    def script_extra_env(self):
        env = super(CreateHermeticXcode, self).script_extra_env()
        env.update({
            'YAV_TOKEN': sdk2.Vault.data(self.Parameters.yav_token),
        })
        return env
