import json
import logging
import time

from sandbox.common import errors
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.common.types.task import Status
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess

from sandbox.projects.browser.builds.compiler.BuildBrowserClang import BuildBrowserClang, BrowserClang
from sandbox.projects.browser.builds.compiler.UploadBrowserClang import UploadBrowserClang
from sandbox.projects.browser.common.binary_tasks import LinuxBinaryTaskMixin
from sandbox.projects.browser.common.depot_tools import DepotToolsEnvironment
from sandbox.projects.browser.common.git import ConfigureGitEnvironment, GitEnvironment, repositories
from sandbox.projects.browser.common.git.git_cli import GitCli


LXC_CONTAINER_ID = 1414674927
PLATFORM_TO_PARAMETERS = {
    'linux': {
        'binary_platform': 'linux',
        'additional_tags': 'LINUX',
        'lxc_container_id': LXC_CONTAINER_ID,
    },
    'mac': {
        'binary_platform': 'osx',
        'additional_tags': 'OSX_MONTEREY & INTEL_8700B',
    },
    'mac_arm64': {
        'binary_platform': 'osx',

        # For now LLVM is able to build clang for arm64 only by cross-compiling
        # on macOS Intel host.
        'additional_tags': 'OSX_MONTEREY & INTEL_8700B',
    },
    'win': {
        'binary_platform': 'win_nt',
        'additional_tags': 'WINDOWS',
    },
}
S3_BUCKET = 'broinfra-tools'
S3_ENDPOINT = 'https://s3.mds.yandex.net'


class TriggerBrowserClangBuilds(LinuxBinaryTaskMixin, sdk2.Task):
    """
    Run child tasks to build & publish clang distribution as Sandbox resource.
    Than upload that resources to s3 and create PR to browser repo to add a
    mapping for current clang version to uploaded distributions.
    Also if a linux version is built, launches Sandbox task to upload it to
    Distclang remotes.
    """
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 1 * 1024
        ram = 1 * 1024
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        dns = ctm.DnsType.DNS64
        environments = (
            GitEnvironment('2.24.1'),
            ConfigureGitEnvironment('robot-bro-commiter@yandex-team.ru', 'Avatar Claang'),
        )
        container_resource = LXC_CONTAINER_ID

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 3
        kill_timeout = 5 * 60 * 60

        with sdk2.parameters.Group('Repositories settings') as repositories_settings:
            branch = sdk2.parameters.String('Branch to checkout browser repo on', default='master')
            commit = sdk2.parameters.String('Commit to checkout browser repo on')

            depot_tools_revision = sdk2.parameters.String('Depot tools revision',
                                                          default='bad22e5d34feb42fae6303c79d72713bb9354a67')

        with sdk2.parameters.Group('Update settings') as update_settings:
            with sdk2.parameters.CheckGroup('Platforms to build and update') as platforms:
                platforms.values.linux = platforms.Value('linux', checked=True)
                platforms.values.mac = platforms.Value('mac', checked=True)
                platforms.values.mac_arm64 = platforms.Value('mac-arm64', checked=True)
                platforms.values.win = platforms.Value('win', checked=True)

            reviewers = sdk2.parameters.List('Reviewers', default=['matthewtff'],
                                             description='Users to add as reviewers to PR with clang update',
                                             value_type=sdk2.parameters.Staff)

            force_update = sdk2.parameters.Bool(
                'Force update', default=False,
                description='Update even if a mapping for current LLVM revision already exists')

            dry_run = sdk2.parameters.Bool('Dry run', default=False,
                                           description='Build clang distributions, but do not push them to s3')

        with sdk2.parameters.Group('Credentials') as credentials:
            bb_token = sdk2.parameters.YavSecret(
                'Bitbucket OAuth token', default='sec-01ckrgypb5aarnasyfp2b2bzjw#bitbucket_oauth_token')
            s3_access_key = sdk2.parameters.YavSecret(
                's3 access token', default='sec-01ckrgypb5aarnasyfp2b2bzjw#s3_infra_access_key')
            s3_secret_key = sdk2.parameters.YavSecret(
                's3 secret access token', default='sec-01ckrgypb5aarnasyfp2b2bzjw#s3_infra_access_secret_key')
            ssh_key = sdk2.parameters.YavSecret(
                'YAV secret with SSH key', default='sec-01ckrgypb5aarnasyfp2b2bzjw#ssh_key')

    class Context(sdk2.Task.Context):
        launched_tasks = dict()

        # LLVM revision used by specified browser branch/commit.
        llvm_revision = None

        # Id of UPLOAD_BROWSER_CLANG task that uploads linux version of clang to separate place in s3 to make sure
        # it would be distributed across Distclang remotes.
        upload_task_id = None

    @property
    def browser_sparse_checkout_paths(self):
        return ['src/tools/clang/scripts']

    @property
    def update_message(self):
        assert self.Context.llvm_revision is not None
        return 'Update clang on {platforms} from LLVM revision {llvm_revision}'.format(
            platforms=', '.join(self.Parameters.platforms), llvm_revision=self.Context.llvm_revision)

    @property
    def reviewers(self):
        return self.Parameters.reviewers or [self.author]

    def browser_path(self, *args):
        return self.path('browser', *args)

    def upload_resources_to_s3(self, child_tasks):
        import boto3

        s3_session = boto3.session.Session(
            aws_access_key_id=self.Parameters.s3_access_key.data()[self.Parameters.s3_access_key.default_key],
            aws_secret_access_key=self.Parameters.s3_secret_key.data()[self.Parameters.s3_secret_key.default_key])
        s3_client = s3_session.client(service_name='s3', endpoint_url=S3_ENDPOINT)

        uploaded_archives = dict()
        for platform, task_id in child_tasks.items():
            clang_archive_resource = BrowserClang.find(task_id=task_id).first()
            clang_archive = sdk2.ResourceData(clang_archive_resource)
            name = 'clang-{}-{}.tar.gz'.format(self.Context.llvm_revision, clang_archive_resource.browser_commit)
            directory = {
                'linux': 'linux-x64',
                'mac': 'mac-x64',
                'mac_arm64': 'mac-arm64',
                'win': 'win-x64',
            }[platform]
            s3_path = 'compilers/browser-clang/{}/{}'.format(directory, name)
            s3_client.upload_file(str(clang_archive.path), S3_BUCKET, s3_path)
            uploaded_archives[platform] = '{endpoint}/{bucket}/{path}'.format(
                endpoint=S3_ENDPOINT, bucket=S3_BUCKET, path=s3_path)
            self.set_info('Uploaded archive for {platform} to {url}'.format(
                platform=platform, url=uploaded_archives[platform]))
        return uploaded_archives

    def checkout_browser_repo(self):
        repositories.Stardust.browser(filter_branches=False).clone(
            str(self.browser_path()), self.Parameters.branch, self.Parameters.commit,
            sparse_checkout_paths=self.browser_sparse_checkout_paths)

    def mapping_file_path(self, platform):
        platform_prefix = {
            'mac_arm64': 'mac-arm64',
        }.get(platform, platform)
        return self.browser_path('src', 'tools', 'clang', 'scripts', '{}-llvm-mapping.json'.format(platform_prefix))

    def get_llvm_revision(self, python3_path):
        update_script = str(self.browser_path('src', 'tools', 'clang', 'scripts', 'update.py'))
        try:
            revision = subprocess.check_output([python3_path, update_script, '--print-revision'],
                                               stderr=subprocess.STDOUT).strip()
        except subprocess.CalledProcessError as e:
            logging.error('Unable to determine llvm revision: %s', e.stdout)
            raise

        return revision

    def provide_depot_tools(self):
        depot_tools_env = DepotToolsEnvironment(revision=self.Parameters.depot_tools_revision)
        depot_tools_env.prepare()
        return depot_tools_env.depot_tools_folder

    def update_json_files(self, uploaded_archives):
        repo = GitCli(str(self.browser_path()))
        branch_name = 'wp/robots/update-clang/{}'.format(str(int(time.time())))
        repo.branch('-m', branch_name)
        for platform, url in uploaded_archives.items():
            mapping_file = self.mapping_file_path(platform)
            mapping = json.loads(mapping_file.read_text())
            mapping[self.Context.llvm_revision] = url
            mapping_file.write_text(
                json.dumps(mapping, indent=4, separators=(',', ': '), sort_keys=True).decode('latin1'))
            repo.add(str(mapping_file))
        repo.commit('-m', self.update_message)
        with sdk2.ssh.Key(private_part=self.Parameters.ssh_key.data()[self.Parameters.ssh_key.default_key]):
            repo.push('origin', branch_name)
        self.set_info('Pushed updated json file(s) to branch {}'.format(branch_name))
        return branch_name

    def create_pr(self, branch_name):
        from bitbucket import BitBucket

        bitbucket = BitBucket('https://bitbucket.browser.yandex-team.ru/',
                              token=self.Parameters.bb_token.data()[self.Parameters.bb_token.default_key])
        pull_request = bitbucket.projects['STARDUST'].repos['browser'].pull_requests.create_pull_request(
            title=self.update_message,
            description='Created by Sandbox task [#{task_id}](https://sandbox.yandex-team.ru/task/{task_id})'.format(
                task_id=self.id),
            source_branch=branch_name,
            dest_branch=self.Parameters.branch,
            reviewers=self.reviewers)
        self.set_info('Pull request was successfully created: {}'.format(pull_request.web_url))

    def on_execute(self):
        # If we have non-null uploading task id, that means that all previous steps have succeeded:
        # - clang distributions were built for all the required platforms
        # - distributions were uploaded to s3
        # - PR with update of mappings to new distributions was created
        # - Uploading task for Distclang remotes was launched
        # So now just check the status of uploading task and return it as it's own status.
        if self.Context.upload_task_id:
            upload_task = sdk2.Task[self.Context.upload_task_id]
            if upload_task.status not in Status.Group.SUCCEED:
                raise errors.TaskFailure('Uploading task has failed')

        if self.Context.launched_tasks:
            failed_tasks = []
            for platform, task_id in self.Context.launched_tasks.items():
                child_task = sdk2.Task[task_id]
                if child_task.status not in Status.Group.SUCCEED:
                    failed_tasks += [(task_id, platform)]

            if failed_tasks:
                raise errors.TaskFailure('The following child tasks failed: {}'.format(', '.join([
                    '#{} for {}'.format(task_id, platform) for (task_id, platform) in failed_tasks
                ])))

            if not self.Parameters.dry_run:
                uploaded_archives = self.upload_resources_to_s3(self.Context.launched_tasks)
                self.checkout_browser_repo()
                branch_name = self.update_json_files(uploaded_archives)
                self.create_pr(branch_name)

                if 'linux' in self.Context.launched_tasks:
                    upload_task = UploadBrowserClang(
                        self, description='Upload browser clang {}'.format(self.Context.llvm_revision),
                        branch=branch_name, depot_tools_revision=self.Parameters.depot_tools_revision,
                        s3_access_key=self.Parameters.s3_access_key, s3_secret_key=self.Parameters.s3_secret_key,
                        bitbucket_oauth_token=self.Parameters.bb_token)
                    upload_task.enqueue()
                    self.Context.upload_task_id = upload_task.id
                    self.Context.save()
                    raise sdk2.WaitTask(self.Context.upload_task_id, list(Status.Group.FINISH + Status.Group.BREAK))
        else:
            self.checkout_browser_repo()
            depot_tools_dir = self.provide_depot_tools()
            python3_path = str(depot_tools_dir.joinpath('python3'))
            self.Context.llvm_revision = self.get_llvm_revision(python3_path)
            self.hint(self.Context.llvm_revision)
            tasks_to_wait = []
            for platform in self.Parameters.platforms:
                parameters = PLATFORM_TO_PARAMETERS[platform]
                mapping_file = self.mapping_file_path(platform)
                mapping = json.loads(mapping_file.read_text())
                if self.Context.llvm_revision in mapping and not self.Parameters.force_update:
                    self.set_info('Skipping updating clang for {} as mapping for it already exists'.format(platform))
                    continue

                description = '[{platform}] {description}'.format(
                    platform=platform, description=self.Parameters.description)
                build_task = BuildBrowserClang(self, description=description, platform=platform,
                                               branch=self.Parameters.branch, commit=self.Parameters.commit,
                                               depot_tools_revision=self.Parameters.depot_tools_revision,
                                               llvm_revision=self.Context.llvm_revision,
                                               additional_tags=parameters['additional_tags'],
                                               lxc_container_resource_id=parameters.get('lxc_container_id'),
                                               binary_platform=parameters['binary_platform'])
                build_task.enqueue()
                tasks_to_wait.append(build_task)
                self.Context.launched_tasks[platform] = build_task.id
            self.Context.save()
            if tasks_to_wait:
                raise sdk2.WaitTask(tasks_to_wait, list(Status.Group.FINISH + Status.Group.BREAK), wait_all=True)
