import json
import os
import tarfile
import time

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.common.types.task import Status
from sandbox import sdk2

from sandbox.projects.browser.builds.compiler.BuildPortableDistclang import BuildPortableDistclang, Distclang
from sandbox.projects.browser.common.binary_tasks import LinuxBinaryTaskMixin
from sandbox.projects.browser.common.git import ConfigureGitEnvironment, GitEnvironment, repositories
from sandbox.projects.browser.common.git.git_cli import GitCli


ALL_UPDATE_CHANNELS = ('beta', 'stable')
PLATFORM_TO_PARAMETERS = {
    'linux': {
        'binary_platform': 'linux',
        'additional_tags': 'LINUX',
        'lxc_container_id': 1414674927,
    },
    'mac': {
        'binary_platform': 'osx',
        'additional_tags': '(OSX_BIG_SUR | OSX_MONTEREY) & INTEL_8700B',
    },
    'mac_arm64': {
        'binary_platform': 'osx',
        'additional_tags': '(OSX_BIG_SUR | OSX_MONTEREY) & M1',
    },
    'win': {
        'binary_platform': 'win_nt',
        'additional_tags': 'WINDOWS',
    },
}
S3_BUCKET = 'distclang'
S3_ENDPOINT = 'https://s3.mds.yandex.net'


class TriggerPortableDistclangBuilds(LinuxBinaryTaskMixin, sdk2.Task):
    """
    Trigger portable distclang builds on specified platforms, upload to s3 and
    update json files in specified browser branch.
    """
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 1 * 1024
        ram = 1 * 1024
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        dns = ctm.DnsType.DNS64
        environments = (
            GitEnvironment('2.24.1'),
            ConfigureGitEnvironment('robot-bro-commiter@yandex-team.ru', 'Avatar Claang'),
        )

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 3
        kill_timeout = 5 * 60 * 60

        with sdk2.parameters.Group('Repositories settings') as repositories_settings:
            browser_branch = sdk2.parameters.String('Branch to checkout browser repo on', default='master')
            browser_commit = sdk2.parameters.String('Commit to checkout browser repo on')

            distclang_branch = sdk2.parameters.String('Branch to checkout distclang repo on', default='browser')
            distclang_commit = sdk2.parameters.String('Commit to checkout distclang repo on')

            depot_tools_revision = sdk2.parameters.String('Depot tools revision',
                                                          default='bad22e5d34feb42fae6303c79d72713bb9354a67')

        with sdk2.parameters.Group('Update settings') as update_settings:
            with sdk2.parameters.CheckGroup('Platforms to update distclang for') as platforms:
                platforms.values.linux = platforms.Value('linux', checked=True)
                platforms.values.mac = platforms.Value('mac', checked=True)
                platforms.values.mac_arm64 = platforms.Value('mac_arm64', checked=True)
                platforms.values.win = platforms.Value('win', checked=True)

            with sdk2.parameters.CheckGroup('Channels to update') as update_channels:
                update_channels.values.beta = update_channels.Value('Beta channel', checked=True)
                update_channels.values.stable = update_channels.Value('Stable channel', checked=True)

            reviewers = sdk2.parameters.List('Reviewers', default=['matthewtff'],
                                             description='Users to add as reviewers to PR with Distclang update',
                                             value_type=sdk2.parameters.Staff)

        with sdk2.parameters.Group('Credentials') as credentials:
            bb_token = sdk2.parameters.YavSecret(
                'Bitbucket OAuth token', default='sec-01ckrgypb5aarnasyfp2b2bzjw#bitbucket_oauth_token')
            s3_access_key = sdk2.parameters.YavSecret(
                's3 access token', default='sec-01ckrgypb5aarnasyfp2b2bzjw#s3_distclang_access_key')
            s3_secret_key = sdk2.parameters.YavSecret(
                's3 secret access token', default='sec-01ckrgypb5aarnasyfp2b2bzjw#s3_distclang_access_secret_key')
            ssh_key = sdk2.parameters.YavSecret(
                'YAV secret with SSH key', default='sec-01ckrgypb5aarnasyfp2b2bzjw#ssh_key')

    class Context(sdk2.Task.Context):
        launched_tasks = dict()

    @property
    def browser_sparse_checkout_paths(self):
        return ['src/tools/dist-clang']

    @property
    def update_message(self):
        return 'Update distclang from branch {}'.format(self.Parameters.distclang_branch)

    def browser_path(self, *args):
        return self.path('browser', *args)

    def pack_directory(self, platform, directory_path):
        result = str(self.path('distclang_{}.tgz'.format(platform)))
        with tarfile.open(result, 'w:gz') as archive:
            for file_path in os.listdir(directory_path):
                archive.add(os.path.join(directory_path, file_path), arcname=file_path)
        return result

    def upload_resources_to_s3(self, child_tasks):
        import boto3

        s3_session = boto3.session.Session(
            aws_access_key_id=self.Parameters.s3_access_key.data()[self.Parameters.s3_access_key.default_key],
            aws_secret_access_key=self.Parameters.s3_secret_key.data()[self.Parameters.s3_secret_key.default_key])
        s3_client = s3_session.client(service_name='s3', endpoint_url=S3_ENDPOINT)

        uploaded_archives = dict()
        for platform, task_id in child_tasks.items():
            distclang_resource = Distclang.find(task_id=task_id).first()
            distclang_dir = sdk2.ResourceData(distclang_resource)
            distclang_archive = self.pack_directory(platform, str(distclang_dir.path))
            name = 'distclang-{}.tar.gz'.format(distclang_resource.commit)
            directory = {
                'linux': 'linux-x64',
                'mac': 'mac-x64',
                'mac_arm64': 'mac-arm64',
                'win': 'win-x64',
            }[platform]
            s3_path = 'portable-distclangs/{}/{}'.format(directory, name)
            s3_client.upload_file(distclang_archive, S3_BUCKET, s3_path)
            uploaded_archives[platform] = '{endpoint}/{bucket}/{path}'.format(
                endpoint=S3_ENDPOINT, bucket=S3_BUCKET, path=s3_path)
            self.set_info('Uploaded archive for {platform} to {url}'.format(
                platform=platform, url=uploaded_archives[platform]))
        return uploaded_archives

    def checkout_browser_repo(self):
        repositories.Stardust.browser(filter_branches=False).clone(
            str(self.browser_path()), self.Parameters.browser_branch, self.Parameters.browser_commit,
            sparse_checkout_paths=self.browser_sparse_checkout_paths)

    def update_json_files(self, uploaded_archives):
        repo = GitCli(str(self.browser_path()))
        branch_name = 'wp/robots/update-distclang/{}'.format(str(int(time.time())))
        repo.branch('-m', branch_name)
        for platform, url in uploaded_archives.items():
            path = self.browser_path('src', 'tools', 'dist-clang', '{}-distclang.json'.format(platform))
            with open(str(path), 'r') as config_file:
                try:
                    config = json.load(config_file)
                except ValueError:
                    # It's possible that json contains conflict markers, in that case if we update all update channels,
                    # than we could ignore parsing errors and just overwrite entire config with new values.
                    if sorted(self.Parameters.update_channels) != sorted(ALL_UPDATE_CHANNELS):
                        raise
                    config = {}

            for channel in self.Parameters.update_channels:
                config[channel] = url
            with open(str(path), 'w') as config_file:
                json.dump(config, config_file, indent=4, separators=(',', ': '), sort_keys=True)
            repo.add(str(path))
        repo.commit('-m', self.update_message)
        with sdk2.ssh.Key(private_part=self.Parameters.ssh_key.data()[self.Parameters.ssh_key.default_key]):
            repo.push('origin', branch_name)
        self.set_info('Pushed updated json file(s) to branch {}'.format(branch_name))
        return branch_name

    def create_pr(self, branch_name):
        from bitbucket import BitBucket

        bitbucket = BitBucket('https://bitbucket.browser.yandex-team.ru/',
                              token=self.Parameters.bb_token.data()[self.Parameters.bb_token.default_key])
        pull_request = bitbucket.projects['STARDUST'].repos['browser'].pull_requests.create_pull_request(
            title=self.update_message,
            description='Created by Sandbox task [#{task_id}](https://sandbox.yandex-team.ru/task/{task_id})'.format(
                task_id=self.id),
            source_branch=branch_name,
            dest_branch=self.Parameters.browser_branch,
            reviewers=self.Parameters.reviewers)
        self.set_info('Pull request was successfully created: {}'.format(pull_request.web_url))

    def on_execute(self):
        if self.Context.launched_tasks:
            uploaded_archives = self.upload_resources_to_s3(self.Context.launched_tasks)
            self.checkout_browser_repo()
            branch_name = self.update_json_files(uploaded_archives)
            self.create_pr(branch_name)
        else:
            tasks_to_wait = []
            for platform in self.Parameters.platforms:
                parameters = PLATFORM_TO_PARAMETERS[platform]
                description = '[{platform}] {description}'.format(
                    platform=platform, description=self.Parameters.description)
                build_task = BuildPortableDistclang(self, description=description, platform=platform,
                                                    branch=self.Parameters.distclang_branch,
                                                    commit=self.Parameters.distclang_commit,
                                                    depot_tools_revision=self.Parameters.depot_tools_revision,
                                                    additional_tags=parameters['additional_tags'],
                                                    lxc_container_resource_id=parameters.get('lxc_container_id'),
                                                    binary_platform=parameters['binary_platform'])
                build_task.enqueue()
                tasks_to_wait.append(build_task)
                self.Context.launched_tasks[platform] = build_task.id
            self.Context.save()
            raise sdk2.WaitTask(tasks_to_wait, list(Status.Group.FINISH + Status.Group.BREAK), wait_all=True)
