import contextlib
import logging
import re
import tarfile
import tempfile

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.common import utils
from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessRegistry, subprocess

from sandbox.projects.browser.common.binary_tasks import LinuxBinaryTaskMixin
from sandbox.projects.browser.common.bitbucket import BitBucket, DEFAULT_BITBUCKET_URL
from sandbox.projects.browser.common.depot_tools import DepotToolsEnvironment
from sandbox.projects.browser.common.git import GitEnvironment


CLANG_VERSION_RE = re.compile(
    r'clang version (?P<version>\d+\.\d+\.\d+) '
    r'\((?P<is_browser>https://bitbucket.browser)?.* (?P<llvm_revision>\S+)\)'
)

S3_BUCKET = 'broinfra-tools'
S3_ENDPOINT = 'https://s3.mds.yandex.net'
COMPILERS_PREFIX = 'compilers/linux/'


class UploadBrowserClang(LinuxBinaryTaskMixin, sdk2.Task):
    """
    Upload clang in given revision of browser or chromium repo and upload it
    as tar archive to s3 if was not uploaded already.
    """
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 50 * 1024
        ram = 1 * 1024
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        dns = ctm.DnsType.DNS64
        environments = (
            GitEnvironment('2.24.1'),
        )

        container_resource = 3129798193

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 3
        kill_timeout = 1 * 60 * 60

        with sdk2.parameters.Group('Repositories settings') as repositories_settings:
            branch = sdk2.parameters.String('Branch to checkout repo on', default='upstream/dev')
            commit = sdk2.parameters.String('Commit to checkout repo on')

        dry_run = sdk2.parameters.Bool('Dry run', default=False)
        depot_tools_revision = sdk2.parameters.String('Depot tool revision', default='master')

        with sdk2.parameters.Group('Credentials') as credentials:
            s3_access_key = sdk2.parameters.YavSecret(
                's3 access token', default='sec-01ckrgypb5aarnasyfp2b2bzjw#s3_infra_access_key')
            s3_secret_key = sdk2.parameters.YavSecret(
                's3 secret access token', default='sec-01ckrgypb5aarnasyfp2b2bzjw#s3_infra_access_secret_key')
            bitbucket_oauth_token = sdk2.parameters.YavSecret(
                'Bitbucket OAuth token', default='sec-01ckrgypb5aarnasyfp2b2bzjw#bitbucket_oauth_token')

    @property
    def checkout_revision(self):
        return self.Parameters.commit or self.Parameters.branch

    @property
    def git_archive_paths(self):
        return [
            'src/tools/clang/scripts',
        ]

    def browser_path(self, *args):
        return self.path('browser', *args)

    @utils.singleton_property
    def custom_logs(self):
        custom_log_resource = sdk2.service_resources.TaskCustomLogs(self, "Clang update logs", "clang_update_logs")
        custom_logs = sdk2.ResourceData(custom_log_resource)
        custom_logs.path.mkdir(0o755, parents=True, exist_ok=True)
        return custom_logs

    @utils.singleton_property
    def bb(self):
        return BitBucket(
            DEFAULT_BITBUCKET_URL, 'x-oauth-token',
            self.Parameters.bitbucket_oauth_token.data()[self.Parameters.bitbucket_oauth_token.default_key])

    def provide_depot_tools(self):
        depot_tools_env = DepotToolsEnvironment(revision=self.Parameters.depot_tools_revision)
        depot_tools_env.prepare()
        return depot_tools_env.depot_tools_folder

    def checkout_repo(self):
        with contextlib.closing(
            self.bb.get_archive('STARDUST', 'browser', at=self.checkout_revision,
                                archive_format='tgz', path=self.git_archive_paths)
        ) as archive_response:
            with tarfile.open(fileobj=archive_response.raw, mode='r|*') as repo_archive:
                repo_archive.extractall(str(self.browser_path()))

    def checkout_clang(self, python3_path, fetch_log):
        update_script = self.browser_path('src', 'tools', 'clang', 'scripts', 'update.py')
        with ProcessRegistry:
            with open(str(fetch_log), 'w') as log:
                subprocess.check_call([python3_path, str(update_script)], cwd=str(self.browser_path()),
                                      stdout=log, stderr=log)

    def clang_full_name(self, clang_path):
        clang_binary = clang_path.joinpath('bin', 'clang')
        with ProcessRegistry:
            clang_version = subprocess.check_output([str(clang_binary), '--version'])
        logging.info('Clang version output: %s', clang_version)
        search_result = CLANG_VERSION_RE.search(clang_version)
        if not search_result:
            logging.error('Unable to determine clang revision from version output')
            raise RuntimeError('Unable to determine clang revision')
        clang_version = search_result.group('version')
        revision = search_result.group('llvm_revision')

        provider = 'browser' if search_result.group('is_browser') else 'upstream'
        full_name = 'clang-{provider}-{version}-{revision}'.format(
            provider=provider, version=clang_version, revision=revision)
        self.set_info('Clang full name determined as {}'.format(full_name))
        return full_name

    def list_existing_compilers(self, s3_client):
        objects = s3_client.list_objects_v2(Bucket=S3_BUCKET, Prefix=COMPILERS_PREFIX)
        return [compiler['Key'][len(COMPILERS_PREFIX):] for compiler in objects.get('Contents', [])]

    def upload_if_missing(self, archive_name, clang_path):
        import boto3

        s3_session = boto3.session.Session(
            aws_access_key_id=self.Parameters.s3_access_key.data()[self.Parameters.s3_access_key.default_key],
            aws_secret_access_key=self.Parameters.s3_secret_key.data()[self.Parameters.s3_secret_key.default_key])
        s3_client = s3_session.client(service_name='s3', endpoint_url=S3_ENDPOINT)
        existing_compilers = self.list_existing_compilers(s3_client)

        if archive_name in existing_compilers:
            logging.info('Clang distribution already uploaded to s3, exiting...')
            return

        upload_url = '/'.join([S3_ENDPOINT, S3_BUCKET, COMPILERS_PREFIX + archive_name])
        if self.Parameters.dry_run:
            logging.info('Skipping upload due to dry-run mode. Otherwise '
                         "compiler would've been uploaded to %s", upload_url)
            return

        with tempfile.NamedTemporaryFile() as clang_archive:
            logging.info('Packing clang...')
            with tarfile.open(name=clang_archive.name, mode='w:gz') as archive:
                archive.add(str(clang_path), arcname='.')
            logging.info('Uploading clang to %s', upload_url)
            s3_client.upload_file(clang_archive.name, S3_BUCKET,
                                  COMPILERS_PREFIX + archive_name)
            self.set_info('Uploaded clang to {}'.format(upload_url))

    def on_execute(self):
        self.checkout_repo()
        python3_path = str(self.provide_depot_tools().joinpath('python3'))
        self.checkout_clang(python3_path, self.custom_logs.path.joinpath('fetch_clang.log'))
        clang_path = self.browser_path('src', 'third_party', 'llvm-build', 'Release+Asserts')
        clang_full_name = self.clang_full_name(clang_path)
        self.upload_if_missing('{}.tar.gz'.format(clang_full_name), clang_path)
