import contextlib
import os
from urlparse import urlparse
import shutil

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, ProcessRegistry, subprocess


def _get_session_with_retries():
    session = requests.Session()
    retry = Retry(
        total=5,
        backoff_factor=0.3,
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


class DownloadableToolEnvironment(SandboxEnvironment):
    def __init__(self, tool_name, download_url):
        super(DownloadableToolEnvironment, self).__init__()
        self.tool_name = tool_name
        self.download_url = download_url

    @property
    def extract_dir(self):
        return str(sdk2.Task.current.path(self.tool_name))

    def unpack_archive(self, archive_stream):
        os.makedirs(self.extract_dir)
        archive_name = os.path.basename(urlparse(self.download_url).path)
        archive_path = os.path.join(self.extract_dir, archive_name)
        with open(archive_path, 'wb') as archive_file:
            shutil.copyfileobj(archive_stream, archive_file)
        with ProcessRegistry, ProcessLog(sdk2.Task.current, logger='{}_extract.log'.format(self.tool_name)) as log:
            subprocess.check_call('tar -xf {}'.format(archive_path), cwd=self.extract_dir, shell=True,
                                  stdout=log.stdout, stderr=log.stderr)
        os.unlink(archive_path)

    def prepare(self):
        session = _get_session_with_retries()
        with contextlib.closing(session.get(self.download_url, timeout=(6, 12), stream=True)) as archive_response:
            archive_response.raise_for_status()
            self.unpack_archive(archive_response.raw)


class LLVMToolchainEnvironment(DownloadableToolEnvironment):
    DOWNLOAD_URL = 'https://s3.mds.yandex.net/broinfra-tools/llvm-archives/{}/llvm-{}.tar.xz'

    def __init__(self, platform, version):
        download_url = self.DOWNLOAD_URL.format(platform, version)
        super(LLVMToolchainEnvironment, self).__init__('llvm', download_url)

    def prepare(self):
        super(LLVMToolchainEnvironment, self).prepare()
        inner_directories = os.listdir(self.extract_dir)
        assert len(inner_directories) == 1
        bin_path = os.path.join(self.extract_dir, inner_directories[0], 'bin')
        self.update_os_env('PATH', bin_path)
        os.environ['CC'] = os.path.join(bin_path, 'clang')
        os.environ['CXX'] = os.path.join(bin_path, 'clang++')


class CMakeEnvironment(DownloadableToolEnvironment):
    DOWNLOAD_URL = 'https://s3.mds.yandex.net/broinfra-tools/cmake/cmake-{}-{}.tar.gz'

    def __init__(self, platform, version):
        suffix = {
            'linux': 'linux-x86_64',
            'mac': 'macos-universal',
        }[platform]
        download_url = self.DOWNLOAD_URL.format(version, suffix)
        super(CMakeEnvironment, self).__init__('cmake', download_url)
        self.platform = platform

    def prepare(self):
        super(CMakeEnvironment, self).prepare()
        binary_directory_path = {
            'linux': ('bin', ),
            'mac': ('CMake.app', 'Contents', 'bin', ),
        }[self.platform]
        inner_directories = os.listdir(self.extract_dir)
        assert len(inner_directories) == 1
        self.update_os_env('PATH', os.path.join(self.extract_dir, inner_directories[0], *binary_directory_path))
