import os
import shlex

from sandbox.projects.browser.builds.BuildBrowserTargets import BuildBrowserTargets
from sandbox.projects.browser.common.git import ConfigureGitEnvironment
from sandbox.projects.browser.common.keychain import MacKeychainEnvironment

from sandbox import sdk2


class BuildBrowserBranded(BuildBrowserTargets):
    class Requirements(BuildBrowserTargets.Requirements):
        disk_space = 50 * 1024
        environments = BuildBrowserTargets.Requirements.environments.default + (
            ConfigureGitEnvironment('teamcity@yandex-team.ru', 'Teamcity'),
        )

    class Parameters(BuildBrowserTargets.Parameters):
        # We need up-to 8 hours for app notarization in mac builds
        # and one more external hour for all other stuff.
        kill_timeout = 9 * 60 * 60

        with BuildBrowserTargets.Parameters.general_settings() as general_settings:
            brands = sdk2.parameters.String('Brand names', default='')
            partners = sdk2.parameters.String('Partner names', default='')
            distrib_partners = sdk2.parameters.Bool('Build distribution partners')
            brandings_branch = sdk2.parameters.String('Brandings branch', default='auto-select')

            with BuildBrowserTargets.Parameters.platform.value['default'],\
                    BuildBrowserTargets.Parameters.platform.value['mac']:
                disable_notarization = sdk2.parameters.Bool('Disable mac notarization', default=False)

            with BuildBrowserTargets.Parameters.platform.value['default'],\
                    BuildBrowserTargets.Parameters.platform.value['win']:
                yasigner_token = sdk2.parameters.YavSecret(
                    'YaSigner token secret ID',
                    default='sec-01cn4cz7j3tryd6n540b5ebgkj#yasigner_token')
                add_yadisk = sdk2.parameters.Bool('Add Yandex Disk')
                add_cryptopro = sdk2.parameters.Bool('Add CryptoPro')

    TARGET_MODULE = 'yandex.dist.branded'
    secret_envvars = BuildBrowserTargets.secret_envvars + (
        'KEYCHAIN_KEY',
        'SIGN_CONFIG_PRIVATE_KEY',
        'YANDEX_SIGNER_TOKEN',
    )

    def on_prepare(self):
        super(BuildBrowserBranded, self).on_prepare()
        if self.platform == 'mac':
            self._mac_keychain_env().prepare()

    def script_cmd(self, python_executable):
        cmd = super(BuildBrowserBranded, self).script_cmd(python_executable)

        brands_cmd = ['--brands'] + shlex.split(self.Parameters.brands)
        partners_cmd = ['--partners'] + shlex.split(self.Parameters.partners)
        cmd += [
            '--brandings-branch', self.Parameters.brandings_branch,
        ] + brands_cmd + partners_cmd

        if self.Parameters.distrib_partners:
            cmd.append('--distrib-partners')

        if self.platform == 'mac':
            if self.Parameters.disable_notarization:
                cmd.append('--disable-notarization')
        elif self.platform == 'win':
            if self.Parameters.add_yadisk:
                cmd.append('--add-yadisk')
            if self.Parameters.add_cryptopro:
                cmd.append('--add-cryptopro')

        return cmd

    @property
    def dot_net_path(self):
        path = r'C:\Windows\Microsoft.NET\Framework\v4.0.30319'
        if not os.path.isdir(path):
            raise EnvironmentError('No .NET found in {}'.format(path))
        return path

    def script_extra_env(self):
        env = super(BuildBrowserBranded, self).script_extra_env()
        env.update({
            'SIGN_CONFIG_PRIVATE_KEY': sdk2.Vault.data('browser_sign_config_key'),
        })
        if self.platform == 'mac':
            browser_keychain = self._mac_keychain_env().browser_keychain
            env.update({
                'KEYCHAIN_KEY': browser_keychain.password,
            })
        elif self.platform == 'win':
            env.update({
                'PATH': os.pathsep.join((env.get('PATH', os.environ['PATH']), self.dot_net_path)),
                'YANDEX_SIGNER_TOKEN': str(
                    self.Parameters.yasigner_token.data()[self.Parameters.yasigner_token.default_key]),
            })
        return env

    def _mac_keychain_env(self):
        return MacKeychainEnvironment()
