import shlex

from sandbox.projects.browser.builds.BuildBrowserTargets import BuildBrowserTargets
from sandbox.projects.browser.common.contextmanagers import TmpdirContext
from sandbox.projects.browser.common.bitbucket import BitBucket, DEFAULT_BITBUCKET_URL

from sandbox import sdk2


class BuildSignedIOSBrowser(BuildBrowserTargets):
    TARGET_MODULE = 'yandex.dist.ios.signed_build'

    class Requirements(BuildBrowserTargets.Requirements):
        cores = 12
        disk_space = 110 * 1024

    class Parameters(BuildBrowserTargets.Parameters):
        build_type = sdk2.parameters.String('Build type to run', required=True)
        signing_account = sdk2.parameters.String('Account to use for signed artifacts', required=True)
        dump_libs = sdk2.parameters.Bool('Wheter libs should by published', default=True)
        build_ipa = sdk2.parameters.Bool('Whether .ipa should be build', default=True)
        send_notification_to_st_ticket = sdk2.parameters.Bool('Send notification to st ticket after build', default=False)
        upload_symbols_branch_matcher = sdk2.parameters.String('Branch matcher to upload symbols to AppMetrica after build')
        override_targets = sdk2.parameters.String('Override build targets')
        extra_env = sdk2.parameters.String('Additional environment variables to use while build')

        with sdk2.parameters.Group('Beta upload parameters') as beta_upload_parameters_group:
            upload_to_beta = sdk2.parameters.Bool('Whether built targets should be uploaded')
            beta_signing_account = sdk2.parameters.String('Account to use for uploaded artifacts')
            beta_upload_branches = sdk2.parameters.String('Whitespace separated list of branches to upload to')
            beta_upload_branch_suffixes = sdk2.parameters.String(
                'Whitespace separated list of suffixes for default branches')

        with sdk2.parameters.Group('Credentials') as credentials_group:
            bitbucket_token_vault = sdk2.parameters.Vault(
                'Vault item with oauth token for bitbucket', default='robot-bro-infra_bitbucket_token')

    def script_cmd(self, python_executable):
        cmd = [str(python_executable), '-m', self.TARGET_MODULE, self.Parameters.build_type]
        cmd += ['--signing-account', self.Parameters.signing_account]
        if self.Parameters.dump_libs:
            cmd += ['--dump-libs']
        if self.Parameters.build_ipa:
            cmd += ['--build-ipa']
        if self.Parameters.send_notification_to_st_ticket:
            cmd += ['--send-notification-to-st-ticket']
        if self.Parameters.upload_symbols_branch_matcher:
            cmd += ['--upload-symbols-branch-matcher', self.Parameters.upload_symbols_branch_matcher]
        if self.Parameters.override_targets:
            cmd += ['--override-targets']
            cmd += self.Parameters.override_targets.split()
        if self.Parameters.extra_gn_args:
            cmd += ['--extra-gn-args', self.Parameters.extra_gn_args]

        if self.Parameters.upload_to_beta:
            cmd += ['--upload-to-beta']
            if self.Parameters.beta_signing_account:
                cmd += ['--beta-signing-account', self.Parameters.beta_signing_account]
            if self.Parameters.beta_upload_branches:
                cmd += ['--beta-upload-branches'] + self.Parameters.beta_upload_branches.split()
            else:
                cmd += ['--beta-upload-branch-suffixes'] + self.Parameters.beta_upload_branch_suffixes.split()
        cmd += shlex.split(self.Parameters.extra_args)
        return cmd

    def script_extra_env(self):
        env = super(BuildSignedIOSBrowser, self).script_extra_env()
        env.update(tuple(env_pair.split('=', 1)) for env_pair in self.Parameters.extra_env.split())
        return env

    def run_script(self, python_executable):
        # TODO(BYIN-5350): remove TmpdirContext from here.
        with TmpdirContext('/tmp'):
            super(BuildSignedIOSBrowser, self).run_script(python_executable)

    def on_execute(self):
        bb = BitBucket(DEFAULT_BITBUCKET_URL, 'x-oauth-token', sdk2.Vault.data(self.Parameters.bitbucket_token_vault))
        force_flag_file = 'src/build/yandex/dist/ios/force_sandbox_task'
        if bb.file_exists('stardust', 'browser', path=force_flag_file, commit=self.Parameters.commit or None,
                          branch=self.Parameters.branch):
            super(BuildSignedIOSBrowser, self).on_execute()
