import contextlib
import logging
import os
import re
import platform
import shutil
import socket
import tarfile
import time

import requests

from sandbox.common import errors, utils
import sandbox.common.types.misc as ctm

from sandbox.projects.browser.common import binary_tasks
from sandbox.projects.browser.common import SecretEnv
from sandbox.projects.browser.common.contextmanagers import ExitStack
from sandbox.projects.browser.common.depot_tools import DepotToolsEnvironment
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment
from sandbox.projects.common.decorators import retries
from sandbox.projects.common.teamcity import TeamcityArtifactsContext, TeamcityServiceMessagesLog

from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, ProcessRegistry, subprocess

_TC_LOG_NAME_BUILD = 'build'
RETRY_EXIT_CODE = 57


class RunBrowserScript(binary_tasks.CrossPlatformBinaryTaskMixin, sdk2.Task):
    """
    Run custom script in browser repository.
    """
    _PYTHON_VERSION = '3.9.7'
    _PIP_VERSION = '20.3.4'

    class Requirements(sdk2.Task.Requirements):
        disk_space = 20 * 1024  # Approximate size of browser repository.
        dns = ctm.DnsType.DNS64
        environments = (
            GitEnvironment('2.32.0'),
        )

    class Parameters(sdk2.Task.Parameters):
        # Skip creating disk_usage.yaml and peak_disk_usage.yaml because we do not use them,
        # and dumping is slow sometimes (especially on Windows).
        dump_disk_usage = False

        with sdk2.parameters.Group('Repositories settings') as repositories_settings:
            branch = sdk2.parameters.String('Branch to checkout on', default='master')
            commit = sdk2.parameters.String('Commit to checkout on')
            use_test_bitbucket = sdk2.parameters.Bool('Use test BitBucket')
            use_archive_checkout = sdk2.parameters.Bool(
                'Download repo files via rest/api/archive instead of git checkout',
                default=True
            )

        with sdk2.parameters.Group('General settings') as general_settings:
            with sdk2.parameters.String('Target platform') as platform:
                platform.values.default = platform.Value('default (host OS)', default=True)
                platform.values.android = platform.Value('android')
                platform.values.ios = platform.Value('ios')
                platform.values.linux = platform.Value('linux')
                platform.values.mac = platform.Value('mac')
                platform.values.win = platform.Value('win')

        with sdk2.parameters.Group('Debug settings') as debug_settings:
            with sdk2.parameters.String('Suspend after', ui=sdk2.parameters.String.UI('select')) as suspend_after:
                suspend_after.values.no = suspend_after.Value('No', default=True)
                suspend_after.values.start = suspend_after.Value('Start')
                suspend_after.values.checkout = suspend_after.Value('Checkout')
                suspend_after.values.depot_tools = suspend_after.Value('Providing depot_tools')
                suspend_after.values.gclient_sync = suspend_after.Value('GClient sync')
                suspend_after.values.hpe = suspend_after.Value('Providing HPE')
                suspend_after.values.script = suspend_after.Value('Running script')

        _binary_task_params = binary_tasks.cross_platform_binary_task_parameters()

    class Context(sdk2.Task.Context):
        statistics_from_log = {}
        # Extra stats that will be reported as Teamcity build statistics by plugin.
        teamcity_build_statistics = {}

    git_browser_sparse_checkout_paths = None
    bitbucket_browser_sparse_checkout_paths = None
    secret_envvars = ()

    def browser_path(self, *args):
        return self.path('browser', *args)

    @property
    def platform(self):
        if self.Parameters.platform == 'default':
            return {
                'Darwin': 'mac',
                'Linux': 'linux',
                'Windows': 'win',
            }[platform.system()]
        else:
            return self.Parameters.platform

    @contextlib.contextmanager
    def step(self, name, log_name, duration_statistic=None):
        self.set_info(name)
        start_time = time.time()

        tac = TeamcityArtifactsContext(
            path_suffix='step-{}'.format(log_name), log_name=log_name,
            tc_service_messages_description='Step[{}]'.format(name))
        with tac:
            try:
                yield
            except errors.TemporaryError:
                tac.logger.exception('Step "{}" failed - TemporaryError'.format(name))
                raise
            except Exception:
                tac.logger.exception('Step "{}" failed'.format(name))
                tac.logger.info("##teamcity[buildProblem description='{}']".format(
                    'Sandbox step "{}" failed.'.format(name)))
                raise
            finally:
                if duration_statistic:
                    self.Context.teamcity_build_statistics[duration_statistic] = time.time() - start_time

    @retries(5, delay=5, backoff=2, exceptions=(socket.error,
                                                requests.exceptions.ConnectionError,
                                                requests.exceptions.Timeout))
    def get_and_unpack_archive(self):
        with contextlib.closing(
            self.bb.get_archive('STARDUST', 'browser',
                                at=self.checkout_ref,
                                archive_format='tgz',
                                path=self.bitbucket_browser_sparse_checkout_paths)
        ) as archive_response:
            with tarfile.open(fileobj=archive_response.raw, mode='r|*') as repo_archive:
                repo_archive.extractall(str(self.browser_path()))

    def checkout_repositories(self):
        if self.Parameters.use_archive_checkout and self.bitbucket_browser_sparse_checkout_paths is not None:
            try:
                self.get_and_unpack_archive()
            except (socket.error, requests.exceptions.ConnectionError, requests.exceptions.Timeout):
                raise errors.TemporaryError('Archive checkout error')
        else:
            repositories.Stardust.browser(filter_branches=False, testing=self.Parameters.use_test_bitbucket).clone(
                str(self.browser_path()), self.Parameters.branch, self.Parameters.commit,
                sparse_checkout_paths=self.git_browser_sparse_checkout_paths,
            )

    @property
    def bb(self):
        raise NotImplementedError()

    @utils.singleton_property
    def checkout_ref(self):
        checkout_ref = self.Parameters.commit or self.Parameters.branch
        if not checkout_ref:
            raise ValueError("A branch or commit should be specified.")
        return checkout_ref

    def clean_files(self):
        logging.debug('Cleaning %s', self.browser_path())
        # Remove browser repository manually, because Sandbox does it too slowly
        # (it iterates through all files and check if file is resource).
        shutil.rmtree(str(self.browser_path()), ignore_errors=True)
        logging.debug('Cleaning finished')

    def validate_deps_snapshot(self):
        """
        Fail if .DEPS.snapshot contains conflict markers (that can be in merge branches).
        Use `TaskFailure` instead of `TaskError` because it is regular error.
        """
        deps_snapshot_content = self.browser_path('src', '.DEPS.snapshot').read_bytes()
        if re.search('<<<<<<<.*=======.*>>>>>>>', deps_snapshot_content, flags=re.DOTALL):
            raise errors.TaskFailure('.DEPS.snapshot contains conflict markers')

    def provide_depot_tools(self):
        depot_tools_env = DepotToolsEnvironment(
            deps_file=str(self.browser_path('src', '.DEPS.snapshot')),
            dep_name='src/third_party/depot_tools',
        )
        depot_tools_env.prepare()
        return depot_tools_env.depot_tools_folder

    def depends_on_yin(self):
        return not self.browser_path('src', 'build', 'yandex', 'yin_legacy', 'no_yin_dep.flag').is_file()

    @property
    def gclient_cache_path(self):
        return SandboxEnvironment.exclusive_build_cache_dir('gclient')

    def gclient_sync_extra_env(self):
        return {
            'CIPD_CACHE_DIR': SandboxEnvironment.exclusive_build_cache_dir('cipd_cache'),
            'GCLIENT_CACHE_DIR': self.gclient_cache_path,
            'GIT_CACHE_PATH': self.gclient_cache_path,
            'USE_CIPD_PROXY': '1',
        }

    def gclient_sync(self, depot_tools_dir):
        gclient_name = 'gclient.bat' if os.name == 'nt' else 'gclient'
        gclient_path = depot_tools_dir.joinpath(gclient_name)
        deps_platform = 'unix' if self.platform == 'linux' else self.platform
        cmd = [
            str(gclient_path), 'sync',
            '--nohooks',
            '--verbose',
            '--force',
            '--deps={}'.format(deps_platform),
            '--ignore_locks',
        ]

        extra_env = self.gclient_sync_extra_env()
        logging.info('GClient sync extra environment:\n%r', SecretEnv(extra_env, self.secret_envvars))
        env = dict(os.environ, **extra_env)

        with ProcessLog(self, logger='browser_gclient_sync') as log:
            subprocess.check_call(cmd, cwd=str(self.browser_path()), stdout=log.stdout, stderr=log.stdout, env=env)

    def provide_hpe(self, exit_stack):
        requirements_files = [self.browser_path('requirements.txt')]
        hpe = HermeticPythonEnvironment(
            python_version=self._PYTHON_VERSION,
            pip_version=self._PIP_VERSION,
            requirements_files=requirements_files,
        )
        exit_stack.enter_context(hpe)
        return hpe.python_executable

    def script_cmd(self, python_executable):
        """
        :type python_executable: sdk2.Path
        :rtype: list[str]
        """
        raise NotImplementedError()

    def script_cwd(self):
        return self.browser_path()

    def script_extra_env(self):
        return {
            # Do not pass PYTHONPATH of Sandbox task to the child script.
            'PYTHONPATH': str(self.browser_path('src', 'build')),
        }

    def _find_script_tc_log(self):
        return TeamcityServiceMessagesLog.find(
            task=self,
            attrs=dict(log_name=_TC_LOG_NAME_BUILD)
        ).first()

    def run_script(self, python_executable):
        extra_env = self.script_extra_env()
        logging.info('Script extra environment:\n%r', SecretEnv(extra_env, self.secret_envvars))
        env = dict(os.environ, **extra_env)

        # On Windows, run process in another group in order to support gracefully killing
        # (see `GracefulKillBeforeTimeoutMixin`).
        creation_flags = subprocess.CREATE_NEW_PROCESS_GROUP if os.name == 'nt' else 0

        secret_tokens = SecretEnv(env, self.secret_envvars).secret.values()
        tac = TeamcityArtifactsContext(self.script_cwd(), secret_tokens=secret_tokens, log_name=_TC_LOG_NAME_BUILD)
        try:
            with tac, ProcessRegistry:
                subprocess.check_call(
                    self.script_cmd(python_executable),
                    cwd=str(self.script_cwd()),
                    stdout=tac.output, stderr=subprocess.STDOUT,
                    env=env,
                    creationflags=creation_flags,
                )
        except subprocess.CalledProcessError as error:
            if error.returncode == RETRY_EXIT_CODE:
                raise errors.TemporaryError('Script: {} ended with {} (Retry) exit code'
                                            .format(python_executable, RETRY_EXIT_CODE))
            else:
                raise
        finally:
            self.Context.statistics_from_log.update(tac.build_statistics)

    def on_execute_impl(self, exit_stack):
        if self.Parameters.suspend_after == 'start':
            self.suspend()

        with self.step('Checkout repositories', 'checkout', 'checkoutDuration'):
            self.checkout_repositories()
        if self.Parameters.suspend_after == 'checkout':
            self.suspend()

        with self.step('Validate .DEPS.snapshot', 'validate-deps'):
            self.validate_deps_snapshot()
        with self.step('Provide depot_tools', 'depot-tools', 'provideDepotToolsExecutionTime'):
            depot_tools_dir = self.provide_depot_tools()
        if self.Parameters.suspend_after == 'depot_tools':
            self.suspend()

        if self.depends_on_yin():
            with self.step('GClient sync', 'gclient-sync', 'gclientSyncDuration'):
                self.gclient_sync(depot_tools_dir)
        if self.Parameters.suspend_after == 'gclient_sync':
            self.suspend()

        with self.step('Provide Hermetic Python Environment', 'hpe', 'provideHPEExecutionTime'):
            python_executable = self.provide_hpe(exit_stack)
        if self.Parameters.suspend_after == 'hpe':
            self.suspend()

        with self.step('Run script', 'run'):
            self.run_script(python_executable)
        if self.Parameters.suspend_after == 'script':
            self.suspend()

    def on_execute(self):
        with ExitStack() as stack:
            self.on_execute_impl(stack)
        self.set_info('Execution finished')

    def on_finish(self, prev_status, status):
        self.clean_files()
        super(RunBrowserScript, self).on_finish(prev_status, status)

    def on_break(self, prev_status, status):
        self.clean_files()
        super(RunBrowserScript, self).on_break(prev_status, status)
