import contextlib
import logging
import os
import time

import sandbox.common.types.misc as ctm

from sandbox.projects.browser.common import binary_tasks
from sandbox.projects.browser.common import SecretEnv
from sandbox.projects.browser.common.contextmanagers import ExitStack
from sandbox.projects.browser.common.depot_tools import DepotToolsEnvironment
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment
from sandbox.projects.common.teamcity import TeamcityArtifactsContext, TeamcityServiceMessagesLog

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessRegistry, subprocess

_TC_LOG_NAME_BUILD = 'build'


class RunYinScript(binary_tasks.CrossPlatformBinaryTaskMixin, sdk2.Task):
    """
    Run custom script in yin repository.
    """
    _PYTHON_VERSION = '3.9.7'
    _PIP_VERSION = '20.3.4'

    class Requirements(sdk2.Task.Requirements):
        disk_space = 5 * 1024
        dns = ctm.DnsType.DNS64
        environments = (
            GitEnvironment('2.24.1'),
        )

    class Parameters(sdk2.Task.Parameters):
        # Skip creating disk_usage.yaml and peak_disk_usage.yaml because we do not use them,
        # and dumping is slow sometimes (especially on Windows).
        dump_disk_usage = False

        with sdk2.parameters.Group('Repositories settings') as repositories_settings:
            branch = sdk2.parameters.String('Branch to checkout on', default='master')
            commit = sdk2.parameters.String('Commit to checkout on')
            depot_tools_revision = sdk2.parameters.String('depot_tools revision', default='master')

        with sdk2.parameters.Group('Debug settings') as debug_settings:
            with sdk2.parameters.String('Suspend after', ui=sdk2.parameters.String.UI('select')) as suspend_after:
                suspend_after.values.no = suspend_after.Value('No', default=True)
                suspend_after.values.start = suspend_after.Value('Start')
                suspend_after.values.checkout = suspend_after.Value('Checkout')
                suspend_after.values.depot_tools = suspend_after.Value('Providing depot_tools')
                suspend_after.values.hpe = suspend_after.Value('Providing HPE')
                suspend_after.values.script = suspend_after.Value('Running script')

        _binary_task_params = binary_tasks.cross_platform_binary_task_parameters()

    class Context(sdk2.Task.Context):
        statistics_from_log = {}
        # Extra stats that will be reported as Teamcity build statistics by plugin.
        teamcity_build_statistics = {}

    secret_envvars = ()
    PROVIDE_DEPOT_TOOLS = False

    def yin_path(self, *args):
        return self.path('yin', *args)

    @contextlib.contextmanager
    def step(self, name, log_name, duration_statistic=None):
        self.set_info(name)
        start_time = time.time()

        tac = TeamcityArtifactsContext(
            path_suffix='step-{}'.format(log_name), log_name=log_name,
            tc_service_messages_description='Step[{}]'.format(name))
        with tac:
            try:
                yield
            except Exception:
                tac.logger.exception('Step "{}" failed'.format(name))
                tac.logger.info("##teamcity[buildProblem description='{}']".format(
                    'Sandbox step "{}" failed.'.format(name)))
                raise
            finally:
                if duration_statistic:
                    self.Context.teamcity_build_statistics[duration_statistic] = time.time() - start_time

    def checkout_repositories(self):
        repositories.Stardust.yin(filter_branches=False).clone(
            str(self.yin_path()), self.Parameters.branch, self.Parameters.commit)

    def provide_depot_tools(self):
        DepotToolsEnvironment(self.Parameters.depot_tools_revision).prepare()

    def requirements_files(self):
        return [
            self.yin_path('requirements.txt'),
        ]

    def provide_hpe(self, exit_stack):
        hpe = HermeticPythonEnvironment(
            python_version=self._PYTHON_VERSION,
            pip_version=self._PIP_VERSION,
            requirements_files=self.requirements_files(),
        )
        exit_stack.enter_context(hpe)
        return hpe.python_executable

    def script_cmd(self, python_executable):
        """
        :type python_executable: sdk2.Path
        :rtype: list[str]
        """
        raise NotImplementedError()

    def script_cwd(self):
        return self.yin_path()

    def script_extra_env(self):
        return {
            # Do not pass PYTHONPATH of Sandbox task to the child script.
            'PYTHONPATH': '',
        }

    def _find_script_tc_log(self):
        return TeamcityServiceMessagesLog.find(
            task=self,
            attrs=dict(log_name=_TC_LOG_NAME_BUILD)
        ).first()

    def handle_script_error(self, error):
        """
        Custom handler for script errors. Note that error will be raised after it, but it is possible to raise
        another exception (such as `TaskFailure`) instead.

        :type error: subprocess.CalledProcessError
        """
        pass

    def run_script(self, python_executable):
        extra_env = self.script_extra_env()
        logging.info('Script extra environment:\n%r', SecretEnv(extra_env, self.secret_envvars))
        env = dict(os.environ, **extra_env)

        secret_tokens = SecretEnv(env, self.secret_envvars).secret.values()
        tac = TeamcityArtifactsContext(self.script_cwd(), secret_tokens=secret_tokens, log_name=_TC_LOG_NAME_BUILD)
        try:
            with tac, ProcessRegistry:
                try:
                    subprocess.check_call(
                        self.script_cmd(python_executable),
                        cwd=str(self.script_cwd()),
                        stdout=tac.output, stderr=subprocess.STDOUT,
                        env=env,
                    )
                except subprocess.CalledProcessError as e:
                    self.handle_script_error(e)
                    raise
        finally:
            self.Context.statistics_from_log.update(tac.build_statistics)

    def on_execute_impl(self, exit_stack):
        if self.Parameters.suspend_after == 'start':
            self.suspend()

        with self.step('Checkout repositories', 'checkout', 'checkoutDuration'):
            self.checkout_repositories()
        if self.Parameters.suspend_after == 'checkout':
            self.suspend()

        if self.PROVIDE_DEPOT_TOOLS:
            with self.step('Provide depot_tools', 'depot-tools', 'provideDepotToolsExecutionTime'):
                self.provide_depot_tools()
        if self.Parameters.suspend_after == 'depot_tools':
            self.suspend()

        with self.step('Provide Hermetic Python Environment', 'hpe', 'provideHPEExecutionTime'):
            python_executable = self.provide_hpe(exit_stack)
        if self.Parameters.suspend_after == 'hpe':
            self.suspend()

        with self.step('Run script', 'run'):
            self.run_script(python_executable)
        if self.Parameters.suspend_after == 'script':
            self.suspend()

    def on_execute(self):
        with ExitStack() as stack:
            self.on_execute_impl(stack)
        self.set_info('Execution finished')
