import errno
import functools
import inspect
import logging
import os
import shutil
import stat

import six

LXC_RESOURCE_ID = 1255740695


def log_calls(log_args=True, args_format=None, log_result=False):
    def _log_calls(function):
        """This is a decorator, that writes message about calling decorated
        function to output.
        """
        @functools.wraps(function)
        def __log_calls(*args, **kwargs):
            def args_string():
                if not log_args:
                    return ''
                if args_format:
                    return args_format.format(**inspect.getcallargs(
                        function, *args, **kwargs))
                args_str = ', '.join('{0}'.format(repr(a)) for a in args)
                kwargs_str = ', '.join('{0}={1}'.format(k, repr(v))
                                       for k, v in kwargs.items())
                return ', '.join(s for s in [args_str, kwargs_str] if s)
            if log_result:
                result = function(*args, **kwargs)
                logging.info('{f}({a})\nresult:{r}'.format(
                    f=function.__name__, a=args_string(), r=result))
                return result
            else:
                logging.info('{f}({a})'.format(
                    f=function.__name__, a=args_string()))
                return function(*args, **kwargs)
        __log_calls.nolog = function
        return __log_calls

    return _log_calls


def rmtree_forcedly(path):
    """
    Remove directory, trying to add write permissions if necessary.
    """
    def chmod(path, mode):
        os.chmod(path, mode)
        logging.debug('Changed permissions of %s to %d', path, mode)

    def onerror(func, path, exc_info):
        if func in (os.rmdir, os.remove) and exc_info[1].errno == errno.EACCES:
            chmod(path, os.stat(path).st_mode | stat.S_IWRITE)
            func(path)
        else:
            six.reraise(*exc_info)

    shutil.rmtree(path, onerror=onerror)


class SecretEnv(object):
    """
    Helper class for getting secret and nonsecret environment variables.
    """
    def __init__(self, env, secret_keys):
        self._env = env
        if os.name == 'nt':
            # On Windows, envvar keys are case-insensitive.
            self._secret_keys = {k.lower() for k in secret_keys} | {k.upper() for k in secret_keys}
        else:
            self._secret_keys = set(secret_keys)

    @property
    def secret(self):
        return {
            key: value
            for key, value in self._env.items()
            if key in self._secret_keys
        }

    @property
    def nonsecret(self):
        return {
            key: value
            for key, value in self._env.items()
            if key not in self._secret_keys
        }

    def __repr__(self):
        env = {
            key: '*****'
            for key in self.secret
        }
        env.update(self.nonsecret)
        return '\n'.join(
            '{}={}'.format(k, v)
            for k, v in sorted(env.items())
        )
