import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt

from sandbox import sdk2

DEFAULT_LXC_CONTAINER_RESOURCE_ID = 3183752982


class BasePlatformBinaryTaskMixin(object):
    """
    Base class for binary task mixins. Do not use it, only inherit from it.
    """
    def find_binary_tasks_resource(self, platform):
        return sdk2.service_resources.SandboxTasksBinary.find(
            arch=platform,
            attrs={
                'release': ctt.ReleaseStatus.STABLE,
                'task_{}'.format(self.__class__.name): True,
            },
            state=ctr.State.READY,
        ).first()


class OnePlatformBinaryTaskMixin(BasePlatformBinaryTaskMixin):
    """
    Mixin "abstract" class to automatically find suitable task binary that works on one platform only.

    Abstract class, do not use it, only inherit from it.

    You can use WinBinaryTaskMixin, MacBinaryTaskMixin, or LinuxBinaryTaskMixin

    Usage example:
    >>> class MyTask(WinBinaryTaskMixin, sdk2.Task):
    >>> ...
    """
    @property
    def update_resource(self):
        return False

    @property
    def platform(self):
        raise NotImplementedError

    def on_enqueue(self):
        super(OnePlatformBinaryTaskMixin, self).on_enqueue()
        if self.update_resource or self.Requirements.tasks_resource is None:
            resource = self.find_binary_tasks_resource(self.platform)
            if resource:
                self.Requirements.tasks_resource = resource
            else:
                raise RuntimeError('No tasks resource found')


class LinuxBinaryTaskMixin(OnePlatformBinaryTaskMixin):
    platform = ctm.OSFamily.LINUX


class MacBinaryTaskMixin(OnePlatformBinaryTaskMixin):
    platform = ctm.OSFamily.OSX


class WinBinaryTaskMixin(OnePlatformBinaryTaskMixin):
    platform = ctm.OSFamily.WIN_NT


def cross_platform_binary_task_parameters(linux=True, mac=True, win=True):
    if not any((linux, mac, win)):
        raise ValueError('At least one platform should be enabled')

    class _CrossPlatformBinaryTaskParameters(sdk2.Parameters):
        with sdk2.parameters.Group('Binary task settings') as binary_task_settings:
            with sdk2.parameters.String('Use last stable task binary for',
                                        ui=sdk2.parameters.String.UI('select')) as binary_platform:
                binary_platform.values.none = binary_platform.Value('None', default=True)
                if linux:
                    binary_platform.values[ctm.OSFamily.LINUX] = binary_platform.Value('linux')
                if mac:
                    binary_platform.values[ctm.OSFamily.OSX] = binary_platform.Value('mac')
                if win:
                    binary_platform.values[ctm.OSFamily.WIN_NT] = binary_platform.Value('win')

    return _CrossPlatformBinaryTaskParameters()


class CrossPlatformBinaryTaskMixin(BasePlatformBinaryTaskMixin):
    """
    Mixin class to automatically find suitable task binary of specified platform.
    Should be used together with `cross_platform_binary_task_parameters`.

    Usage example:
    >>> class MyTask(CrossPlatformBinaryTaskMixin, sdk2.Task):
    >>>    class Parameters(sdk2.Task.Parameters):
    >>>        ...
    >>>        # This task is only for Linux and Mac.
    >>>        _binary_task_params = cross_platform_binary_task_parameters(win=False)
    """

    def on_enqueue(self):
        super(CrossPlatformBinaryTaskMixin, self).on_enqueue()
        if self.Parameters.binary_platform != 'none' and self.Requirements.tasks_resource is None:
            resource = self.find_binary_tasks_resource(self.Parameters.binary_platform)
            if resource:
                self.Requirements.tasks_resource = resource
            else:
                raise RuntimeError('No tasks resource found')
