import copy
import logging

import requests
import six.moves.urllib_parse as urllib_parse

from sandbox.projects.common.decorators import retries


logger = logging.getLogger(__name__)
DEFAULT_BITBUCKET_HOST = 'bitbucket.browser.yandex-team.ru'
DEFAULT_BITBUCKET_URL = 'https://{}/'.format(DEFAULT_BITBUCKET_HOST)
TESTING_BITBUCKET_HOST = 'bitbucket.test.browser.yandex-team.ru'
TESTING_BITBUCKET_URL = 'https://{}/'.format(TESTING_BITBUCKET_HOST)

RESTRICTION_TYPES = (
    READ_ONLY,
    NO_DELETES,
    FAST_FORWARD_ONLY,
    PULL_REQUESTS_ONLY,
) = (
    'read-only',
    'no-deletes',
    'fast-forward-only',
    'pull-request-only',
)

RESTRICTION_MATCHER_TYPES = (
    BRANCH,
    PATTERN,
    MODEL_CATEGORY,
    MODEL_BRANCH,
) = (
    'BRANCH',
    'PATTERN',
    'MODEL_CATEGORY',
    'MODEL_BRANCH',
)


def _to_refs_heads(ref):
    refs_prefix = 'refs/heads/'
    return ref if ref.startswith(refs_prefix) else refs_prefix + ref


class ResourceNotFound(Exception):
    pass


class Unauthorized(Exception):
    pass


class RequestError(Exception):
    pass


class PullRequest(object):
    def __init__(self, response):
        super(PullRequest, self).__init__()
        self.response = response

    def __getattr__(self, key):
        return self.response[key]

    @property
    def web_url(self):
        return self.response['links']['self'][0]['href']


class BitBucket(object):
    def __init__(self, url, login, password):
        self.url = url
        self.session = requests.Session()
        self.session.auth = (login, password)

    @retries(5, delay=5, backoff=2, exceptions=(Unauthorized, RequestError))
    def request(self, method, path, rest_api='api', api_version='1.0', raw_response=False, **kwargs):
        base_url = urllib_parse.urljoin(self.url, 'rest/{}/{}/'.format(rest_api, api_version))
        url = urllib_parse.urljoin(base_url, path)
        logging.debug('Requesting %s', url)
        response = self.session.request(method, url, stream=raw_response, **kwargs)
        if response.status_code == 404:
            raise ResourceNotFound(path)
        elif response.status_code == 401:
            raise Unauthorized('401 Unauthorized')
        elif response.status_code >= 400:
            logger.error('%s %s %s Failed.', method, url, kwargs.get('json'))
            raise RequestError('{} {}'.format(response.status_code, response.json()))
        if raw_response:
            return response
        try:
            return response.json()
        except:
            return response.text

    def paginate(self, path, key='values', params=None, **kwargs):
        if params is None:
            params = {}
        last_page = False
        start = 0
        while not last_page:
            request_params = copy.copy(params)
            request_params['start'] = start
            page = self.request('GET', path, params=request_params, **kwargs)
            last_page = page.get('isLastPage', True)
            start = start + len(page.get(key, []))
            for value in page.get(key, []):
                yield value

    def load_file(self, project, repo, path, at=None):
        params = {}
        if at is not None:
            params['at'] = at
        rest_path = 'projects/{}/repos/{}/browse/{}'.format(
            project, repo, path)

        content = ''
        for line in self.paginate(rest_path, key='lines', params=params):
            if content:
                content += '\n'
            content += line['text']
        return content

    def file_exists(self, project, repo, path, commit=None, branch=None):
        params = {
            'type': 'true',
        }
        if commit is not None:
            params['at'] = commit
        elif branch is not None:
            params['at'] = _to_refs_heads(branch)
        rest_path = 'projects/{}/repos/{}/browse/{}'.format(
            project, repo, path)
        try:
            exists = self.request('GET', rest_path, params=params).get('type') == 'FILE'
        except ResourceNotFound:
            return False
        return exists

    def get_branches(self, project, repo, filter_text=None, order_by='ALPHABETICAL'):
        rest_path = 'projects/{}/repos/{}/branches'.format(
            project, repo)
        params = {'filterText': filter_text, 'orderBy': order_by}
        return self.paginate(rest_path, params=params)

    def create_branch(self, project, repo, branch_name, start_point):
        rest_path = 'projects/{}/repos/{}/branches'.format(
            project, repo)
        return self.request('POST', rest_path, json={
            'name': _to_refs_heads(branch_name),
            'startPoint': _to_refs_heads(start_point)
        })

    def get_repo(self, project, repo):
        rest_path = 'projects/{}/repos/{}'.format(project, repo)
        return self.request('GET', rest_path)

    def get_repo_clone_url(self, project, repo, name='ssh'):
        repo = self.get_repo(project, repo)
        for link in repo['links']['clone']:
            if link['name'] == name:
                if name == 'http':
                    return self.drop_credentials_from_url(link['href'])
                return link['href']

        for link in repo['links']['clone']:
            if name == 'http':
                return self.drop_credentials_from_url(link['href'])
            return link['href']

        raise Exception('No clone links.')

    def get_commits(self, project, repo, since=None, until=None, limit=25, path=None):
        rest_path = 'projects/{}/repos/{}/commits'.format(
            project, repo)
        params = {
            'limit': limit,
            'since': since,
            'until': until,
            'path': path,
        }
        return self.paginate(rest_path, params=params)

    def get_latest_commit(self, project, repo, branch=None):
        rest_path = 'projects/{}/repos/{}/commits'.format(
            project, repo)
        params = {'until': branch,
                  'limit': 1
                  }
        commits = self.request('GET', rest_path, params=params)['values']
        if not commits:
            return None
        return commits[0]['id']

    def get_commit(self, project, repo, commit):
        rest_path = 'projects/{}/repos/{}/commits/{}'.format(
            project, repo, commit)
        return self.request('GET', rest_path)

    def get_tag(self, project, repo, tag):
        rest_path = 'projects/{}/repos/{}/tags/{}'.format(
            project, repo, tag)
        return self.request('GET', rest_path)

    def merge_base(self, project, repo, commits, octopus=False, all=False):
        rest_path = 'projects/{}/repos/{}/merge-base'.format(
            project, repo)
        params = {
            'commits': ','.join(commits),
            'all': all
        }

        if octopus:
            params['octopus'] = 'true'
        return self.request('GET', rest_path, params=params, rest_api='git-extras')

    def compare_commits(self, project, repo, from_ref, to_ref):
        rest_path = 'projects/{}/repos/{}/compare/commits'.format(
            project, repo)
        params = {
            'from': from_ref,
            'to': to_ref,
        }
        return self.request('GET', rest_path, params=params)['values']

    def create_pr(self, project, repo, title, description, from_ref, to_ref,
                  reviewers):
        rest_path = 'projects/{}/repos/{}/pull-requests'.format(
            project, repo)
        return PullRequest(self.request('POST', rest_path, json={
            'title': title,
            'description': description,
            'fromRef': {'id': _to_refs_heads(from_ref)},
            'toRef': {'id': _to_refs_heads(to_ref)},
            'reviewers': [
                {'user': {'name': name}} for name in reviewers
            ],
        }))

    def pull_requests(self, project, repository, at=None):
        rest_path = 'projects/{}/repos/{}/pull-requests'.format(
            project, repository)
        if at:
            rest_path = '{}?at={}'.format(rest_path, _to_refs_heads(at))
        return [PullRequest(pr) for pr in self.paginate(rest_path)]

    def get_pr(self, project, repo, pr_id):
        rest_path = 'projects/{}/repos/{}/pull-requests/{}'.format(
            project, repo, pr_id)
        return PullRequest(self.request('GET', rest_path))

    def get_pull_request_builds(self, project, repo, pr_id, commit=None):
        if not commit:
            commit = self.get_pull_request_merge_pin(project, repo, pr_id)
            if not commit:
                return []
        rest_path = 'projects/{}/repos/{}/pull-requests/{}/builds/commits/{}'.format(
            project, repo, pr_id, commit)
        response = self.request('GET', rest_path, rest_api='teamcity')
        return response['values']

    def get_pull_request_merge_pin(self, project, repo, pr_id):
        rest_path = 'projects/{}/repos/{}/pull-requests/{}/builds/commits'.format(
            project, repo, pr_id)
        response = self.request('GET', rest_path, rest_api='teamcity')
        for commit_info in response:
            if 'merge-pin' in commit_info['reference']:
                return commit_info['commit']
        return None

    def delete_merge_pins_in_branch(self, project, repo, branch, run_builds=False):
        rest_path = 'projects/{}/repos/{}/merge-pin'.format(project, repo)
        return self.request('DELETE', rest_path, rest_api='teamcity', data={'branch': branch, 'runBuilds': run_builds})

    def pull_request_changes(self, project, repository, pr):
        rest_path = 'projects/{}/repos/{}/pull-requests/{}/changes'.format(
            project, repository, pr)
        return self.request('GET', rest_path, rest_api='teamcity')

    def pull_request_activities(self, project, repository, pr):
        rest_path = 'projects/{}/repos/{}/pull-requests/{}/activities'.format(
            project, repository, pr)
        return list(self.paginate(rest_path))

    def comment_pull_request(self, project, repo, pull_request_id, text,
                             parent_comment_id=None):
        rest_path = 'projects/{}/repos/{}/pull-requests/{}/comments'.format(
            project, repo, pull_request_id)
        request = {'text': text}
        if parent_comment_id:
            request.update({'parent': {'id': parent_comment_id}})
        return self.request('POST', rest_path, json=request)

    def press_run_builds(self, project, repository, pr):
        rest_path = 'projects/{}/repos/{}/pull-requests/{}/builds'.format(
            project, repository, pr)
        return self.request('POST', rest_path, rest_api='teamcity')

    def drop_credentials_from_url(self, url):
        """
        BitBucket returns username inside of http clone urls. Drop it from url.
        """

        url_parsed = urllib_parse.urlparse(url)
        netloc = url_parsed.netloc.rsplit('@', 1)[-1]
        return urllib_parse.urlunparse(
            (url_parsed.scheme, netloc) + url_parsed[2:])

    def get_repository_labels(self, project, repository):
        rest_path = 'projects/{}/repos/{}/labels'.format(project, repository)
        return list(self.paginate(rest_path))

    def create_repository_label(self, project, repository, label):
        rest_path = 'projects/{}/repos/{}/labels'.format(project, repository)
        request = {'name': label}
        return self.request('POST', rest_path, json=request)

    def get_restrictions(self, project, repository, restriction_type=None, matcher_type=None, matcher_id=None):
        rest_path = 'projects/{}/repos/{}/restrictions'.format(project, repository)
        params = {}
        if restriction_type:
            params['type'] = restriction_type
        if matcher_type:
            params['matcherType'] = matcher_type
        if matcher_id:
            params['matcherId'] = matcher_id
        return list(self.paginate(rest_path, params=params, rest_api='branch-permissions', api_version='2.0'))

    #  https://docs.atlassian.com/DAC/rest/stash/3.11.3/stash-branch-permissions-rest.html
    def create_restriction(self, project, repository, restriction_type, matcher_type, matcher,
                           users=None, groups=None, accessKeys=None):
        rest_path = 'projects/{}/repos/{}/restrictions'.format(project, repository)
        data = {
            'type': restriction_type,
            'matcher': {
                'id': matcher,
                'type': {
                    'id': matcher_type,
                },
            },
            'users': users or [],
            'groups': groups or [],
            'accessKeys': accessKeys or [],
        }
        return self.request('POST', rest_path, rest_api='branch-permissions', api_version='2.0', json=data)

    def delete_restriction(self, project, repository, restriction_id):
        rest_path = 'projects/{}/repos/{}/restrictions/{}'.format(project, repository, restriction_id)
        return self.request('DELETE', rest_path, rest_api='branch-permissions', api_version='2.0')

    def get_owners(self, project, repo, commit, paths, limit=4, shadows=False, sort_with_absense=False):
        params = {'limit': limit,
                  'shadows': shadows,
                  'sortWithAbsense': str(sort_with_absense).lower()}
        rest_path = 'projects/{project}/repos/{repo}/commits/{commit}/owners'.format(
            project=project,
            repo=repo,
            commit=commit)
        return self.request('POST', rest_path, params=params, rest_api='owners', api_version='1.0', json=paths)

    def get_archive(self, project, repository, at=None, archive_format='zip', path=None):
        params = {
            'at': at,
            'format': archive_format,
            'path': path,
        }
        rest_path = 'projects/{}/repos/{}/archive'.format(project, repository)

        return self.request('GET', rest_path, raw_response=True, params=params)

    def order_cherrypick(self, project, repo, pull_request_id, branch, reviewers):
        rest_path = 'projects/{}/repos/{}/pull-requests/{}/links'.format(project, repo, pull_request_id)
        params = {
            'toRef': 'refs/heads/{}'.format(branch),
            'reviewers': [{'name': user} for user in reviewers],
        }
        return self.request('POST', rest_path, raw_response=True, json=params, rest_api='cherry-pick')

    def skip_vetoes_in_pr(self, project, repo, pull_request_id, rule_id):
        """
        Applies skip veto rule to a certain pr. After that bitbucket replaces
        a bunch of vetoes (defined by rule) with one special veto and leaves
        a comment in pr about it.
        One needs repo admin rights or higher to use this method.
        :rtype: requests.Response
        """
        rest_path = 'projects/{}/repos/{}/pull-requests/{}/skip-veto'.format(project, repo, pull_request_id)
        params = {'ruleId': rule_id}
        return self.request('POST', rest_path, raw_response=True, params=params, rest_api='smart-veto')
