import abc
import datetime
import functools
import logging
import socket

try:
    from urllib.parse import urljoin
except ImportError:
    from urlparse import urljoin

import requests

from sandbox.projects.common.decorators import retries

DEFAULT_TRACKED_PLATFORMS = (
    'win32', 'windows',
    'mac', 'linux', 'ios', 'android',
)
DEFAULT_TRACKED_CHANNELS = ('dev', 'beta', 'stable', 'extended')


@functools.total_ordering
class Release(object):
    @classmethod
    def fromversionfile(cls, content, channels, date=None):
        logging.debug(content)
        if date is None:
            date = datetime.datetime.today()

        variables = {}
        for line in content.splitlines():
            if '=' not in line:
                continue

            key, value = line.split('=', 1)
            key = key.strip()
            value = int(value.strip())
            variables[key] = value

        version = '.'.join(
            str(variables[key])
            for key in ('MAJOR', 'MINOR', 'BUILD', 'PATCH'))

        return cls(version, date, channels)

    def __init__(self, version, date, channels):
        super(Release, self).__init__()
        major, minor, build, patch = map(int, version.split('.'))
        self.major_version = major
        self.minor_version = minor
        self.build_version = build
        self.patch_version = patch
        self.version = version
        self.date = date
        self.channels = channels

    def __hash__(self):
        return hash(self.version)

    def __eq__(self, other):
        self_version = (
            self.major_version, self.minor_version,
            self.build_version, self.patch_version)
        other_version = (
            other.major_version, other.minor_version,
            other.build_version, other.patch_version)
        return self_version == other_version

    def __ne__(self, other):
        self_version = (
            self.major_version, self.minor_version,
            self.build_version, self.patch_version)
        other_version = (
            other.major_version, other.minor_version,
            other.build_version, other.patch_version)
        return self_version != other_version

    def __lt__(self, other):
        self_version = (
            self.major_version, self.minor_version,
            self.build_version, self.patch_version)
        other_version = (
            other.major_version, other.minor_version,
            other.build_version, other.patch_version)
        return self_version < other_version

    @property
    def upstream_branch(self):
        return 'upstream/{}'.format(self.major_version)


class ChromiumReleaseFetcher(object):
    @property
    @abc.abstractmethod
    def releases_url(self):
        pass

    @property
    @abc.abstractmethod
    def releases_endpoint(self):
        pass

    @retries(3, delay=5, backoff=2, exceptions=(requests.RequestException, socket.error))
    def _request_json(self, url, params=None):
        response = requests.get(url, params)
        response.raise_for_status()
        return response.json()

    def __init__(self, tracked_platforms=DEFAULT_TRACKED_PLATFORMS,
                 tracked_channels=DEFAULT_TRACKED_CHANNELS):
        self.tracked_platforms = tracked_platforms
        self.tracked_channels = tracked_channels
        self.url = urljoin(self.releases_url, self.releases_endpoint)

    @staticmethod
    @abc.abstractmethod
    def parse_date(value):
        raise NotImplementedError

    @abc.abstractmethod
    def releases_for_channel(self, channel):
        raise NotImplementedError

    @abc.abstractmethod
    def releases_for_version(self, version):
        raise NotImplementedError


class ChromiumDash(ChromiumReleaseFetcher):
    releases_url = 'https://chromiumdash.appspot.com/'
    releases_endpoint = 'fetch_releases'

    @staticmethod
    def parse_date(value):
        return datetime.datetime.fromtimestamp(value // 1000)

    def releases_for_channel(self, channels, os=None, num=1, offset=0):
        if set(channels) - set(self.tracked_channels):
            logging.warn(
                'Some channels are not tracked; skipping: %s',
                set(channels) - set(self.tracked_channels)
            )
        channels = list(set(channels) & set(self.tracked_channels))
        url_params = {
            'channel': ','.join(channels),
            'platform': os,
            'num': num,
            'offset': offset,
        }

        releases_info_json = self._request_json(self.url, params=url_params)
        logging.debug(releases_info_json)
        releases = {}
        for release_dict in releases_info_json:
            if release_dict['platform'].lower() not in self.tracked_platforms:
                continue

            release = Release(
                channels={release_dict['channel'].lower()},
                version=release_dict['version'],
                date=self.parse_date(release_dict['time'])
            )
            releases.setdefault(release.version, release).channels.update(release.channels)

        return releases

    def releases_for_version(self, version):
        releases_info_json = self._request_json(self.url)
        releases = []

        for release_dict in releases_info_json:
            if release_dict['platform'].lower() not in self.tracked_platforms:
                continue
            if release_dict['channel'].lower() not in self.tracked_channels:
                continue

            if release_dict['version'] == version:
                releases.append(
                    Release(
                        channels={release_dict['channel'].lower()},
                        version=release_dict['version'],
                        date=self.parse_date(release_dict['time'])
                    )
                )

        return releases
