import abc
import logging
import os
import shutil
import string
import tarfile

import six

from sandbox.common.config import Registry
import sandbox.common.types.resource as ctr

from sandbox.projects.browser.common.git import Git
from sandbox.projects.common import decorators

from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess

logger = logging.getLogger(__name__)


def _evaluate_vars(value, vars_):
    if isinstance(value, list):
        return [_evaluate_vars(item, vars_) for item in value]
    elif isinstance(value, dict):
        return {key: _evaluate_vars(item_value, vars_) for key, item_value in value.items()}
    elif isinstance(value, str):
        return value.format(**vars_)
    else:
        return value


def read_deps(deps_path):
    with open(deps_path) as deps_file:
        deps_content = deps_file.read()
    global_vars = {
        'Var': lambda x: '{{{}}}'.format(x),
        'Str': str,
    }
    local_vars = {}
    exec(deps_content, global_vars, local_vars)
    return _evaluate_vars(local_vars, local_vars.get('vars', {}))


def get_dep_git_revision(deps_path, dep_name):
    dep_value = read_deps(deps_path)['deps'][dep_name]
    dep_url = dep_value['url'] if isinstance(dep_value, dict) else dep_value
    revision = dep_url.split('@', 1)[1] if '@' in dep_url else 'master'
    if revision.startswith('remotes/'):
        revision = revision[len('remotes/'):]
    if revision.startswith('origin/'):
        revision = revision[len('origin/'):]
    return revision


@six.add_metaclass(abc.ABCMeta)
class _BaseDepotTools(object):
    _CMD_EXTENSION = abc.abstractproperty()

    def __init__(self, root_dir):
        """
        :type root_dir: sdk2.Path
        """
        self.root_dir = root_dir

    def bootstrap(self, stdout=None):
        # Remove .git to prevent depot_tools updating.
        shutil.rmtree(str(self.root_dir.joinpath('.git')))

        # Call gclient to bootstrap all files.
        gclient = self.root_dir.joinpath('gclient' + self._CMD_EXTENSION)

        env = os.environ.copy()
        env.update({
            'DEPOT_TOOLS_UPDATE': '1',
            'PATH': os.pathsep.join(filter(None, (str(self.root_dir), env.get('PATH')))),
            'USE_CIPD_PROXY': '1',
        })

        retrier = decorators.retries(3, backoff=10, exceptions=subprocess.CalledProcessError)
        check_call = retrier(subprocess.check_call)
        check_call([str(gclient)], env=env, stdout=stdout, stderr=subprocess.STDOUT)

    def pack(self, archive_path):
        """
        Pack depot_tools to archive.

        :type archive_path: sdk2.Path
        """
        with tarfile.open(str(archive_path), 'w:gz') as archive:
            archive.add(str(self.root_dir), arcname='')

    @classmethod
    def unpack(cls, archive_path, target_dir):
        """
        Unpack archived depot_tools to directory.

        :type archive_path: sdk2.Path
        :type target_dir: sdk2.Path
        :rtype: _BaseDepotTools
        """
        with tarfile.open(str(archive_path), 'r:gz') as archive:
            archive.extractall(str(target_dir))
        return cls(target_dir)


class WinDepotTools(_BaseDepotTools):
    _CMD_EXTENSION = '.bat'


class UnixDepotTools(_BaseDepotTools):
    _CMD_EXTENSION = ''


if os.name == 'nt':
    DepotTools = WinDepotTools
else:
    DepotTools = UnixDepotTools


class BrowserDepotTools(sdk2.Resource):
    """
    Prepared depot_tools bundle.
    """
    any_arch = False
    auto_backup = True
    restart_policy = ctr.RestartPolicy.IGNORE
    ttl = 7

    cache_version = sdk2.Attributes.Integer('Cache version')
    commit = sdk2.Attributes.String('depot_tools commit hash', required=True)
    repo_url = sdk2.Attributes.String('depot_tools repository url', required=True)


class DepotToolsEnvironment(SandboxEnvironment):
    DEPOT_TOOLS_URL = 'https://bitbucket.browser.yandex-team.ru/scm/stardust/depot_tools.git'
    DEPOT_TOOLS_FOLDER_FMT = 'depot_tools-{commit}'
    DEPOT_TOOLS_ARCHIVE_FMT = 'depot_tools-{commit}.tar.gz'
    _CACHE_VERSION = 1

    def __init__(self, revision=None, deps_file=None, dep_name=None):
        super(DepotToolsEnvironment, self).__init__()
        self.vcs_root = Git(self.DEPOT_TOOLS_URL, filter_branches=False)
        self.revision = revision
        self.deps_file = deps_file
        self.dep_name = dep_name
        self.commit = None

    @property
    def task(self):
        return sdk2.Task.current

    @property
    def depot_tools_folder(self):
        if not self.commit:
            raise ValueError('depot_tools commit not set.')
        return self.task.path(self.DEPOT_TOOLS_FOLDER_FMT.format(commit=self.commit))

    @property
    def depot_tools_archive(self):
        if not self.commit:
            raise ValueError('depot_tools commit not set.')
        return self.task.path(self.DEPOT_TOOLS_ARCHIVE_FMT.format(commit=self.commit))

    def determine_revision(self):
        if self.revision:
            return self.revision
        if self.deps_file and self.dep_name:
            return get_dep_git_revision(self.deps_file, self.dep_name)
        return 'master'

    @staticmethod
    def is_git_commit(revision):
        return len(revision) == 40 and all(c in string.hexdigits for c in revision)

    def resolve_revision(self):
        revision = self.determine_revision()
        logger.debug('depot_tools revision is %s', revision)
        if self.is_git_commit(revision):
            self.commit = revision
        else:
            self.vcs_root.update_cache_repo(revision)
            self.commit = subprocess.check_output(['git', 'rev-parse', revision],
                                                  cwd=self.vcs_root.cache_repo_dir).strip()
        logger.info('Revision %s was resolved as commit %s', revision, self.commit)

    def download_depot_tools(self):
        resource = BrowserDepotTools.find(
            attrs=dict(
                cache_version=self._CACHE_VERSION,
                commit=self.commit,
                repo_url=self.DEPOT_TOOLS_URL,
            ),
            arch=Registry().this.system.family,
            state=ctr.State.READY,
        ).first()
        if not resource:
            logger.debug('No suitable depot_tools resources found.')
            return None
        logger.debug('Will use depot_tools from resource %d.', resource.id)
        resource_data = sdk2.ResourceData(resource)
        return resource_data.path

    def prepare_depot_tools(self):
        self.vcs_root.clone(str(self.depot_tools_folder), commit=self.commit)
        depot_tools = DepotTools(self.depot_tools_folder)

        with ProcessLog(self.task, logger='bootstrap_depot_tools') as log:
            depot_tools.bootstrap(stdout=log.stdout)

        return depot_tools

    def publish_depot_tools(self, depot_tools):
        depot_tools.pack(self.depot_tools_archive)

        resource = BrowserDepotTools(
            self.task, 'Browser depot_tools', self.depot_tools_archive,
            cache_version=self._CACHE_VERSION,
            commit=self.commit,
            repo_url=self.DEPOT_TOOLS_URL,
        )
        sdk2.ResourceData(resource).ready()
        logger.debug('Published depot_tools as resource %d.', resource.id)

    def prepare(self):
        self.resolve_revision()

        logger.debug('depot_tools target dir: %s', self.depot_tools_folder)

        depot_tools_archive = self.download_depot_tools()
        if depot_tools_archive:
            logger.debug('depot_tools archive path: %s', depot_tools_archive)
            depot_tools = DepotTools.unpack(depot_tools_archive, self.depot_tools_folder)
            logger.debug('Unpacked depot_tools archive to %s', self.depot_tools_folder)
        else:
            depot_tools = self.prepare_depot_tools()
            self.publish_depot_tools(depot_tools)

        os.environ['DEPOT_TOOLS_COMMIT'] = six.ensure_str(self.commit)
        os.environ['PATH'] = os.pathsep.join((str(depot_tools.root_dir), os.environ['PATH']))


class ChromiumDepotToolsEnvironment(DepotToolsEnvironment):
    DEPOT_TOOLS_URL = 'https://bitbucket.browser.yandex-team.ru/scm/chromium/tools-depot_tools.git'
