"""
Author: Anatoly Matyukhin <amatyukhin@yandex-team.ru>
"""
import functools
import logging
import os
import subprocess
import sys

from sandbox.common.errors import TemporaryError

from sandbox.sandboxsdk.environments import SandboxEnvironment, TarballToolkitBase
from sandbox import sdk2
from sandbox.sdk2.vcs.git import Git as SandboxGit


class ConfigureGitEnvironment(SandboxEnvironment):
    def __init__(self, email, username):
        super(ConfigureGitEnvironment, self).__init__()
        self.email = email
        self.username = username

    def prepare(self):
        subprocess.check_call(['git', 'config', '--global', 'user.email', self.email])
        subprocess.check_call(['git', 'config', '--global', 'user.name', self.username])


class BrowserGitEnvironment(sdk2.Resource):
    """
    Git, built from source and packed in tar.gz archive.
    To make it portable, bin-wrappers/* entrypoints should be modified
    (like that: https://github.yandex-team.ru/gist/unlok/d9bc1a22141faf72ebb6).
    """
    any_arch = False
    auto_backup = True
    version = sdk2.Attributes.String('Git version', required=True)
    platform = sdk2.Attributes.String('Platform', required=True)


class GitEnvironment(TarballToolkitBase):
    resource_type = 'BROWSER_GIT_ENVIRONMENT'
    name = 'browser_git'
    sys_path_utils = ('git',)
    use_cache = True

    def prepare(self):
        env_dir = super(GitEnvironment, self).prepare()
        bin_dir_name = 'bin' if os.name == 'nt' else 'bin-wrappers'
        bin_dir = os.path.join(env_dir, bin_dir_name)
        if not os.path.isdir(bin_dir):
            raise RuntimeError('Git binary dir is not found: {}'.format(bin_dir))
        self.update_os_env('PATH', bin_dir)
        self.check_environment()
        return env_dir


def _temporary_error_if_failed(func):
    @functools.wraps(func)
    def wrapper(self, *args, **kwargs):
        try:
            return func(self, *args, **kwargs)
        except Exception:
            message = '{}: {} failed'.format(self._repo_name, func.__name__)
            logging.exception(message)
            raise TemporaryError(message)
    return wrapper


class Git(SandboxGit):
    DEFAULT_GIT_CONFIG = {
        'linux2': {
            # Keep git config in sync with this docs:
            # https://chromium.googlesource.com/chromium/src/+/master/docs/linux_build_instructions.md
            # Use git protocol v2
            # Needs git 2.18
            # 'protocol.version': 2,
        },
        'darwin': {
            # Keep git config in sync with this docs:
            # https://chromium.googlesource.com/chromium/src/+/master/docs/mac_build_instructions.md
            # Do not handle filesystems.
            'core.precomposeUnicode': False,
            # Use git protocol v2
            # Needs git 2.18
            # 'protocol.version': 2,
        },
        'win32': {
            # Keep git config in sync with this docs:
            # https://chromium.googlesource.com/chromium/src/+/master/docs/windows_build_instructions.md
            # Do not touch eol.
            'core.autocrlf': False,
            # Do not trust file mode on windows.
            'core.filemode': False,
            # Handle long paths on windows.
            'core.longpaths': True,
            # Cache file metadata on windows.
            'core.fscache': True,
            # Use git protocol v2
            # Needs git 2.18
            # 'protocol.version': 2,
        }
    }

    @staticmethod
    def _to_valid_branch(branch):
        if branch is None:
            return None
        elif branch.startswith('refs/heads'):
            return branch.replace('refs/heads/', '', 1)
        else:
            return branch

    def setup_git_config(self, target_dir, config):
        config = config or {}
        platform = {
            'cygwin': 'win32',
        }.get(sys.platform, sys.platform)
        config = dict(self.DEFAULT_GIT_CONFIG[platform], **config)
        return super(Git, self).setup_git_config(target_dir, config)

    @_temporary_error_if_failed
    def update_cache_repo(self, *args, **kwargs):
        return super(Git, self).update_cache_repo(*args, **kwargs)

    @_temporary_error_if_failed
    def clone(self, target_dir, branch=None, *args, **kwargs):
        return super(Git, self).clone(target_dir, self._to_valid_branch(branch), *args, **kwargs)

    @_temporary_error_if_failed
    def clone_bare(self, *args, **kwargs):
        return super(Git, self).clone_bare(*args, **kwargs)
