"""
Author: Sviat Yesipov <svyat@yandex-team.ru>

Interface to git.
Known versions of git to be supported:
1.7.9.5 (Linux agents)
"""
import logging
import os
import sys
from subprocess import PIPE, CalledProcessError

import six
import sandbox.sdk2
from sandbox.sdk2.helpers import subprocess as sp


GIT_OUTPUT_ENCODING = 'UTF-8'


class GitCli(object):
    def __init__(self, work_dir, config=None):
        self.work_dir = work_dir
        self._config = config

    @classmethod
    def _encode_args(cls, args):
        if six.PY2:
            encoded_args = []
            for arg in args:
                if isinstance(arg, six.text_type):
                    encoded_args.append(arg.encode(sys.getfilesystemencoding()))
                else:
                    encoded_args.append(arg)
            return tuple(encoded_args)
        return args

    def _args_head(self):
        args = ['git']
        if self._config:
            for key, value in self._config.items():
                args.extend(['-c', '{0}={1}'.format(key, value)])
        return tuple(args)

    def git(self, command, *args):
        command_args = self._args_head() + (command,) + args
        with sandbox.sdk2.helpers.ProcessLog(logger=logging.getLogger('git')) as pl:
            pl.logger.debug('Running %s', command_args)
            pl.logger.propagate = 1
            sp.check_call(
                self._encode_args(command_args),
                stdout=pl.stdout,
                stderr=pl.stderr,
                cwd=self.work_dir
            )

    def git_with_out(self, command, *args):
        command_args = self._args_head() + (command,) + args
        with sandbox.sdk2.helpers.ProcessLog(logger=logging.getLogger('git')) as pl:
            pl.logger.debug('Running %s', command_args)
            pl.logger.propagate = 1
            return sp.check_output(
                self._encode_args(command_args),
                stderr=pl.stderr,
                cwd=self.work_dir
            )

    def git_with_out_and_err(self, command, *args):
        command_args = self._args_head() + (command,) + args
        logging.debug('Running %s', command_args)
        process = sp.Popen(
            self._encode_args(command_args),
            cwd=self.work_dir,
            stdout=PIPE,
            stderr=PIPE,
        )
        out, err = process.communicate()
        return_code = process.poll()
        if return_code:
            error_text = '\n'.join([out, err]) if all([out, err]) else (out or err)
            raise CalledProcessError(return_code, command_args, output=error_text)
        return out, err

    def am(self, *args):
        return self.git('am', *args)

    def branch(self, *args):
        return self.git_with_out('branch', *args).decode(GIT_OUTPUT_ENCODING)

    def log(self, *args):
        return self.git_with_out('log', *args).decode(GIT_OUTPUT_ENCODING)

    def diff(self, *args):
        return self.git_with_out('diff', *args)

    def diff_tree(self, *args):
        return self.git_with_out('diff-tree', *args)

    def remote(self, *args):
        return self.git_with_out('remote', *args).decode(GIT_OUTPUT_ENCODING)

    def ls_remote(self, *args):
        return self.git_with_out('ls-remote', *args).decode(GIT_OUTPUT_ENCODING)

    def checkout(self, *args):
        return self.git('checkout', *args)

    def clean(self, *args):
        return self.git('clean', *args)

    def clone(self, *args):
        return self.git('clone', *args)

    def config(self, *args):
        return self.git_with_out('config', *args).decode(GIT_OUTPUT_ENCODING)

    def fetch(self, *args):
        return self.git('fetch', *args)

    def init(self, *args):
        return self.git('init', *args)

    def pull(self, *args):
        return self.git('pull', *args)

    def reset(self, *args):
        return self.git('reset', *args)

    def add(self, *args):
        return self.git('add', *args)

    def commit(self, *args):
        return self.git('commit', *args)

    def push(self, *args):
        return self.git('push', *args)

    def update_ref(self, *args):
        return self.git('update-ref', *args)

    def for_each_ref(self, *args):
        return self.git_with_out('for-each-ref', *args)

    def format_patch(self, *args):
        return self.git_with_out('format-patch', *args)

    def blame(self, *args):
        return self.git_with_out('blame', *args)

    def cat_file(self, *args):
        return self.git_with_out('cat-file', *args)

    def rev_parse(self, *args):
        return self.git_with_out('rev-parse', *args).decode(GIT_OUTPUT_ENCODING)

    def rev_list(self, *args):
        return self.git_with_out('rev-list', *args).decode(GIT_OUTPUT_ENCODING)

    def grep(self, *args):
        return self.git_with_out('grep', *args)

    def merge(self, *args):
        return self.git_with_out('merge', *args)

    def merge_base(self, *args):
        return self.git_with_out('merge-base', *args).decode(GIT_OUTPUT_ENCODING)

    def mergetool(self, *args):
        return self.git('mergetool', *args)

    def show(self, *args):
        return self.git_with_out('show', *args)

    def ls_files(self, *args):
        return self.git_with_out('ls-files', *args).decode(GIT_OUTPUT_ENCODING)

    def ls_tree(self, *args):
        return self.git_with_out('ls-tree', *args).decode(GIT_OUTPUT_ENCODING)

    def show_ref(self, *args):
        return self.git_with_out('show-ref', *args).decode(GIT_OUTPUT_ENCODING)

    def status(self, *args):
        return self.git_with_out('status', *args).decode(GIT_OUTPUT_ENCODING)

    def gc(self, *args):
        return self.git('gc', *args)

    def prune(self, *args):
        return self.git('prune', *args)

    def symbolic_ref(self, *args):
        return self.git_with_out('symbolic-ref', *args).decode(GIT_OUTPUT_ENCODING)

    def rebase(self, *args):
        return self.git('rebase', *args)

    def check_attr(self, *args):
        return self.git_with_out('check-attr', *args).decode(GIT_OUTPUT_ENCODING)

    def mv(self, *args):
        return self.git('mv', *args)

    def archive(self, *args):
        return self.git('archive', *args)

    def rm(self, *args):
        return self.git('rm', *args)

    def apply(self, *args):
        return self.git_with_out_and_err('apply', *args)


class GitException(Exception):
    def __init__(self, orig_exception, git_command):
        self.value = str(orig_exception) + '. Git command: ' + git_command

    def __str__(self):
        return self.value


class GitHelper(object):
    def __init__(self, repo_dir, config=None):
        self.repo_dir = os.path.abspath(repo_dir)
        self.git = GitCli(self.repo_dir, config=config)

    def commit_by_message(self, commit_pattern, branch=None,
                          strict=True, oneline=False, all_branches=False):
        """
        Returns hash of latest commit on branch
        with message that matched commit_message grep-pattern.

        When oneline is True - hash and commit message are returned.
        """
        cmd = ['--max-count=1', '--grep={0}'.format(commit_pattern)]
        if oneline:
            cmd.append('--oneline')
        if all_branches:
            cmd.append('--all')
        else:
            cmd.append(branch if branch else 'HEAD')
            cmd.append('--')
        result = self.git.rev_list(*cmd).strip() or None
        if strict:
            assert result, 'Cannot find commit "{0}" in "{1}" branch'.format(
                commit_pattern, branch)
        return result

    def current_branch(self):
        return self.git.symbolic_ref('-q', 'HEAD').strip().replace('refs/heads/', '')

    def fetch_and_checkout(self, branch_name):
        if branch_name == self.current_branch():
            logging.debug('Already on %s, pulling', branch_name)
            self.git.pull('origin', branch_name)
        else:
            logging.debug('Fetching %s', branch_name)
            self.git.fetch('-f', 'origin', '+{b}:{b}'.format(b=branch_name))
            logging.debug('Checking out %s', branch_name)
            self.git.checkout(branch_name)
