"""
Author: Anatoly Matyukhin <amatyukhin@yandex-team.ru>

Enumeration of browser-related repositories.

Usage example:
from sandbox.projects.browser.common.git import repositories
repositories.Stardust.browser(testing=True, filter_branches=False).clone(target_dir, 'master')
"""
from functools import partial

from sandbox.projects.browser.common.git import Git


# TODO: use 'from sandbox.projects.browser.common import bitbucket'
# if sandbox.projects.browser.common.bitbucket is py23-compatible.
DEFAULT_BITBUCKET_HOST = 'bitbucket.browser.yandex-team.ru'
TESTING_BITBUCKET_HOST = 'bitbucket.test.browser.yandex-team.ru'


def get_https_git_url(repo, project, host=DEFAULT_BITBUCKET_HOST, login=None, token=None):
    oauth_prefix = ''
    assert (login is None) == (token is None), 'Specify both login and token'
    if login is not None and token is not None:
        oauth_prefix = 'x-oauth-{login}:{token}@'.format(login=login, token=token)
    return 'https://{oauth_prefix}{host}/scm/{project}/{repo}.git'.format(
        oauth_prefix=oauth_prefix,
        host=host.lower(),
        project=project.lower(),
        repo=repo.lower()
    )


def get_ssh_git_url(repo, project, host=DEFAULT_BITBUCKET_HOST):
    return 'ssh://{host}/{project}/{repo}.git'.format(
        host=host.lower(),
        project=project.lower(),
        repo=repo.lower()
    )


def bitbucket_vcs_root(project, repo, testing=False, **kwargs):
    bitbucket_host = TESTING_BITBUCKET_HOST if testing else DEFAULT_BITBUCKET_HOST
    return Git(
        url=get_https_git_url(host=bitbucket_host, project=project, repo=repo),
        push_url=get_ssh_git_url(host=bitbucket_host, project=project, repo=repo),
        **kwargs
    )


class Autotest(object):
    browser_dsl_scripts_generator = partial(bitbucket_vcs_root, 'autotest', 'browser-dsl-scripts-generator')
    browser_test_framework = partial(bitbucket_vcs_root, 'autotest', 'browser-test-framework')


class Chromium(object):
    src = partial(bitbucket_vcs_root, 'chromium', 'src')


class Stardust(object):
    browser = partial(bitbucket_vcs_root, 'stardust', 'browser')
    browser_bisect_patches = partial(bitbucket_vcs_root, 'stardust', 'browser-bisect-patches')
    browser_brandings = partial(bitbucket_vcs_root, 'stardust', 'browser-brandings')
    browser_deps_src_third_party_android_tools = partial(
        bitbucket_vcs_root, 'stardust', 'browser-deps-src-third_party-android_tools')
    bryt = partial(bitbucket_vcs_root, 'stardust', 'bryt')
    icons2020 = partial(bitbucket_vcs_root, 'stardust', 'icons2020')
    performance = partial(bitbucket_vcs_root, 'stardust', 'performance')
    yin = partial(bitbucket_vcs_root, 'stardust', 'yin')
    build_launcher_configs = partial(bitbucket_vcs_root, 'stardust', 'build-launcher-configs')
    ninja = partial(bitbucket_vcs_root, 'stardust', 'ninja')
    browser_uploads = partial(bitbucket_vcs_root, 'stardust', 'browser-uploads')


class DC(object):
    distclang = partial(bitbucket_vcs_root, 'dc', 'dist-clang')


class ML(object):
    mobile_alice_library_crossplatform = partial(bitbucket_vcs_root, 'ml', 'mobile-alice-library-crossplatform')


class Personal(object):
    browser_brandings_test = partial(bitbucket_vcs_root, '~distrib-fellow', 'browser-brandings-test')
    stump = partial(bitbucket_vcs_root, '~frisbeeman', 'stump')
    gemor = partial(bitbucket_vcs_root, '~kriomant', 'gemor')
