import contextlib
import os
import shutil

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess


def _get_session_with_retries():
    session = requests.Session()
    retry = Retry(
        total=5,
        backoff_factor=0.3,
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


class HermeticXcodeEnvironment(SandboxEnvironment):
    BASE_URL = 'https://s3.mds.yandex.net/broinfra-tools/xcode-archives/Xcode{}.xip'

    def __init__(self, version):
        super(HermeticXcodeEnvironment, self).__init__()
        self.xcode_archive_url = self.BASE_URL.format(version)

    @property
    def task(self):
        return sdk2.Task.current

    @property
    def extract_dir(self):
        return str(sdk2.Task.current.path('hermetic_xcode'))

    @property
    def xcode_app_path(self):
        for entry in os.listdir(self.extract_dir):
            if os.path.splitext(entry)[1] == '.app':
                return os.path.join(self.extract_dir, entry)
        return None

    def accept_license_if_required(self):
        import plistlib

        xcode_plist_path = '/Library/Preferences/com.apple.dt.Xcode.plist'
        if os.path.exists(xcode_plist_path):
            # Determine if license for this xcode version is already accepted.
            target_license_plist_path = os.path.join(self.xcode_app_path, 'Contents', 'Resources', 'LicenseInfo.plist')
            target_license_plist = plistlib.readPlist(target_license_plist_path)
            build_type = target_license_plist['licenseType']
            build_version = target_license_plist['licenseID']

            accepted_license_plist = plistlib.readPlist(xcode_plist_path)
            agreed_to_key = 'IDELast{}LicenseAgreedTo'.format(build_type)
            last_license_agreed_to = accepted_license_plist.get(agreed_to_key)

            # Historically all Xcode build numbers have been in the format of
            # AANNNN, so a simyle string compare works. If Xcode's build numbers
            # change this may need a more complex compare.
            if last_license_agreed_to is not None and build_version <= last_license_agreed_to:
                # Don't accept the license of older toolchain builds, this will
                # break the license of newer builds.
                return

        with ProcessLog(self.task, logger='accept_xcode_license') as log:
            subprocess.check_call(['sudo', '-E', '/usr/bin/xcodebuild', '-license', 'accept'],
                                  stdout=log.stdout, stderr=log.stderr)

    def setup(self):
        with ProcessLog(self.task, logger='setup_hermetic_xcode') as log:
            subprocess.check_call(['sudo', '-E', '/usr/bin/xcodebuild', '-runFirstLaunch'],
                                  stdout=log.stdout, stderr=log.stderr)

            # Workaround simctl unable to find liblaunch_sim.dylib.
            # See st/BYIN-11596 for more info.
            subprocess.check_call(['sudo', '-E', '/usr/bin/xcrun', 'simctl', 'shutdown', 'all'],
                                  stdout=log.stdout, stderr=log.stderr)
            subprocess.check_call(['sudo', '-E', '/usr/bin/xcrun', 'simctl', 'erase', 'all'],
                                  stdout=log.stdout, stderr=log.stderr)
            subprocess.check_call(['sudo', '/bin/rm', '-rf', os.path.expanduser('~/Library/Logs')],
                                  stdout=log.stdout, stderr=log.stderr)

    def unpack_archive(self, archive_stream):
        os.makedirs(self.extract_dir)
        xcode_xip_path = os.path.join(self.extract_dir, 'Xcode.xip')
        with open(xcode_xip_path, 'wb') as xcode_xip:
            shutil.copyfileobj(archive_stream, xcode_xip)
        subprocess.check_call(['/usr/bin/xip', '--expand', xcode_xip_path], cwd=self.extract_dir)

    def prepare(self):
        session = _get_session_with_retries()
        with contextlib.closing(session.get(self.xcode_archive_url, timeout=(6, 12), stream=True)) as archive_response:
            archive_response.raise_for_status()
            self.unpack_archive(archive_response.raw)
        os.environ['DEVELOPER_DIR'] = os.path.join(self.xcode_app_path, 'Contents', 'Developer')
        self.accept_license_if_required()
        self.setup()
